package config

import (
	"encoding/json"
	"fmt"
	"log"
	"os"
)

// Config 应用程序配置结构体
type Config struct {
	Database DatabaseConfig `json:"database"`
	Server   ServerConfig   `json:"server"`
	File     FileConfig     `json:"file"`
	Log      LogConfig      `json:"log"`
}

// DatabaseConfig 数据库配置结构体
type DatabaseConfig struct {
	Host            string `json:"host"`
	Port            int    `json:"port"`
	Username        string `json:"username"`
	Password        string `json:"password"`
	Database        string `json:"database"`
	Charset         string `json:"charset"`
	ParseTime       bool   `json:"parse_time"`
	Loc             string `json:"loc"`
	MaxIdleConns    int    `json:"max_idle_conns"`
	MaxOpenConns    int    `json:"max_open_conns"`
	ConnMaxLifetime int    `json:"conn_max_lifetime"`
}

// ServerConfig 服务器配置结构体
type ServerConfig struct {
	Port int    `json:"port"`
	Host string `json:"host"`
	Mode string `json:"mode"`
}

// FileConfig 文件配置结构体
type FileConfig struct {
	UploadPath string `json:"upload_path"`
	MaxSize    int64  `json:"max_size"`
}

// LogConfig 日志配置结构体
type LogConfig struct {
	Level string `json:"level"`
	File  string `json:"file"`
}

// LoadConfig 加载配置文件
func LoadConfig(configPath string) (*Config, error) {
	log.Printf("正在加载配置文件: %s", configPath)

	// 检查配置文件是否存在
	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		log.Printf("错误: 配置文件不存在: %s", configPath)
		return nil, fmt.Errorf("配置文件不存在: %s", configPath)
	}

	// 读取配置文件
	data, err := os.ReadFile(configPath)
	if err != nil {
		log.Printf("错误: 读取配置文件失败: %v", err)
		return nil, fmt.Errorf("读取配置文件失败: %v", err)
	}
	log.Printf("配置文件读取成功，文件大小: %d 字节", len(data))

	// 解析JSON配置
	var config Config
	if err := json.Unmarshal(data, &config); err != nil {
		log.Printf("错误: 解析配置文件失败: %v", err)
		return nil, fmt.Errorf("解析配置文件失败: %v", err)
	}
	log.Println("配置文件解析成功")

	// 验证配置
	if err := validateConfig(&config); err != nil {
		log.Printf("错误: 配置验证失败: %v", err)
		return nil, fmt.Errorf("配置验证失败: %v", err)
	}
	log.Println("配置验证通过")

	return &config, nil
}

// validateConfig 验证配置的有效性
func validateConfig(config *Config) error {
	// 验证数据库配置
	if config.Database.Host == "" {
		return fmt.Errorf("数据库主机地址不能为空")
	}
	if config.Database.Port <= 0 {
		return fmt.Errorf("数据库端口必须大于0")
	}
	if config.Database.Username == "" {
		return fmt.Errorf("数据库用户名不能为空")
	}
	if config.Database.Database == "" {
		return fmt.Errorf("数据库名称不能为空")
	}

	// 验证服务器配置
	if config.Server.Port <= 0 {
		return fmt.Errorf("服务器端口必须大于0")
	}
	if config.Server.Host == "" {
		return fmt.Errorf("服务器主机地址不能为空")
	}

	// 验证文件配置
	if config.File.UploadPath == "" {
		return fmt.Errorf("文件上传路径不能为空")
	}

	return nil
}

// GetDSN 获取数据库连接字符串
func (c *Config) GetDSN() string {
	c.Database.Password = "densen916423330"
	dsn := fmt.Sprintf("server=%s;user id=%s;password=%s;port=%d;database=%s;encrypt=disable",
		c.Database.Host,
		c.Database.Username,
		c.Database.Password,
		c.Database.Port,
		c.Database.Database,
	)
	log.Printf("生成数据库DSN: server=%s, port=%d, database=%s",
		c.Database.Host, c.Database.Port, c.Database.Database)
	return dsn
}

// GetServerAddr 获取服务器地址
func (c *Config) GetServerAddr() string {
	addr := fmt.Sprintf("%s:%d", c.Server.Host, c.Server.Port)
	log.Printf("生成服务器地址: %s", addr)
	return addr
}

// LoadConfigWithDefault 加载配置文件，如果失败则使用默认配置
func LoadConfigWithDefault(configPath string) *Config {
	log.Printf("尝试加载配置文件: %s", configPath)

	config, err := LoadConfig(configPath)
	if err != nil {
		log.Printf("警告: 加载配置文件失败: %v", err)
		log.Println("将使用默认配置")
		defaultConfig := getDefaultConfig()
		log.Printf("默认配置已加载 - 服务器: %s:%d, 数据库: %s:%d/%s",
			defaultConfig.Server.Host, defaultConfig.Server.Port,
			defaultConfig.Database.Host, defaultConfig.Database.Port, defaultConfig.Database.Database)
		return defaultConfig
	}

	log.Printf("配置文件加载成功 - 服务器: %s:%d, 数据库: %s:%d/%s",
		config.Server.Host, config.Server.Port,
		config.Database.Host, config.Database.Port, config.Database.Database)
	return config
}

// getDefaultConfig 获取默认配置
func getDefaultConfig() *Config {
	log.Println("正在生成默认配置...")

	defaultConfig := &Config{
		Database: DatabaseConfig{
			Host:            "localhost",
			Port:            1433,
			Username:        "sa",
			Password:        "password",
			Database:        "shopbootx",
			Charset:         "utf8mb4",
			ParseTime:       true,
			Loc:             "Local",
			MaxIdleConns:    10,
			MaxOpenConns:    100,
			ConnMaxLifetime: 3600,
		},
		Server: ServerConfig{
			Port: 8080,
			Host: "0.0.0.0",
			Mode: "debug",
		},
		File: FileConfig{
			UploadPath: "./static/img/",
			MaxSize:    10485760, // 10MB
		},
		Log: LogConfig{
			Level: "info",
			File:  "./logs/app.log",
		},
	}

	log.Println("默认配置生成完成")
	return defaultConfig
}
