package config

import (
	"os"
	"testing"
)

func TestLoadConfig(t *testing.T) {
	// 创建临时配置文件
	tempConfig := `{
		"database": {
			"host": "testhost",
			"port": 3307,
			"username": "testuser",
			"password": "testpass",
			"database": "testdb",
			"charset": "utf8",
			"parse_time": false,
			"loc": "UTC",
			"max_idle_conns": 5,
			"max_open_conns": 50,
			"conn_max_lifetime": 1800
		},
		"server": {
			"port": 9090,
			"host": "127.0.0.1",
			"mode": "release"
		},
		"file": {
			"upload_path": "/tmp/uploads/",
			"max_size": 5242880
		},
		"log": {
			"level": "debug",
			"file": "/tmp/test.log"
		}
	}`

	// 写入临时文件
	err := os.WriteFile("test_config.json", []byte(tempConfig), 0644)
	if err != nil {
		t.Fatalf("创建测试配置文件失败: %v", err)
	}
	defer os.Remove("test_config.json")

	// 加载配置
	config, err := LoadConfig("test_config.json")
	if err != nil {
		t.Fatalf("加载配置失败: %v", err)
	}

	// 验证数据库配置
	if config.Database.Host != "testhost" {
		t.Errorf("数据库主机地址不匹配，期望: testhost, 实际: %s", config.Database.Host)
	}
	if config.Database.Port != 3307 {
		t.Errorf("数据库端口不匹配，期望: 3307, 实际: %d", config.Database.Port)
	}
	if config.Database.Username != "testuser" {
		t.Errorf("数据库用户名不匹配，期望: testuser, 实际: %s", config.Database.Username)
	}

	// 验证服务器配置
	if config.Server.Port != 9090 {
		t.Errorf("服务器端口不匹配，期望: 9090, 实际: %d", config.Server.Port)
	}
	if config.Server.Host != "127.0.0.1" {
		t.Errorf("服务器主机不匹配，期望: 127.0.0.1, 实际: %s", config.Server.Host)
	}

	// 验证DSN生成
	expectedDSN := "testuser:testpass@tcp(testhost:3307)/testdb?charset=utf8&parseTime=false&loc=UTC"
	if config.GetDSN() != expectedDSN {
		t.Errorf("DSN不匹配，期望: %s, 实际: %s", expectedDSN, config.GetDSN())
	}

	// 验证服务器地址
	expectedAddr := "127.0.0.1:9090"
	if config.GetServerAddr() != expectedAddr {
		t.Errorf("服务器地址不匹配，期望: %s, 实际: %s", expectedAddr, config.GetServerAddr())
	}
}

func TestLoadConfigWithDefault(t *testing.T) {
	// 测试加载不存在的配置文件，应该返回默认配置
	config := LoadConfigWithDefault("nonexistent.json")

	// 验证默认配置
	if config.Database.Host != "localhost" {
		t.Errorf("默认数据库主机不匹配，期望: localhost, 实际: %s", config.Database.Host)
	}
	if config.Server.Port != 8080 {
		t.Errorf("默认服务器端口不匹配，期望: 8080, 实际: %d", config.Server.Port)
	}
}

func TestGetDefaultConfig(t *testing.T) {
	config := getDefaultConfig()

	// 验证默认配置的完整性
	if config.Database.Host == "" {
		t.Error("默认配置中数据库主机为空")
	}
	if config.Server.Port == 0 {
		t.Error("默认配置中服务器端口为0")
	}
	if config.File.UploadPath == "" {
		t.Error("默认配置中文件上传路径为空")
	}
} 