package controllers

import (
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

// 负责处理所有与库存相关的HTTP请求，包括盘点单管理、库存更新等功能
type InventoryController struct {
	db *gorm.DB // 数据库连接实例
}

// NewInventoryController 创建新的库存控制器实例
// 参数: db - 数据库连接实例
// 返回: InventoryController实例
func NewInventoryController(db *gorm.DB) *InventoryController {
	return &InventoryController{db: db}
}

// GetPurchaseListByNodeCash 获取盘点单列表接口
// 根据条件查询盘点单列表，支持按albaranId和supplierCompanyName筛选
// 路由: GET /inventory/getPurchaseListByNodeCash
// 查询参数: albaranId - 采购单号, supplierCompanyName - 供应商公司名称
func (ic *InventoryController) GetPurchaseListByNodeCash(c *gin.Context) {
	albaranId := c.Query("albaranId")
	supplierCompanyName := c.Query("supplierCompanyName")

	var orderIDs []int

	// 第一步：先查询符合条件的前 30 个 OrderID
	subQuery := ic.db.Table("Purchase as p").
		Select("TOP 30 p.OrderID").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").Order("p.AlbaranID DESC")
	//Where("p.Cash IS NULL")

	if albaranId != "" {
		if id, err := strconv.Atoi(albaranId); err == nil {
			subQuery = subQuery.Where("p.AlbaranID = ?", id)
		}
	}
	if supplierCompanyName != "" {
		subQuery = subQuery.Where("s.CompanyName LIKE ?", "%"+supplierCompanyName+"%")
	}

	//subQuery = subQuery.Order("p.AlbaranID DESC")
	if err := subQuery.Pluck("OrderID", &orderIDs).Error; err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询订单ID失败",
		})
		return
	}

	// 没有数据直接返回
	if len(orderIDs) == 0 {
		c.JSON(http.StatusOK, gin.H{
			"code":    200,
			"message": "获取订单列表成功",
			"data":    []any{},
		})
		return
	}

	// 第二步：用 IN 查询获取完整字段（等效于 JOIN OrderedIDs）
	var purchasesWithSupplier []models.PurchaseWithSupplier
	err := ic.db.Table("Purchase as p").
		Select("p.*, s.CompanyName AS SupplierCompanyName").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Where("p.OrderID IN ?", orderIDs).Order("p.AlbaranID DESC").
		Find(&purchasesWithSupplier).Error

	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取订单列表失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取订单列表成功",
		"data":    purchasesWithSupplier,
	})
}

// UpdateProductAndInsertPurchase 新增盘点单接口
// 更新商品库存并创建相应的盘点单和明细记录
// 路由: POST /inventory/updateProductAndInsertPurchase
// 请求体: 包含ProductID和UnitsInStock的JSON对象
func (ic *InventoryController) UpdateProductAndInsertPurchase(c *gin.Context) {
	// 定义请求参数结构
	var request struct {
		ProductId    string `json:"productId"` // 商品ID
		UnitsInStock string `json:"unitsInStock"`
	}
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	// 开始数据库事务
	tx := ic.db.Begin()
	// 根据ProductID查找商品
	var product models.Products
	result := tx.Where("UserCode = ?", request.ProductId).Select("Top 1 *").Find(&product)
	if result.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "未查询到该商品",
		})
		return
	}
	if product.ProductId == 0 {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "未查询到该商品",
		})
		return
	}
	// 解析库存数量字符串为浮点数
	oldUnitsInStock := product.UnitsInStock
	unitsInStock, err := strconv.ParseFloat(request.UnitsInStock, 64)
	if err != nil {
		tx.Rollback()
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "库存数量格式错误",
		})
		return
	}

	// 更新商品库存
	result = tx.Model(&product).Where("ProductID = ?", product.ProductId).Update("UnitsInStock", unitsInStock)
	if result.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新商品库存失败",
		})
		return
	}
	//TODO:添加进货单
	purchaseWithSuppliers, err := ic.GetTopPurchases()
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    500,
			"message": err.Error(),
		})
	}

	albaranId := 0
	if len(purchaseWithSuppliers) > 0 {
		albaranId = *purchaseWithSuppliers[0].AlbaranId + 1
	}

	// 计算库存差异（新库存 - 旧库存）
	newUnit := unitsInStock - oldUnitsInStock

	// 创建盘点单记录
	now := time.Now()
	purchase := models.Purchase{
		SupplierId:   &product.SupplierId,                                                                      // 供应商ID
		AlbaranId:    &albaranId,                                                                               // 采购单号
		Cash:         nil,                                                                                      // 现金金额（盘点单为null）
		Discount:     nil,                                                                                      // 折扣（盘点单为null）
		EmployeeId:   &[]int{1}[0],                                                                             // 员工ID
		ExchangeRate: 0,                                                                                        // 汇率
		FacturaId:    &[]int{0}[0],                                                                             // 发票ID
		OrderDate:    &now,                                                                                     // 订单日期
		OrderType:    &[]int{0}[0],                                                                             // 订单类型
		Paid:         nil,                                                                                      // 已付金额（盘点单为null）
		SubTotal:     0,                                                                                        // 小计
		Status:       "已结账",                                                                                    // 状态
		Remark:       fmt.Sprintf("仓库盘点： 条码：%s 库存: %v 盘点: %v", product.BarCode, oldUnitsInStock, unitsInStock), // 备注
		Size:         &[]float64{0}[0],                                                                         // 尺寸
		ShipAddress:  "PDA",                                                                                    // 发货地址
	}

	// 保存盘点单到数据库
	result = tx.Create(&purchase)
	if result.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新商品库存失败",
		})
		return
	}

	// 创建盘点单明细记录
	purchaseDetails := models.PurchaseDetails{
		OrderId:            &purchase.OrderId,                                                                                                                                  // 订单ID
		ProductId:          int(product.ProductId),                                                                                                                             // 商品ID
		UserCode:           &product.UserCode,                                                                                                                                  // 用户编码
		BarCode:            &product.BarCode,                                                                                                                                   // 条形码
		ProductName:        &product.ProductName,                                                                                                                               // 商品名称
		Quantity:           newUnit,                                                                                                                                            // 数量（库存差异）
		Discount:           &[]int{0}[0],                                                                                                                                       // 折扣
		Tax:                &product.Tax,                                                                                                                                       // 税率
		Height:             &[]float64{0}[0],                                                                                                                                   // 高度
		Length:             &[]float64{0}[0],                                                                                                                                   // 长度
		Weight:             &[]float64{0}[0],                                                                                                                                   // 重量
		Width:              &[]float64{0}[0],                                                                                                                                   // 宽度
		UnitPrice:          product.PricePurchase,                                                                                                                              // 单价
		PriceSale:          &product.UnitPrice,                                                                                                                                 // 销售价
		QuantityPerUnit:    &[]int{0}[0],                                                                                                                                       // 每单位数量
		Bultos:             &[]int{0}[0],                                                                                                                                       // 包装数量
		QuantityPerUnit2nd: &[]int{0}[0],                                                                                                                                       // 第二单位数量
		Bultos2nd:          &[]int{0}[0],                                                                                                                                       // 第二包装数量
		Fjson:              `{"StockID":1,"StockName":"","OrgQtq":"","SQL_QtyOrg":null,"check":"","strPanDian":null,"StorageLocation":null,"Moreinfo1":null,"Moreinfo2":null}`, // JSON数据
		PriceFactory:       &[]float64{0}[0],                                                                                                                                   // 工厂价格
		Option1:            &[]int{0}[0],                                                                                                                                       // 选项1
		Entrada:            &[]float64{0}[0],                                                                                                                                   // 入库数量
	}

	// 保存盘点单明细到数据库
	result = tx.Create(&purchaseDetails)
	if result.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新商品库存失败",
		})
		return
	}

	// 提交事务
	tx.Commit()

	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "更新商品库存成功",
	})
}

func (ic *InventoryController) GetTopPurchases() ([]models.PurchaseWithSupplier, error) {
	var orderIDs []int
	if err := ic.db.Table("Purchase AS p").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Select("TOP 30 p.OrderID").
		Order("p.AlbaranID DESC").
		Pluck("p.OrderID", &orderIDs); err.Error != nil {
		return nil, err.Error
	}

	if len(orderIDs) == 0 {
		return []models.PurchaseWithSupplier{}, nil
	}

	var results []models.PurchaseWithSupplier
	if err := ic.db.Table("Purchase AS p").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Where("p.OrderID IN ?", orderIDs).
		Select("p.*, s.CompanyName AS supplier_company_name").
		Order("p.AlbaranID DESC").
		Scan(&results); err.Error != nil {
		return nil, err.Error
	}

	return results, nil
}
