// Package controllers 包含所有的HTTP控制器
// 负责处理HTTP请求和响应，实现RESTful API接口
package controllers

import (
	"fmt"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strconv"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

// parseFloat64 将字符串转换为float64，如果转换失败则返回0
func parseFloat64(s string) float64 {
	if s == "" {
		return 0
	}
	val, err := strconv.ParseFloat(s, 64)
	if err != nil {
		log.Printf("警告: 无法解析float64值 '%s': %v", s, err)
		return 0
	}
	return val
}

// parseFloat64Ptr 将字符串转换为*float64指针，如果转换失败则返回nil
func parseFloat64Ptr(s string) *float64 {
	if s == "" {
		return nil
	}
	val, err := strconv.ParseFloat(s, 64)
	if err != nil {
		log.Printf("警告: 无法解析float64指针值 '%s': %v", s, err)
		return nil
	}
	return &val
}

// parseInt 将字符串转换为int，如果转换失败则返回1（默认值）
func parseInt(s string) int {
	if s == "" {
		return 1
	}
	val, err := strconv.Atoi(s)
	if err != nil {
		log.Printf("警告: 无法解析int值 '%s': %v", s, err)
		return 1
	}
	return val
}

// parseFloat32 将字符串转换为float32，如果转换失败则返回0
func parseFloat32(s string) float32 {
	if s == "" {
		return 0
	}
	val, err := strconv.ParseFloat(s, 32)
	if err != nil {
		log.Printf("警告: 无法解析float32值 '%s': %v", s, err)
		return 0
	}
	return float32(val)
}

// ProductController 产品控制器
// 负责处理所有与产品相关的HTTP请求，包括商品的增删改查、扫描等功能
type ProductController struct {
	db *gorm.DB // 数据库连接实例
}

// NewProductController 创建新的产品控制器实例
func NewProductController(db *gorm.DB) *ProductController {
	log.Println("创建产品控制器实例")
	return &ProductController{db: db}
}

// getFloat64Value 获取float64指针的值，如果为nil则返回0
func getFloat64Value(v *float64) float64 {
	if v == nil {
		return 0
	}
	return *v
}

// getFloat64String 将float64指针转换为字符串，如果为nil则返回空字符串
func getFloat64String(v *float64) string {
	if v == nil {
		return ""
	}
	return strconv.FormatFloat(*v, 'f', -1, 64)
}

// Add 添加商品接口
// 用于添加新商品到数据库，包含重复检查逻辑
// 路由: POST /articulo/add
func (pc *ProductController) Add(c *gin.Context) {
	log.Println("=== 处理添加商品请求 ===")

	var product models.AddProductRequest
	// 绑定JSON请求体到product结构体
	if err := c.ShouldBindJSON(&product); err != nil {
		log.Printf("错误: 请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收添加商品请求: UserCode=%s, BarCode=%s, ProductName=%s",
		product.UserCode, product.BarCode, product.ProductName)

	// 检查商品是否已存在 - 通过usercode查询
	log.Printf("检查商品是否已存在: UserCode=%s", product.UserCode)
	var existingProduct models.Products
	result := pc.db.Where("UserCode = ?", product.UserCode).Select("TOP 1 *").Find(&existingProduct)
	if result.Error != nil {
		log.Printf("警告: 查询异常，UserCode=%s，err:%v", product.UserCode, result.Error.Error())
		// 商品已存在，返回错误信息
		c.JSON(http.StatusOK, gin.H{
			"code":    -1,
			"message": "商品查询异常",
		})
		return
	}
	if existingProduct.ProductId != 0 {
		log.Printf("警告: 商品已存在，UserCode=%s", product.UserCode)
		// 商品已存在，返回错误信息
		c.JSON(http.StatusOK, gin.H{
			"code":    1,
			"message": "商品ID已经存在",
		})
		return
	}

	log.Println("商品不存在，可以添加新商品")
	var newID int64
	pc.db.Raw(`INSERT INTO Products (
			UserCode, BarCode, ProductName, ProductName2,
			PricePurchase, UnitPrice, UnitPrice2, UnitsInStock, Discount, Tax, Remark,QuantityPerUnit,QuantityPerUnit2nd
		) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,"1","1")`, product.UserCode, product.BarCode, product.ProductName, product.ProductName2,
		product.PricePurchase, product.UnitPrice, product.UnitPrice2, product.UnitsInStock, product.Discount, product.Tax, product.Observacion).Scan(&newID)

	log.Printf("商品创建成功，ID: %d", newID)
	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "添加商品成功",
	})
}

// Scan 扫描商品接口
// 通过usercode或barcode扫描查找商品信息
// 路由: POST /articulo/scan
func (pc *ProductController) Scan(c *gin.Context) {
	log.Println("=== 处理扫描商品请求 ===")

	// 定义请求结构体
	var request struct {
		UserCode string `json:"usercode"` // 用户编码或条形码
	}

	// 绑定JSON请求体
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 扫描请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("扫描商品请求: UserCode=%s", request.UserCode)

	var product models.Products

	// 先通过usercode查找商品
	log.Printf("通过UserCode查找商品: %s", request.UserCode)
	result := pc.db.Where("UserCode = ?", request.UserCode).Select("TOP 1 *").Find(&product)
	if result.Error != nil || product.ProductId == 0 {
		log.Printf("通过UserCode未找到商品，尝试通过BarCode查找: %s", request.UserCode)
		// 如果没找到，再通过barcode查找
		result = pc.db.Where("barcode = ?", request.UserCode).Select("TOP 1 *").Find(&product)
		if result.Error != nil || product.ProductId == 0 {
			log.Printf("错误: 未找到商品，UserCode/BarCode=%s", request.UserCode)
			// 两种方式都没找到商品
			c.JSON(http.StatusNotFound, gin.H{
				"code":    -1,
				"message": "未找到商品",
			})
			return
		}
		log.Printf("通过BarCode找到商品: ID=%d", product.ProductId)
	} else {
		log.Printf("通过UserCode找到商品: ID=%d", product.ProductId)
	}

	// 检查商品图片是否存在，如果存在则设置图片路径
	imgPath := filepath.Join("static", "img", product.UserCode+".jpg")
	log.Printf("检查商品图片: %s", imgPath)
	if _, err := os.Stat(imgPath); err == nil {
		product.Img = "/img/" + product.UserCode + ".jpg"
		log.Printf("商品图片存在: %s", product.Img)
	} else {
		log.Printf("商品图片不存在: %s", imgPath)
	}

	// 将Products转换为Product
	productData := models.Product{
		ProductID:     fmt.Sprintf("%d", product.ProductId),
		BarCode:       product.BarCode,
		UserCode:      product.UserCode,
		ProductName:   product.ProductName,
		ProductName2:  product.ProductName2,
		PricePurchase: product.PricePurchase,
		UnitPrice:     product.UnitPrice,
		UnitPrice2:    getFloat64Value(product.UnitPrice2),
		Tax:           product.Tax,
		SupplierID:    fmt.Sprintf("%d", product.SupplierId),
		CategoryID:    fmt.Sprintf("%d", product.CategoryId),
		Discount:      product.Discount,
		Height:        getFloat64String(product.Height),
		Width:         getFloat64String(product.Width),
		Color:         product.Color,
		UnitsInStock:  product.UnitsInStock,
	}

	// 创建ScanResponse
	response := models.ScanResponse{
		Code:    "200",
		Message: "查询成功",
		Data:    productData,
	}

	log.Printf("查询商品成功: ID=%d, ProductName=%s", product.ProductId, product.ProductName)
	// 返回ScanResponse
	c.JSON(http.StatusOK, response)
}

// Scan6 通过GET参数扫描商品接口
// 通过URL参数中的id扫描查找商品信息
// 路由: GET /articulo/scan6
func (pc *ProductController) Scan6(c *gin.Context) {
	log.Println("=== 处理GET参数扫描商品请求 ===")

	// 获取URL参数中的id
	id := c.Query("id")
	if id == "" {
		log.Println("错误: ID参数为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "ID参数不能为空",
		})
		return
	}

	log.Printf("GET参数扫描商品: ID=%s", id)

	var product models.Products

	// 先通过usercode查找
	log.Printf("通过UserCode查找商品: %s", id)
	result := pc.db.Where("UserCode = ?", id).Select("TOP 1 *").Find(&product)
	if result.Error != nil {
		log.Printf("通过UserCode未找到商品，尝试通过BarCode查找: %s", id)
		// 如果没找到，再通过barcode查找
		result = pc.db.Where("BarCode = ?", id).Select("TOP 1 *").Find(&product)
		if result.Error != nil {
			log.Printf("错误: 未找到商品，ID=%s", id)
			c.JSON(http.StatusNotFound, gin.H{
				"code":    -1,
				"message": "未找到商品",
			})
			return
		}
		log.Printf("通过BarCode找到商品: ID=%d", product.ProductId)
	} else {
		log.Printf("通过UserCode找到商品: ID=%d", product.ProductId)
	}

	log.Printf("GET参数扫描商品成功: ID=%d, ProductName=%s", product.ProductId, product.ProductName)
	// 返回商品信息
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "扫描成功",
		"data":    product,
	})
}

// Update 更新商品接口
// 用于更新商品信息，特别是价格信息
// 路由: POST /articulo/update
func (pc *ProductController) Update(c *gin.Context) {
	var request models.UpdateProductRequest
	// 绑定JSON请求体
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 更新商品请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收更新商品请求: ProductID=%s", request.ProductID)

	// 先查询现有商品
	var existingProduct models.Products
	result := pc.db.Where("ProductID = ?", request.ProductID).Select("TOP 1 *").Find(&existingProduct)
	if result.Error != nil {
		log.Printf("错误: 未找到商品，ProductID=%s", request.ProductID)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "商品不存在",
		})
		return
	}
	if existingProduct.ProductId == 0 {
		log.Printf("错误: 未找到商品，ProductID=%s", request.ProductID)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "商品不存在",
		})
		return
	}

	// 构建更新字段映射，只更新有值的字段
	updates := make(map[string]interface{})

	// 检查并添加需要更新的字段
	if request.UserCode != "" {
		updates["UserCode"] = request.UserCode
	}
	if request.BarCode != "" {
		updates["BarCode"] = request.BarCode
	}
	if request.ProductName != "" {
		updates["ProductName"] = request.ProductName
	}
	if request.ProductName2 != "" {
		updates["ProductName2"] = request.ProductName2
	}
	if request.PricePurchase != "" {
		pricePurchase := parseFloat64(request.PricePurchase)
		updates["PricePurchase"] = pricePurchase
	}
	if request.UnitPrice != "" {
		unitPrice := parseFloat64(request.UnitPrice)
		updates["UnitPrice"] = unitPrice
	}
	if request.UnitPrice2 != "" {
		unitPrice2 := parseFloat64Ptr(request.UnitPrice2)
		updates["UnitPrice2"] = unitPrice2
	}
	if request.CategoryID != "" {
		categoryID := parseInt(request.CategoryID)
		updates["CategoryID"] = categoryID
	}
	if request.SupplierID != "" {
		supplierID := parseInt(request.SupplierID)
		updates["SupplierID"] = supplierID
	}
	if request.UnitsInStock != "" {
		unitsInStock := parseFloat64(request.UnitsInStock)
		updates["UnitsInStock"] = unitsInStock
	}
	if request.Discount != "" {
		discount := parseInt(request.Discount)
		updates["Discount"] = discount
	}
	if request.Tax != "" {
		tax := parseFloat32(request.Tax)
		updates["Tax"] = tax
	}

	// 如果没有需要更新的字段，直接返回成功
	if len(updates) == 0 {
		log.Printf("没有需要更新的字段，ProductID=%s", request.ProductID)
		c.JSON(http.StatusOK, gin.H{
			"code":    200,
			"message": "没有需要更新的字段",
		})
		return
	}

	// 执行更新操作
	log.Printf("更新商品字段: %v", updates)
	result = pc.db.Model(&existingProduct).Updates(updates)
	if result.Error != nil {
		log.Printf("错误: 更新商品失败: %v", result.Error)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	log.Printf("商品更新成功，ProductID=%s", request.ProductID)
	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "更新商品成功",
	})
}

// GetClass 获取商品类别接口
// 返回所有商品类别信息
// 路由: POST /articulo/getClass
func (pc *ProductController) GetClass(c *gin.Context) {
	var categories []models.Categories
	// 查询所有商品类别
	result := pc.db.Find(&categories)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取失败",
		})
		return
	}

	// 返回类别列表
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取商品类别成功",
		"data":    categories,
	})
}

// GetSupplier 获取供应商接口
// 返回所有供应商信息
// 路由: POST /articulo/supplier
func (pc *ProductController) GetSupplier(c *gin.Context) {
	var suppliers []models.Suppliers
	// 查询所有供应商
	result := pc.db.Find(&suppliers)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取失败",
		})
		return
	}

	// 返回供应商列表
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取商品类别成功",
		"data":    suppliers,
	})
}

// UpdateUnitsInStock 更新库存接口
// 用于更新指定商品的库存数量
// 路由: POST /articulo/updateUnitsInStock
func (pc *ProductController) UpdateUnitsInStock(c *gin.Context) {
	// 定义请求结构体
	var request struct {
		ProductId    string `json:"productId"`    // 商品ID
		UnitsInStock string `json:"unitsInStock"` // 新的库存数量
	}

	// 绑定JSON请求体
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	// 更新商品库存
	//    @Update({"UPDATE Products SET UnitsInStock = COALESCE(UnitsInStock, 0) + #{unitsInStock} WHERE ProductID = #{productId}"})
	result := pc.db.Model(&models.Products{}).Where("ProductID = ?", request.ProductId).Update("UnitsInStock", gorm.Expr("COALESCE(UnitsInStock, 0) + ?", request.UnitsInStock))
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新库存失败",
		})
		return
	}

	// 检查是否成功更新
	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "商品不存在",
		})
		return
	}

	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "库存更新成功",
	})
}
