// Package controllers 包含所有的HTTP控制器
// 负责处理HTTP请求和响应，实现RESTful API接口
package controllers

import (
	"log"
	"net/http"
	"strconv"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

// PurchaseController 采购控制器
// 负责处理所有与采购相关的HTTP请求，包括采购单的增删改查等功能
type PurchaseController struct {
	db *gorm.DB // 数据库连接实例
}

// NewPurchaseController 创建新的采购控制器实例
// 参数: db - 数据库连接实例
// 返回: PurchaseController实例
func NewPurchaseController(db *gorm.DB) *PurchaseController {
	log.Println("创建采购控制器实例")
	return &PurchaseController{db: db}
}

// GetByOrderId 根据订单ID获取采购单接口
// 根据订单ID查询采购单详细信息
// 路由: GET /purchase/:orderId
// 路径参数: orderId - 订单ID
func (pc *PurchaseController) GetByOrderId(c *gin.Context) {
	log.Println("=== 处理根据订单ID获取采购单请求 ===")

	// 获取路径参数中的订单ID
	orderIdStr := c.Param("orderId")
	log.Printf("获取订单ID参数: %s", orderIdStr)

	orderId, err := strconv.Atoi(orderIdStr)
	if err != nil {
		log.Printf("错误: 订单ID格式错误: %s, 错误: %v", orderIdStr, err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "订单ID格式错误",
		})
		return
	}

	log.Printf("查询订单ID: %d", orderId)
	var purchase models.PurchaseWithSupplier
	// 根据订单ID查询采购单，包含供应商公司名称和锁定状态
	// SELECT p.*, s.companyname AS supplierCompanyName FROM Purchase p LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID WHERE p.AlbaranID = #{orderId}
	err = pc.db.Raw(`
    SELECT p.*, s.CompanyName AS SupplierCompanyName
    FROM Purchase p 
    LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID 
    WHERE p.AlbaranID = ?
	`, orderId).Scan(&purchase).Error
	//result := pc.db.Where("OrderID = ?", orderId).Select("TOP 1 *").Find(&purchase)
	if err != nil || purchase.OrderId == 0 {
		log.Printf("错误: 订单不存在，OrderID=%d, 错误: %v", orderId, err.Error)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "订单不存在",
		})
		return
	}

	log.Printf("成功获取订单: OrderID=%d", orderId)
	// 返回采购单信息
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取订单成功",
		"data":    purchase,
	})
}

// GetAllPurchases 获取所有采购单
func (pc *PurchaseController) GetAllPurchases(c *gin.Context) {
	log.Println("=== 处理获取所有采购单请求 ===")
	var orderIDs []int
	var purchases []models.PurchaseWithSupplier
	log.Println("正在查询所有采购单...")
	// 第一步：获取符合条件的 OrderID（按 AlbaranID 倒序，最多 50 条）
	err := pc.db.Raw(`
    SELECT  OrderID
    FROM Purchase
    WHERE Cash IS NOT NULL
    ORDER BY AlbaranID DESC`).Scan(&orderIDs).Error
	if err != nil || len(orderIDs) == 0 {
		log.Printf("错误: 获取采购单列表失败: %v", err.Error())
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取订单列表失败",
		})
		return
	}
	// 第二步：根据 orderID 查询详情并联表 Supplier 表
	err = pc.db.Model(&models.Purchase{}).
		Select("p.*, s.CompanyName AS SupplierCompanyName").
		Table("Purchase as p").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Where("p.OrderID IN ?", orderIDs).
		Where("p.SubTotal > 0").
		Order("p.OrderID DESC").
		Scan(&purchases).Error

	if err != nil {
		log.Printf("错误: 获取采购单列表失败: %v", err.Error())
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取订单列表失败",
		})
		return
	}

	log.Printf("成功获取采购单列表，共 %d 条记录", len(purchases))
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取订单列表成功",
		"data":    purchases,
	})
}

// AddPurchase 添加采购单
func (pc *PurchaseController) AddPurchase(c *gin.Context) {
	log.Println("=== 处理添加采购单请求 ===")

	var purchase models.Purchase
	if err := c.ShouldBindJSON(&purchase); err != nil {
		log.Printf("错误: 采购单请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收添加采购单请求: OrderID=%d", purchase.OrderId)

	log.Println("正在创建采购单...")
	result := pc.db.Create(&purchase)
	if result.Error != nil {
		log.Printf("错误: 创建采购单失败: %v", result.Error)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "订单添加失败",
		})
		return
	}

	log.Printf("采购单创建成功，ID: %d", purchase.OrderId)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "订单添加成功",
	})
}

// UpdatePurchase 更新采购单
func (pc *PurchaseController) UpdatePurchase(c *gin.Context) {
	log.Println("=== 处理更新采购单请求 ===")

	var purchase models.Purchase
	if err := c.ShouldBindJSON(&purchase); err != nil {
		log.Printf("错误: 更新采购单请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收更新采购单请求: OrderID=%d", purchase.OrderId)

	log.Println("正在更新采购单...")
	result := pc.db.Model(&purchase).Updates(&purchase)
	if result.Error != nil {
		log.Printf("错误: 更新采购单失败: %v", result.Error)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "订单更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		log.Printf("警告: 采购单不存在或更新失败，OrderID=%d", purchase.OrderId)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "订单不存在或更新失败",
		})
		return
	}

	log.Printf("采购单更新成功，影响行数: %d", result.RowsAffected)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "订单更新成功",
	})
}

// DeletePurchase 删除采购单
func (pc *PurchaseController) DeletePurchase(c *gin.Context) {
	log.Println("=== 处理删除采购单请求 ===")

	orderIdStr := c.Param("orderId")
	log.Printf("获取删除订单ID参数: %s", orderIdStr)

	orderId, err := strconv.Atoi(orderIdStr)
	if err != nil {
		log.Printf("错误: 订单ID格式错误: %s, 错误: %v", orderIdStr, err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "订单ID格式错误",
		})
		return
	}

	log.Printf("正在删除订单ID: %d", orderId)
	result := pc.db.Where("OrderID = ?", orderId).Delete(&models.Purchase{})
	if result.Error != nil {
		log.Printf("错误: 删除采购单失败: %v", result.Error)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "订单删除失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		log.Printf("警告: 采购单不存在或删除失败，OrderID=%d", orderId)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "订单不存在或删除失败",
		})
		return
	}

	log.Printf("采购单删除成功，影响行数: %d", result.RowsAffected)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "订单删除成功",
	})
}

// UpdateRemark 更新备注字段
func (pc *PurchaseController) UpdateRemark(c *gin.Context) {
	log.Println("=== 处理更新备注字段请求 ===")

	var request models.UpdateRemark
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 更新备注请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收更新备注请求: OrderID=%d, Remark=%s", request.OrderId, request.Remark)

	log.Println("正在更新备注字段...")
	//@Update({"UPDATE Purchase SET Bloqueado = -1 WHERE OrderID = #{orderId}"})
	result := pc.db.Model(&models.Purchase{}).Where("OrderID = ?", request.OrderId).Update("Bloqueado", 1)
	if result.Error != nil {
		log.Printf("错误: 更新备注失败: %v", result.Error)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		log.Printf("警告: 未找到指定的OrderID: %d", request.OrderId)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 OrderID",
		})
		return
	}

	log.Printf("备注更新成功，影响行数: %d", result.RowsAffected)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "更新成功",
	})
}

// UpdateSubTotal 更新小计字段
func (pc *PurchaseController) UpdateSubTotal(c *gin.Context) {
	var request struct {
		OrderId  string `json:"orderId"` // 商品ID
		SubTotal string `json:"subTotal"`
	}
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	//   @Update({"UPDATE Purchase SET subTotal = #{subTotal} WHERE OrderID = #{orderId}"})
	result := pc.db.Model(&models.Purchase{}).Where("OrderID = ?", request.OrderId).Update("subTotal", request.SubTotal)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 OrderID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "subTotal 字段更新成功",
	})
}
