package controllers

import (
	"fmt"
	"log"
	"math/big"
	"net/http"
	"strconv"
	"strings"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

type PurchaseDetailsController struct {
	db *gorm.DB
}

func NewPurchaseDetailsController(db *gorm.DB) *PurchaseDetailsController {
	return &PurchaseDetailsController{db: db}
}

// GetAllPurchaseDetails 获取所有采购单明细
func (pdc *PurchaseDetailsController) GetAllPurchaseDetails(c *gin.Context) {
	var detailsList []models.PurchaseDetails
	result := pdc.db.Find(&detailsList)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取成功",
		"data":    detailsList,
	})
}

// GetByOrderId 根据订单ID获取采购单明细
func (pdc *PurchaseDetailsController) GetByOrderId(c *gin.Context) {
	orderIdStr := c.Query("orderId")

	var detailsList []models.PurchaseDetailsWithProduct
	//SELECT pd.*, p.pricepurchase, p.unitprice AS pdunitprice, p.unitprice4 FROM [[Purchase Details]] pd LEFT JOIN Products p ON pd.ProductID = p.ProductID WHERE pd.OrderID = #{orderId}"
	result := pdc.db.Raw("SELECT pd.*, p.pricepurchase, p.unitprice AS pdunitprice, p.unitprice4 FROM [Purchase Details] pd LEFT JOIN Products p ON pd.ProductID = p.ProductID WHERE pd.OrderID = ?", orderIdStr).Find(&detailsList)

	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取成功",
		"data":    detailsList,
	})
}

// UpdateProductUnitPrice2 更新商品单价（通过ProductId）
func (pdc *PurchaseDetailsController) UpdateProductUnitPrice2(c *gin.Context) {
	var request models.UpdateProductUnitPrice
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	//   @Update({"UPDATE Products SET unitprice4 = #{unitPrice} WHERE ProductID = #{productId}"})
	result := pdc.db.Model(&models.Products{}).Where("ProductId = ?", request.ProductId).Update("unitprice4", request.UnitPrice)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 ProductID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "unitprice 字段更新成功",
	})
}

// UpdateProductUnitPrice 更新商品单价（通过ProductId）
func (pdc *PurchaseDetailsController) UpdateProductUnitPrice(c *gin.Context) {
	var request models.UpdateProductUnitPrice
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	//@Update({"UPDATE Products SET unitprice = #{unitPrice} WHERE ProductID = #{productId}"})
	result := pdc.db.Model(&models.Products{}).Where("ProductId = ?", request.ProductId).Update("unitprice", request.UnitPrice)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 ProductID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "unitprice 字段更新成功",
	})
}

// UpdateUnitPrice 更新单价（通过ID）
func (pdc *PurchaseDetailsController) UpdateUnitPrice(c *gin.Context) {
	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	//UPDATE [[Purchase Details]] SET unitprice = #{unitPrice} WHERE ID = #{id}
	result := pdc.db.Model(&models.PurchaseDetails{}).Where("ID = ?", request.Id).Update("unitprice", request.UnitPrice)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 BarCode",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "unitprice 字段更新成功",
	})
}

// UpdateEntrada 更新Entrada字段
func (pdc *PurchaseDetailsController) UpdateEntrada(c *gin.Context) {
	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	id, err := strconv.Atoi(request.Id)
	if err != nil {
		log.Printf("错误: Id 类型转换错误:%v", request.Id)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	entrada, err := strconv.ParseFloat(request.Entrada, 64)
	if err != nil {
		log.Printf("错误: entrada 类型转换错误:%v", request.Entrada)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	//UPDATE [[Purchase Details]] SET Entrada = COALESCE(Entrada, 0) + #{entrada} WHERE ID = #{id}
	result := pdc.db.Exec("UPDATE [Purchase Details] SET Entrada = COALESCE(Entrada, 0) + ? WHERE ID = ?", entrada, id)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的ID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "Entrada 字段更新成功",
	})
}

// GetPurchaseDetailsById 根据条码获取采购单明细
func (pdc *PurchaseDetailsController) GetPurchaseDetailsById(c *gin.Context) {
	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	var details models.PurchaseDetails
	result := pdc.db.Where("barcode = ? AND order_id = ?", request.BarCode, request.OrderId).First(&details)
	if result.Error != nil {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "记录未找到",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取成功",
		"data":    details,
	})
}

// GetByBarCode 根据条码获取采购单明细（包含产品信息）
// 实现联合查询：SELECT pd.*, p.pricepurchase, p.unitprice AS pdunitprice, p.unitprice4
// FROM [[Purchase Details]] pd LEFT JOIN Products p ON pd.ProductID = p.ProductID
// WHERE pd.BarCode = #{barCode} AND pd.OrderID = #{orderID}
func (pdc *PurchaseDetailsController) GetByBarCode(c *gin.Context) {
	log.Println("=== 处理根据条码获取采购单明细请求 ===")

	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	orderId, err := strconv.Atoi(request.OrderId)
	if err != nil {
		log.Printf("错误: 请求参数错误: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收查询请求: BarCode=%s, OrderID=%d", request.BarCode, orderId)

	// 执行联合查询
	var details models.PurchaseDetailsWithProduct
	result := pdc.db.Table("PurchaseDetails").
		Select("TOP 1 [Purchase Details].*, Products.PricePurchase as pricepurchase, Products.UnitPrice as pdunitprice, Products.UnitPrice4 as unitprice4").
		Joins("LEFT JOIN Products ON [Purchase Details].ProductID = Products.ProductID").
		Where("[Purchase Details].BarCode = ? AND [Purchase Details].OrderID = ?", request.BarCode, orderId).
		Find(&details)

	if result.Error != nil || *details.OrderId == 0 {
		log.Printf("错误: 查询失败，BarCode=%s, OrderID=%d, 错误: %v", request.BarCode, orderId, result.Error)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "记录未找到",
		})
		return
	}

	log.Printf("成功获取采购单明细: BarCode=%s, OrderID=%d", request.BarCode, request.OrderId)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取成功",
		"data":    details,
	})
}

// GetByUserCode 根据用户编码获取采购单明细
func (pdc *PurchaseDetailsController) GetByUserCode(c *gin.Context) {
	log.Println("=== 处理根据条码获取采购单明细请求 ===")

	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收查询请求: BarCode=%s, OrderID=%d", request.BarCode, request.OrderId)

	// 执行联合查询
	var details models.PurchaseDetailsWithProduct
	result := pdc.db.Table("[Purchase Details]").
		Select("Top 1 [Purchase Details].*, Products.PricePurchase as pricepurchase, Products.UnitPrice as pdunitprice, Products.UnitPrice4 as unitprice4").
		Joins("LEFT JOIN Products ON [Purchase Details].ProductID = Products.ProductID").
		Where("[Purchase Details].UserCode = ? AND [Purchase Details].OrderID = ?", request.BarCode, request.OrderId).
		Find(&details)

	if result.Error != nil || details.ProductId == 0 {
		log.Printf("错误: 查询失败，BarCode=%s, OrderID=%d, 错误: %v", request.BarCode, request.OrderId, result.Error)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "记录未找到",
		})
		return
	}

	log.Printf("成功获取采购单明细: BarCode=%s, OrderID=%d", request.BarCode, request.OrderId)
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取成功",
		"data":    details,
	})
}

// AddPurchaseDetails 添加采购单明细
func (pdc *PurchaseDetailsController) AddPurchaseDetails(c *gin.Context) {
	var purchaseDetails models.PurchaseDetails
	if err := c.ShouldBindJSON(&purchaseDetails); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	result := pdc.db.Create(&purchaseDetails)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "添加失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "添加成功",
	})
}

// UpdatePurchaseDetails 更新采购单明细
func (pdc *PurchaseDetailsController) UpdatePurchaseDetails(c *gin.Context) {
	var purchaseDetails models.PurchaseDetails
	if err := c.ShouldBindJSON(&purchaseDetails); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	result := pdc.db.Model(&purchaseDetails).Updates(&purchaseDetails)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "记录未找到或更新失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "更新成功",
	})
}

// DeletePurchaseDetails 删除采购单明细
func (pdc *PurchaseDetailsController) DeletePurchaseDetails(c *gin.Context) {
	idStr := c.Param("id")
	id, err := strconv.Atoi(idStr)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "ID格式错误",
		})
		return
	}

	result := pdc.db.Where("ID = ?", id).Delete(&models.PurchaseDetails{})
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "删除失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "记录未找到或删除失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "删除成功",
	})
}

// UpdateQuantity 更新数量并处理备注
func (pdc *PurchaseDetailsController) UpdateQuantity(c *gin.Context) {
	var request models.UpdateQuantity
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	// 获取进货单
	var purchase models.PurchaseWithSupplier
	////  @Select({"SELECT p.*, s.companyname AS supplierCompanyName FROM Purchase p LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID WHERE p.AlbaranID = #{orderId}"})
	result := pdc.db.Raw("SELECT p.*, s.companyname AS SupplierCompanyName FROM Purchase p LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID WHERE p.AlbaranID = ?", request.OrderId).Find(&purchase)
	if result.Error != nil {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的订单",
		})
		return
	}

	// 获取进货单明细
	var purchaseDetailsList []models.PurchaseDetailsWithProduct
	//@Select({"SELECT pd.*, p.pricepurchase, p.unitprice AS pdunitprice, p.unitprice4 FROM [[Purchase Details]] pd LEFT JOIN Products p ON pd.ProductID = p.ProductID WHERE pd.OrderID = #{orderId}"})
	result = pdc.db.Raw("SELECT pd.*, p.pricepurchase, p.unitprice AS pdunitprice, p.unitprice4 FROM [Purchase Details] pd LEFT JOIN Products p ON pd.ProductID = p.ProductID WHERE pd.OrderID = ?", request.OrderId).Find(&purchaseDetailsList)
	if result.Error != nil {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    500,
			"message": "获取明细失败",
		})
		return
	}
	// 处理备注逻辑
	if len(purchaseDetailsList) > 0 {
		remark := ""
		for _, detail := range purchaseDetailsList {
			quantity := detail.Quantity
			entrada := detail.Entrada

			// 使用big包进行精确计算
			quantityBig := new(big.Float).SetFloat64(quantity)
			entradaBig := new(big.Float).SetFloat64(entrada)

			if entradaBig.Cmp(quantityBig) < 0 { // 实际数量比明细单数量小
				diff := new(big.Float).Sub(entradaBig, quantityBig)
				remark += " 条码：" + *detail.BarCode + " 差数：" + diff.Text('f', 0) + " 数量：" + quantityBig.Text('f', 0) + " 收货：" + entradaBig.Text('f', 0) + " ;"
			}
		}
		purchase.Remark = remark

	}
	// 更新备注字段
	////@Update({"UPDATE Purchase SET Remark = #{remark},Bloqueado = -1 WHERE OrderID = #{orderId}"})
	pdc.db.Model(&purchase).Where("OrderID = ?", purchase.OrderId).Update("Remark", purchase.Remark).Update("Bloqueado", -1)
	// 更新数量
	//@Update({"UPDATE [[Purchase Details]] SET Quantity = COALESCE(Entrada, 0) WHERE OrderID = #{orderId}"})
	result = pdc.db.Model(&models.PurchaseDetails{}).Where("OrderID = ?", request.OrderId).Update("Quantity", gorm.Expr("COALESCE(Entrada, 0)"))
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 OrderID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "Quantity 字段更新成功",
	})
}

// UpdateEntradaById 根据ID更新Entrada字段
func (pdc *PurchaseDetailsController) UpdateEntradaById(c *gin.Context) {
	var request models.UpdateEntradaRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	// UPDATE [[Purchase Details]] SET Entrada = Quantity WHERE ID = #{id}
	result := pdc.db.Model(&models.PurchaseDetails{}).Where("ID = ?", request.Id).Update("Entrada", gorm.Expr("Quantity"))
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 BarCode",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "Entrada 字段更新成功",
	})
}

// UpdateUnitPriceShenHe 审核更新单价
func (pdc *PurchaseDetailsController) UpdateUnitPriceShenHe(c *gin.Context) {
	var request struct {
		ProductId string `json:"productId"` // 商品ID
		UnitPrice string `json:"unitPrice"`
	}
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}
	// @Update({"UPDATE Products SET PricePurchase = #{unitPrice} WHERE ProductID = #{productId}"})
	result := pdc.db.Model(&models.Products{}).Where("ProductID = ?", request.ProductId).Update("PricePurchase", request.UnitPrice)
	if result.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新失败",
		})
		return
	}

	if result.RowsAffected == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到指定的 OrderID",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "UnitPrice 字段更新成功",
	})
}

// UpdateUnitPriceShenHe 审核更新单价
func (pdc *PurchaseDetailsController) UpdateUnitPriceShenHeBatch(c *gin.Context) {
	// 定义请求结构体为数组类型，直接接收数组作为入参
	var request []struct {
		ProductId string `json:"productId" binding:"required"` // 商品ID，必填
		UnitPrice string `json:"unitPrice" binding:"required"` // 价格字符串，必填
	}

	// 绑定并验证请求参数（直接绑定数组）
	if err := c.ShouldBindJSON(&request); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误：" + err.Error(),
		})
		return
	}

	// 验证数组不为空
	if len(request) == 0 {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "商品列表不能为空",
		})
		return
	}

	// 解析价格字符串为float64并验证
	productIDs := make([]string, 0, len(request))
	priceMap := make(map[string]float64)

	for i, item := range request {
		// 尝试将字符串价格转换为数字（支持多位小数如"30.0000"）
		price, err := strconv.ParseFloat(item.UnitPrice, 64)
		if err != nil {
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "第" + strconv.Itoa(i+1) + "个商品的价格格式错误，必须是有效的数字（支持小数）",
			})
			return
		}

		// 验证价格非负
		if price < 0 {
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "第" + strconv.Itoa(i+1) + "个商品的价格不能为负数",
			})
			return
		}

		productIDs = append(productIDs, item.ProductId)
		priceMap[item.ProductId] = price
	}

	// 使用事务确保数据一致性
	tx := pdc.db.Begin()
	if tx.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "开启事务失败：" + tx.Error.Error(),
		})
		return
	}
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 构建CASE语句进行批量更新
	caseStmt := "CASE ProductID "
	args := make([]interface{}, 0, 2*len(request))
	for _, item := range request {
		// 使用已解析的float64价格
		price, _ := priceMap[item.ProductId]
		caseStmt += "WHEN ? THEN ? "
		args = append(args, item.ProductId, price)
	}
	caseStmt += "END"

	// 执行批量更新
	result := tx.Model(&models.Products{}).
		Where("ProductID IN (?)", productIDs).
		Update("PricePurchase", gorm.Expr(caseStmt, args...))

	if result.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新数据库失败：" + result.Error.Error(),
		})
		return
	}

	// 检查是否有记录被更新
	if result.RowsAffected == 0 {
		tx.Rollback()
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到任何指定的商品ID，更新失败",
		})
		return
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败：" + err.Error(),
		})
		return
	}

	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "批量更新成功",
		//"data": gin.H{
		//	"updatedCount": result.RowsAffected,
		//	"productIds":   productIDs,
		//},
	})
}

func (pc *PurchaseDetailsController) BatchUpdateStockAndSubTotal(c *gin.Context) {
	// 定义批量操作的单项结构体
	type BatchItem struct {
		ProductId    string `json:"productId" binding:"required"`    // 商品ID
		UnitsInStock string `json:"unitsInStock" binding:"required"` // 库存数量
		OrderId      string `json:"orderId" binding:"required"`      // 订单ID
		SubTotal     string `json:"subTotal" binding:"required"`     // 小计金额
	}

	// 直接接收数组作为请求体
	var items []BatchItem
	if err := c.ShouldBindJSON(&items); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误: " + err.Error(),
		})
		return
	}

	// 检查数组是否为空
	if len(items) == 0 {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求数组不能为空",
		})
		return
	}

	// 开启数据库事务
	tx := pc.db.Begin()
	if tx.Error != nil {
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "开启事务失败: " + tx.Error.Error(),
		})
		return
	}
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 批量更新库存 - 修复参数类型问题
	var (
		stockCase   strings.Builder
		stockParams []interface{}
		productIds  []string
	)

	stockCase.WriteString("CASE ProductID ")
	for _, item := range items {
		// 使用位置参数而非命名参数，避免类型问题
		stockCase.WriteString(fmt.Sprintf("WHEN ? THEN COALESCE(UnitsInStock, 0) + ? "))
		stockParams = append(stockParams, item.ProductId, item.UnitsInStock)
		productIds = append(productIds, item.ProductId)
	}
	stockCase.WriteString("ELSE UnitsInStock END")

	// 执行库存批量更新
	stockResult := tx.Model(&models.Products{}).
		Where("ProductID IN (?)", productIds).
		Update("UnitsInStock", gorm.Expr(stockCase.String(), stockParams...))

	if stockResult.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "批量更新库存失败: " + stockResult.Error.Error(),
			"sql":     stockResult.Statement.SQL.String(),
		})
		return
	}

	// 检查库存更新影响的行数
	//if int(stockResult.RowsAffected) < len(productIds) {
	//	tx.Rollback()
	//	c.JSON(http.StatusNotFound, gin.H{
	//		"code":     404,
	//		"message":  "部分商品不存在",
	//		"affected": stockResult.RowsAffected,
	//		"requested": len(productIds),
	//	})
	//	return
	//}

	// 批量更新小计 - 修复参数类型问题
	var (
		subTotalCase   strings.Builder
		subTotalParams []interface{}
		orderIds       []string
	)

	subTotalCase.WriteString("CASE OrderID ")
	for _, item := range items {
		// 使用位置参数而非命名参数
		subTotalCase.WriteString(fmt.Sprintf("WHEN ? THEN ? "))
		subTotalParams = append(subTotalParams, item.OrderId, item.SubTotal)
		orderIds = append(orderIds, item.OrderId)
	}
	subTotalCase.WriteString("ELSE subTotal END")

	// 执行小计批量更新
	subTotalResult := tx.Model(&models.Purchase{}).
		Where("OrderID IN (?)", orderIds).
		Update("subTotal", gorm.Expr(subTotalCase.String(), subTotalParams...))

	if subTotalResult.Error != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "批量更新小计失败: " + subTotalResult.Error.Error(),
			"sql":     subTotalResult.Statement.SQL.String(),
		})
		return
	}

	//// 检查小计更新影响的行数
	//if int(subTotalResult.RowsAffected) < len(orderIds) {
	//	tx.Rollback()
	//	c.JSON(http.StatusNotFound, gin.H{
	//		"code":     404,
	//		"message":  "部分订单不存在",
	//		"affected": subTotalResult.RowsAffected,
	//		"requested": len(orderIds),
	//	})
	//	return
	//}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败: " + err.Error(),
		})
		return
	}

	// 所有操作成功完成
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": fmt.Sprintf("已成功批量更新%d项数据", len(items)),
		"count":   len(items),
	})
}
