package controllers

import (
	"log"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

type ReturnController struct {
	db *gorm.DB // 数据库连接实例
}

// NewReturnController 创建新的退货控制器实例
// 参数: db - 数据库连接实例
// 返回: ReturnController实例
func NewReturnController(db *gorm.DB) *ReturnController {
	log.Println("创建退货单实例")
	return &ReturnController{db: db}
}

// findProductByValue 根据输入值查找商品
// 按照ProductID、BarCode、UserCode的顺序查询，如果都不存在则返回错误
// 参数: tx - 数据库事务, value - 查询值
// 返回: 商品信息和错误
func (re *ReturnController) findProductByValue(tx *gorm.DB, value string) (models.Products, error) {
	var product models.Products
	var err error

	// 1. 首先尝试按ProductID查询
	//if productId, err := strconv.ParseInt(value, 10, 64); err == nil {
	//	err = tx.Raw("SELECT TOP 1 * FROM Products WHERE ProductID = ?", productId).Scan(&product).Error
	//	if err == nil && product.ProductId != 0 {
	//		log.Printf("通过ProductID找到商品: ProductID=%d, 名称=%s", product.ProductId, product.ProductName)
	//		return product, nil
	//	}
	//}

	// 2. 如果ProductID查询失败，尝试按BarCode查询
	err = tx.Raw("SELECT TOP 1 * FROM Products WHERE BarCode = ?", value).Scan(&product).Error
	if err == nil && product.ProductId != 0 {
		log.Printf("通过BarCode找到商品: BarCode=%s, ProductID=%d, 名称=%s", value, product.ProductId, product.ProductName)
		return product, nil
	}

	// 3. 如果BarCode查询失败，尝试按UserCode查询
	err = tx.Raw("SELECT TOP 1 * FROM Products WHERE UserCode = ?", value).Scan(&product).Error
	if err == nil && product.ProductId != 0 {
		log.Printf("通过UserCode找到商品: UserCode=%s, ProductID=%d, 名称=%s", value, product.ProductId, product.ProductName)
		return product, nil
	}

	// 4. 如果所有查询都失败，返回错误
	log.Printf("错误: 商品不存在, 查询值=%s (已尝试ProductID、BarCode、UserCode)", value)
	return models.Products{}, gorm.ErrRecordNotFound
}

// SubmitReturnGoods 提交退货单
// 处理多个商品的退货请求，更新相关表数据
// 路由: POST /returnGoods/submitReturnGoods
func (re *ReturnController) SubmitReturnGoods(c *gin.Context) {
	log.Println("=== 处理退货单请求 ===")

	// 解析请求参数
	var request models.SubmitReturnGoodsRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 请求参数解析失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误: " + err.Error(),
		})
		return
	}

	// 验证参数
	//if len(request.ArticuloList) == 0 {
	//	log.Printf("错误: 商品列表不能为空")
	//	c.JSON(http.StatusBadRequest, gin.H{
	//		"code":    400,
	//		"message": "商品列表不能为空",
	//	})
	//	return
	//}

	if request.EmpresaId == "" {
		log.Printf("错误: 企业ID不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "供应商不能为空",
		})
		return
	}

	// 开启数据库事务
	tx := re.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 生成新的退货订单ID（使用当前时间戳）
	now := time.Now()
	purchaseAlbaranID, err := re.GetNewLatestAlbaranID()
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    500,
			"message": err.Error(),
		})
	}
	purchaseOrderId, err := re.GetNewLatestOrderId()
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    500,
			"message": err.Error(),
		})
	}
	albaranId := 0
	returnOrderId := 0
	if *purchaseAlbaranID.AlbaranId > 0 {
		albaranId = *purchaseAlbaranID.AlbaranId + 1
	}
	if purchaseOrderId.OrderId > 0 {
		returnOrderId = purchaseOrderId.OrderId + 1
	}
	var totalSubTotal float64 = 0
	var processedProducts []map[string]interface{}

	// 处理每个商品
	for _, articulo := range request.ArticuloList {
		// 验证商品数量（必须为负数）
		if articulo.Amount >= 0 {
			log.Printf("错误: 退货数量必须为负数, 商品: %s, 数量: %.3f", articulo.Nombre, articulo.Amount)
			tx.Rollback()
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "退货数量必须为负数",
			})
			return
		}
		var product models.Products
		var err error
		// 当 value 为 0 号商品时，按 nombre 作为 ProductName 查询
		if articulo.Value == "0" {
			product.ProductName = articulo.Nombre
			product.ProductId = 0
			product.BarCode = "0"
			product.UserCode = "0"
		} else if articulo.Type == 0 {
			// 通过商品ID查询
			productId, err := strconv.ParseInt(articulo.Value, 10, 64)
			if err != nil {
				tx.Rollback()
				log.Printf("错误: 商品ID格式错误, ProductID=%s, 错误: %v", articulo.Value, err)
				c.JSON(http.StatusBadRequest, gin.H{
					"code":    400,
					"message": "商品ID格式错误: " + articulo.Value,
				})
				return
			}
			err = tx.Raw("SELECT TOP 1 * FROM Products WHERE ProductID = ?", productId).
				Scan(&product).Error
		} else if articulo.Type == 1 {
			// 通过条码查询
			err = tx.Raw("SELECT TOP 1 * FROM Products WHERE BarCode = ?", articulo.Value).
				Scan(&product).Error
		} else {
			tx.Rollback()
			log.Printf("错误: 无效的商品类型, type=%d", articulo.Type)
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "无效的商品类型",
			})
			return
		}
		if err != nil {
			tx.Rollback()
			log.Printf("错误: err:%v", err.Error())
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": err.Error(),
			})
			return
		}
		// 计算退货小计（负数）
		subTotal := articulo.Amount * articulo.Precio
		totalSubTotal += subTotal

		// 在PurchaseDetail表中插入退货明细记录
		returnPurchaseDetail := models.PurchaseDetails{
			OrderId:            &returnOrderId,
			ProductId:          int(product.ProductId),
			UserCode:           &product.UserCode,
			BarCode:            &product.BarCode,
			ProductName:        &product.ProductName,
			Quantity:           articulo.Amount, // 退货数量（负数）
			Discount:           nil,
			Tax:                &product.Tax,
			UnitPrice:          articulo.Precio, // 退货单价
			PriceSale:          &product.UnitPrice,
			QuantityPerUnit:    &product.QuantityPerUnit,
			Bultos:             nil,
			QuantityPerUnit2nd: &product.QuantityPerUnit2nd,
			Bultos2nd:          nil,
			Fjson:              "",
			PriceFactory:       product.PriceFactory,
			Height:             product.Height,
			Length:             product.Length,
			Weight:             product.Weight,
			Width:              product.Width,
			Option1:            &product.Option1,
			Action:             nil,
			BatchCode:          nil,
			Entrada:            nil, // 入库数量等于退货数量
		}

		if err := tx.Create(&returnPurchaseDetail).Error; err != nil {
			tx.Rollback()
			log.Printf("错误: 创建退货明细失败: %v", err)
			c.JSON(http.StatusInternalServerError, gin.H{
				"code":    500,
				"message": "创建退货明细失败",
			})
			return
		}
		if articulo.Value != "0" {
			// 更新Products表中的库存（增加库存，因为退货数量为负数）
			newStock := product.UnitsInStock + (-articulo.Amount)
			if err := tx.Model(&product).Where("ProductID = ?", product.ProductId).Update("UnitsInStock", newStock).Error; err != nil {
				tx.Rollback()
				log.Printf("错误: 更新商品库存失败: %v", err)
				c.JSON(http.StatusInternalServerError, gin.H{
					"code":    500,
					"message": "更新商品库存失败",
				})
				return
			}
		}

		// 记录处理的商品信息
		processedProducts = append(processedProducts, map[string]interface{}{
			"productId":   product.ProductId,
			"productName": product.ProductName,
			"amount":      articulo.Amount,
			"precio":      articulo.Precio,
			"subTotal":    subTotal,
		})

		log.Printf("处理商品: ProductID=%d, 名称=%s, 数量=%.3f, 单价=%.4f, 小计=%.4f",
			product.ProductId, product.ProductName, articulo.Amount, articulo.Precio, subTotal)
	}
	empresaId, _ := strconv.ParseInt(request.EmpresaId, 10, 64)
	supplierId := int(empresaId)
	orderType := 0
	// 在Purchase表中插入退货记录
	returnPurchase := models.Purchase{
		OrderId:         returnOrderId,
		SupplierId:      &supplierId, // 暂时设为nil，后续可根据需要调整
		AccountPayable:  nil,         // 应付账款为负数
		AlbaranId:       &albaranId,
		Amount:          nil,            // 金额为负数
		Cash:            &totalSubTotal, // 现金为负数
		CloseDate:       nil,
		CloseTime:       nil,
		Del:             nil,
		Discount:        nil,
		Employe:         nil,
		EmployeeId:      &supplierId, // 暂时设为nil，后续可根据需要调整
		ExchangeRate:    8.0,
		ExOid:           "",
		FacturaId:       nil,
		Final:           "1", // 状态为已结账
		Fjson:           "",
		Freight:         0,
		LimitDate:       nil,
		OrderDate:       &now,
		OrderTime:       nil,
		OrderType:       &orderType,
		Paid:            &totalSubTotal, // 已付金额为负数
		PayMode:         "Dinero",
		Remark:          "企业ID: " + request.EmpresaId,
		RequiredDate:    &now,
		ShipAddress:     "pda",
		ShipCity:        "",
		ShipCountry:     "",
		ShipName:        "",
		ShippedDate:     &now,
		ShipPostalCode:  "",
		ShipRegion:      "",
		ShipVia:         nil,
		Size:            nil,
		Status:          "已结账",         // 状态为已结账
		SubTotal:        totalSubTotal, // 小计为负数
		TaxInclude:      nil,
		ValueAddTax:     nil,
		Vouchers:        nil,
		VouchersId:      "",
		Uuid:            "",
		Sn:              "",
		FreePosUserName: "",
		FreePosRemark:   "",
		Bloqueado:       nil,
	}

	if err := tx.Create(&returnPurchase).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 创建退货订单失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "创建退货订单失败",
		})
		return
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("错误: 提交事务失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败",
		})
		return
	}

	log.Printf("退货单创建成功: ReturnOrderId=%d, 总金额=%.4f, 处理商品数量=%d",
		returnOrderId, totalSubTotal, len(processedProducts))

	// 返回成功响应
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "退货单创建成功",
		"data":    "",
	})
}

// ReturnGoodsList 获取退货单列表
// 查询所有退货订单（状态为"已结账"且金额为负数的订单）
// 路由: POST /returnGoods/listReturnGoods
func (re *ReturnController) ReturnGoodsList(c *gin.Context) {
	log.Println("=== 处理获取退货单列表请求 ===")
	bloqueado := c.PostForm("Bloqueado")
	orderId := c.PostForm("orderId") //订单编号
	// SupplierId
	supplierId := c.PostForm("supplierId") // 供应商ID
	var returnOrders []models.PurchaseWithSupplier

	// 查询退货订单（状态为"已结账"且金额为负数的订单），并关联供应商名称
	query := `
		SELECT p.*, s.CompanyName AS SupplierCompanyName
		FROM Purchase p
		LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID
		WHERE p.Status = '已结账' AND p.SubTotal <= 0`

	params := []interface{}{}
	if bloqueado != "" && bloqueado != "-1" {
		query += ` AND p.Bloqueado = ?`
		params = append(params, bloqueado)
	}
	if orderId != "" {
		query += ` AND p.OrderID = ?`
		params = append(params, orderId)
	}
	// 无法使用供应商原因是多个商品拖货 供应商不一定是一个
	if supplierId != "" {
		query += ` AND p.SupplierID = ?`
		params = append(params, supplierId)
	}
	query += ` ORDER BY p.OrderID DESC`
	err := re.db.Raw(query, params...).Scan(&returnOrders).Error

	if err != nil {
		log.Printf("错误: 查询退货单列表失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询退货单列表失败",
		})
		return
	}

	log.Printf("成功获取退货单列表，共 %d 条记录", len(returnOrders))

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取退货单列表成功",
		"data":    returnOrders,
	})
}

// ReturnGoodsProductList 获取退货单商品列表
// 支持通过订单号、供应商编号和审核状态进行查询
// 路由: POST /returnGoods/ReturnGoodsProductList
func (re *ReturnController) ReturnGoodsProductList(c *gin.Context) {
	log.Println("=== 处理获取退货单商品列表请求 ===")

	// 获取查询参数
	orderId := c.Query("documentoNo")   // 订单编号
	supplierId := c.Query("supplierId") // 供应商编号
	bloqueado := c.Query("bloqueado")   // 审核状态，-1表示全部

	// 构建基础查询语句
	baseQuery := `
		SELECT pd.*, p.PricePurchase, p.UnitPrice as pdunitprice, p.UnitPrice4, s.CompanyName as SupplierCompanyName
		FROM "Purchase Details" pd
		LEFT JOIN Products p ON pd.ProductID = p.ProductID
		LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID
		LEFT JOIN Purchase pu ON pd.OrderID = pu.OrderID
		WHERE 1=1`

	var params []interface{}
	var conditions []string

	// 添加订单号条件
	if orderId != "" {
		conditions = append(conditions, "pd.OrderID = ?")
		params = append(params, orderId)
		log.Printf("添加订单号查询条件: %s", orderId)
	}

	// 添加供应商编号条件
	if supplierId != "" {
		conditions = append(conditions, "p.SupplierID = ?")
		params = append(params, supplierId)
		log.Printf("添加供应商编号查询条件: %s", supplierId)
	}

	// 添加审核状态条件（-1表示全部，不添加条件）
	if bloqueado != "" && bloqueado != "-1" {
		conditions = append(conditions, "pu.Bloqueado = ?")
		params = append(params, bloqueado)
		log.Printf("添加审核状态查询条件: %s", bloqueado)
	}

	// 组合查询条件
	if len(conditions) > 0 {
		baseQuery += " AND " + conditions[0]
		for i := 1; i < len(conditions); i++ {
			baseQuery += " AND " + conditions[i]
		}
	}

	// 添加排序
	baseQuery += " ORDER BY pd.OrderID DESC"

	// 查询退货单商品列表
	var returnGoodsProducts []models.PurchaseDetailsWithProductAndSupplier
	err := re.db.Raw(baseQuery, params...).Scan(&returnGoodsProducts).Error

	if err != nil {
		log.Printf("错误: 查询退货单商品列表失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询退货单商品列表失败",
		})
		return
	}

	// 构建查询条件日志
	queryConditions := []string{}
	if orderId != "" {
		queryConditions = append(queryConditions, "订单号:"+orderId)
	}
	if supplierId != "" {
		queryConditions = append(queryConditions, "供应商编号:"+supplierId)
	}
	if bloqueado != "" && bloqueado != "-1" {
		queryConditions = append(queryConditions, "审核状态:"+bloqueado)
	}
	if len(queryConditions) == 0 {
		queryConditions = append(queryConditions, "全部")
	}

	log.Printf("成功获取退货单商品列表，查询条件: %s, 共 %d 条记录",
		strings.Join(queryConditions, ", "), len(returnGoodsProducts))

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "获取退货单商品列表成功",
		"data":    returnGoodsProducts,
	})
}

// AddReturnedArticulo 添加商品到指定退货单
func (re *ReturnController) AddReturnedArticulo(c *gin.Context) {
	log.Println("=== 处理添加退货商品请求 ===")

	// 获取退货单号参数
	documentoNo := c.Query("documentoNo")
	if documentoNo == "" {
		log.Printf("错误: 退货单号参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "退货单号参数不能为空",
		})
		return
	}

	// 解析请求体
	var request models.AddReturnedArticuloRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("错误: 请求参数解析失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误: " + err.Error(),
		})
		return
	}

	// 验证退货单是否存在
	var purchase models.Purchase
	if err := re.db.Raw("SELECT TOP 1 * FROM Purchase where OrderID = ? AND Status = '已结账' AND SubTotal <= 0", documentoNo).Scan(&purchase).Error; err != nil {
		log.Printf("错误: 退货单不存在, documentoNo=%s, 错误: %v", documentoNo, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}
	if purchase.OrderId == 0 {
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}
	// 开启数据库事务
	tx := re.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	var totalSubTotal float64 = 0

	// 处理每个商品
	for _, articulo := range request.ArticuloList {
		var product models.Products
		var err error
		// 当 value 为 0 号商品时，按 nombre 作为 ProductName 查询
		if articulo.Value == "0" {
			product.ProductName = articulo.Nombre
			product.ProductId = 0
			product.BarCode = "0"
			product.UserCode = "0"
		} else if articulo.Type == 0 {
			// 通过商品ID/编码查询（原逻辑）
			// 通过条码查询
			err = tx.Raw("SELECT TOP 1 * FROM Products WHERE UserCode = ?", articulo.Value).
				Scan(&product).Error
		} else if articulo.Type == 1 {
			// 通过条码查询
			err = tx.Raw("SELECT TOP 1 * FROM Products WHERE BarCode = ?", articulo.Value).
				Scan(&product).Error
		} else {
			tx.Rollback()
			log.Printf("错误: 无效的商品类型, type=%d", articulo.Type)
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "无效的商品类型",
			})
			return
		}
		if product.ProductId == 0 && articulo.Value != "0" {
			tx.Rollback()
			log.Printf("商品不存在, code=%v", articulo.Value)
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "无效的商品",
			})
			return
		}
		var existingDetail models.PurchaseDetails
		if articulo.Value == "0" {
			err = tx.Raw("select top 1 * from [Purchase Details] where OrderID = ? AND ProductName = ?", documentoNo, articulo.Nombre).Scan(&existingDetail).Error
			if err != nil {
				tx.Rollback()
				c.JSON(http.StatusBadRequest, gin.H{
					"code":    400,
					"message": "查询退货详情表失败",
				})
				return
			}
		} else {
			// 检查商品是否已存在于退货单中
			err = tx.Raw("select top 1 * from [Purchase Details] where OrderID = ? AND ProductID = ?", documentoNo, product.ProductId).Scan(&existingDetail).Error
			if err != nil {
				tx.Rollback()
				c.JSON(http.StatusBadRequest, gin.H{
					"code":    400,
					"message": "查询退货详情表失败",
				})
				return
			}
		}
		if existingDetail.ProductId != 0 || (existingDetail.ProductName != nil && articulo.Value == "0") {
			// 商品已存在，合并数量
			newQuantity := existingDetail.Quantity + articulo.Amount
			newSubTotal := newQuantity * articulo.Precio

			// 更新商品明细
			if err := tx.Model(&existingDetail).Updates(map[string]interface{}{
				"Quantity":  newQuantity,
				"UnitPrice": articulo.Precio,
				//"Entrada":   newQuantity,
			}).Error; err != nil {
				tx.Rollback()
				log.Printf("错误: 更新商品明细失败: %v", err)
				c.JSON(http.StatusInternalServerError, gin.H{
					"code":    500,
					"message": "更新商品明细失败",
				})
				return
			}

			totalSubTotal += newSubTotal
			log.Printf("商品已存在，合并数量: ProductID=%s, 新数量=%.3f", articulo.Value, newQuantity)
		} else {
			// 商品不存在，创建新的商品明细
			newDetail := models.PurchaseDetails{
				OrderId:            &purchase.OrderId,
				ProductId:          int(product.ProductId),
				UserCode:           &product.UserCode,
				BarCode:            &product.BarCode,
				ProductName:        &product.ProductName,
				Quantity:           articulo.Amount,
				Discount:           nil,
				Tax:                &product.Tax,
				UnitPrice:          articulo.Precio,
				PriceSale:          &product.UnitPrice,
				QuantityPerUnit:    &product.QuantityPerUnit,
				Bultos:             nil,
				QuantityPerUnit2nd: &product.QuantityPerUnit2nd,
				Bultos2nd:          nil,
				Fjson:              "",
				PriceFactory:       product.PriceFactory,
				Height:             product.Height,
				Length:             product.Length,
				Weight:             product.Weight,
				Width:              product.Width,
				Option1:            &product.Option1,
				Action:             nil,
				BatchCode:          nil,
				//Entrada:            &articulo.Amount,
			}

			if err := tx.Create(&newDetail).Error; err != nil {
				tx.Rollback()
				log.Printf("错误: 创建商品明细失败: %v", err)
				c.JSON(http.StatusInternalServerError, gin.H{
					"code":    500,
					"message": "创建商品明细失败",
				})
				return
			}

			totalSubTotal += articulo.Amount * articulo.Precio
			log.Printf("新增商品明细: ProductID=%s, 数量=%.3f", articulo.Value, articulo.Amount)
		}
	}

	// 更新退货单总金额
	if totalSubTotal > 0 {
		totalSubTotal = -totalSubTotal
	}
	totalSubTotal += purchase.SubTotal
	if err := tx.Model(&purchase).Updates(map[string]interface{}{
		"SubTotal": totalSubTotal,
		//"Amount":         totalSubTotal,
		//"AccountPayable": totalSubTotal,
		"Cash": totalSubTotal,
		"Paid": totalSubTotal,
	}).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 更新退货单总金额失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新退货单总金额失败",
		})
		return
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("错误: 提交事务失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败",
		})
		return
	}

	log.Printf("成功添加商品到退货单: documentoNo=%s, 总金额=%.4f", documentoNo, totalSubTotal)

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "",
		"data":    "成功添加商品到退货单",
	})
}

// 更新退货单审核状态 Bloqueado
func (re *ReturnController) Bloqueado(c *gin.Context) {
	log.Println("=== 处理更新退货单审核状态请求 ===")

	// 从表单中获取参数
	documentoNo := c.PostForm("documentoNo")
	bloqueado := c.PostForm("bloqueado")

	// 验证参数
	if documentoNo == "" {
		log.Printf("错误: 退货单号参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "退货单号参数不能为空",
		})
		return
	}

	if bloqueado == "" {
		bloqueado = "0"
	}

	// 验证退货单是否存在
	var purchase models.Purchase
	if err := re.db.Raw("select top 1 * from Purchase where OrderID = ? AND Status = '已结账' AND SubTotal <= 0", documentoNo).Scan(&purchase).Error; err != nil {
		log.Printf("错误: 退货单不存在, documentoNo=%s, 错误: %v", documentoNo, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}

	// 处理审核状态
	var bloqueadoValue int
	if bloqueado == "1" {
		bloqueadoValue = 1 // 已审核
	} else {
		bloqueadoValue = 0 // 未审核
	}

	// 更新退货单审核状态
	if err := re.db.Model(&purchase).Where("OrderID = ?", documentoNo).Update("Bloqueado", bloqueadoValue).Error; err != nil {
		log.Printf("错误: 更新退货单审核状态失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新退货单审核状态失败",
		})
		return
	}

	//statusText := "未审核"
	//if bloqueadoValue == 1 {
	//	statusText = "已审核"
	//}

	log.Printf("成功更新退货单审核状态: documentoNo=%s, bloqueado=%d", documentoNo, bloqueadoValue)

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "成功更新退货单审核状态",
		"data":    "",
	})
}

// DeleteReturnedArticulo 删除退货单中的商品
func (re *ReturnController) DeleteReturnedArticulo(c *gin.Context) {
	log.Println("=== 处理删除退货单商品请求 ===")

	// 从表单中获取参数
	documentoNo := c.PostForm("documentoNo")
	code := c.PostForm("code")   //商品ID
	name := c.PostForm("nombre") // 商品名称

	// 验证参数
	if documentoNo == "" {
		log.Printf("错误: 退货单号参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "退货单号参数不能为空",
		})
		return
	}

	// 验证商品ID或商品名称至少提供一个
	if code == "" && name == "" {
		log.Printf("错误: 商品ID或商品名称至少需要提供一个")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "商品ID或商品名称至少需要提供一个",
		})
		return
	}

	// 验证退货单是否存在
	var purchase models.Purchase
	if err := re.db.Raw("select * from Purchase where OrderID = ? AND Status = '已结账' AND SubTotal <= 0", documentoNo).Scan(&purchase).Error; err != nil {
		log.Printf("错误: 退货单不存在, documentoNo=%s, 错误: %v", documentoNo, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}
	if purchase.OrderId == 0 {
		//log.Printf("错误: 退货单不存在, documentoNo=%s, 错误: %v", documentoNo, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}
	// 开启数据库事务
	tx := re.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 构建删除条件
	var deleteCondition string
	var deleteParams []interface{}

	if code != "" {
		// code 为 "0" 时按商品名删除
		if code == "0" {
			if name == "" {
				tx.Rollback()
				c.JSON(http.StatusBadRequest, gin.H{
					"code":    400,
					"message": "当code为0时需要提供商品名称name",
				})
				return
			}
			deleteCondition = "OrderID = ? AND ProductName = ?"
			deleteParams = []interface{}{documentoNo, name}
			log.Printf("根据商品名称删除商品(code=0): documentoNo=%s, ProductName=%s", documentoNo, name)
		} else {
			// 只提供商品ID
			product, err := re.findProductByValue(tx, code)
			if err != nil {
				tx.Rollback()
				log.Printf("错误: 商品不存在, Value=%s, 错误: %v", code, err)
				c.JSON(http.StatusNotFound, gin.H{
					"code":    404,
					"message": "商品不存在: " + code,
				})
				return
			}
			deleteCondition = "OrderID = ? AND ProductID = ?"
			deleteParams = []interface{}{documentoNo, product.ProductId}
			log.Printf("根据商品ID删除商品: documentoNo=%s, ProductID=%d", documentoNo, product.ProductId)
		}
	} else {
		// 只提供商品名称
		deleteCondition = "OrderID = ? AND ProductName = ?"
		deleteParams = []interface{}{documentoNo, name}
		log.Printf("根据商品名称删除商品: documentoNo=%s, ProductName=%s", documentoNo, name)
	}

	// 查询要删除的商品明细
	var purchaseDetails []models.PurchaseDetails
	if err := tx.Where(deleteCondition, deleteParams...).Find(&purchaseDetails).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 查询要删除的商品明细失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询要删除的商品明细失败",
		})
		return
	}

	if len(purchaseDetails) == 0 {
		tx.Rollback()
		log.Printf("错误: 未找到要删除的商品")
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "未找到要删除的商品",
		})
		return
	}

	// 计算要删除的商品总金额
	var deletedAmount float64
	for _, detail := range purchaseDetails {
		deletedAmount += detail.UnitPrice * detail.Quantity
	}

	// 删除指定的商品明细
	if err := tx.Where(deleteCondition, deleteParams...).Delete(&models.PurchaseDetails{}).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 删除退货单商品明细失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "删除退货单商品明细失败",
		})
		return
	}

	// 查询剩余的商品明细
	var remainingDetails []models.PurchaseDetails
	if err := tx.Where("OrderID = ?", documentoNo).Find(&remainingDetails).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 查询剩余商品明细失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询剩余商品明细失败",
		})
		return
	}

	// 计算剩余商品的总金额
	var remainingAmount float64
	for _, detail := range remainingDetails {
		remainingAmount += detail.UnitPrice * detail.Quantity
	}
	if remainingAmount > 0 {
		remainingAmount = -remainingAmount
	}
	// 更新退货单总金额
	if err := tx.Model(&purchase).Updates(map[string]interface{}{
		"SubTotal": remainingAmount,
		"Cash":     remainingAmount,
	}).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 更新退货单总金额失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新退货单总金额失败",
		})
		return
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("错误: 提交事务失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败",
		})
		return
	}

	log.Printf("成功删除退货单商品: documentoNo=%s, 删除商品数量=%d, 删除金额=%.2f, 剩余金额=%.2f",
		documentoNo, len(purchaseDetails), deletedAmount, remainingAmount)

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "成功删除退货单商品",
		"data":    "",
	})
}

// UpdateReturnedArticulo 修改退货单中的商品信息
func (re *ReturnController) UpdateReturnedArticulo(c *gin.Context) {
	log.Println("=== 处理修改退货单商品信息请求 ===")

	// 从表单中获取参数
	amount := c.PostForm("amount")
	precio := c.PostForm("precio")
	code := c.PostForm("code")
	documentoNo := c.PostForm("documentoNo")
	name := c.PostForm("nombre") // 商品名称

	// 验证参数
	if documentoNo == "" {
		log.Printf("错误: 退货单号参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "退货单号参数不能为空",
		})
		return
	}

	if code == "" && name == "" {
		log.Printf("错误: 商品ID和商品名称参数不能同时为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "商品ID参数不能为空",
		})
		return
	}

	if amount == "" {
		log.Printf("错误: 商品数量参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "商品数量参数不能为空",
		})
		return
	}

	if precio == "" {
		log.Printf("错误: 进价参数不能为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "进价参数不能为空",
		})
		return
	}

	// 验证退货单是否存在
	var purchase models.Purchase
	if err := re.db.Raw("select * from Purchase where OrderID = ? AND Status = '已结账' AND SubTotal <= 0", documentoNo).Scan(&purchase).Error; err != nil {
		log.Printf("错误: 退货单不存在, documentoNo=%s, 错误: %v", documentoNo, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单不存在",
		})
		return
	}
	var product models.Products
	var err error
	if code != "0" {
		// 使用统一的商品查询方法，按照ProductID、BarCode、UserCode的顺序查询
		product, err = re.findProductByValue(re.db, code)
		if err != nil {
			log.Printf("错误: 商品不存在, code=%s, 错误: %v", code, err)
			c.JSON(http.StatusNotFound, gin.H{
				"code":    404,
				"message": "商品不存在: " + code,
			})
			return
		}
	}

	productId := product.ProductId

	// 开启数据库事务
	tx := re.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 查询退货单中的商品明细
	var purchaseDetail models.PurchaseDetails
	if code == "0" && name != "" {
		if err := tx.Raw("select * from [Purchase Details] where OrderID = ? AND ProductName = ?", documentoNo, name).Scan(&purchaseDetail).Error; err != nil {
			tx.Rollback()
			log.Printf("错误: 退货单中不存在该商品, OrderID=%s, ProductID=%s, 错误: %v", documentoNo, code, err)
			c.JSON(http.StatusNotFound, gin.H{
				"code":    404,
				"message": "退货单中不存在该商品",
			})
			return
		}
	} else {
		if err := tx.Raw("select * from [Purchase Details] where OrderID = ? AND ProductID = ?", documentoNo, productId).Scan(&purchaseDetail).Error; err != nil {
			tx.Rollback()
			log.Printf("错误: 退货单中不存在该商品, OrderID=%s, ProductID=%s, 错误: %v", documentoNo, code, err)
			c.JSON(http.StatusNotFound, gin.H{
				"code":    404,
				"message": "退货单中不存在该商品",
			})
			return
		}
	}

	if purchaseDetail.ProductId == 0 && code != "0" {
		tx.Rollback()
		log.Printf("错误: 退货单中不存在该商品, OrderID=%s, ProductID=%s, 错误: %v", documentoNo, code, err)
		c.JSON(http.StatusNotFound, gin.H{
			"code":    404,
			"message": "退货单中不存在该商品",
		})
		return
	}
	// 解析数量和价值
	var newAmount, newPrecio float64
	//var err error
	amount = strings.ReplaceAll(amount, "-", "")
	if newAmount, err = strconv.ParseFloat(amount, 64); err != nil {
		tx.Rollback()
		log.Printf("错误: 数量参数格式错误: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "数量参数格式错误",
		})
		return
	}

	if newPrecio, err = strconv.ParseFloat(precio, 64); err != nil {
		tx.Rollback()
		log.Printf("错误: 进价参数格式错误: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "进价参数格式错误",
		})
		return
	}

	// 计算新的小计
	//newSubTotal := newAmount * newPrecio
	//// 更新商品明细
	if code != "0" {
		if err := tx.Model(&purchaseDetail).Where("ProductID=?", productId).Updates(map[string]interface{}{
			"Quantity":  newAmount,
			"UnitPrice": newPrecio,
			//"Entrada":   newAmount,
		}).Error; err != nil {
			tx.Rollback()
			log.Printf("错误: 更新商品明细失败: %v", err)
			c.JSON(http.StatusInternalServerError, gin.H{
				"code":    500,
				"message": "更新商品明细失败",
			})
			return
		}
	} else {
		if err := tx.Model(&purchaseDetail).Where("ProductName=?", name).Updates(map[string]interface{}{
			"Quantity":  newAmount,
			"UnitPrice": newPrecio,
			//"Entrada":   newAmount,
		}).Error; err != nil {
			tx.Rollback()
			log.Printf("错误: 更新商品明细失败: %v", err)
			c.JSON(http.StatusInternalServerError, gin.H{
				"code":    500,
				"message": "更新商品明细失败",
			})
			return
		}
	}

	// 重新计算退货单总金额
	var allDetails []models.PurchaseDetails
	if err := tx.Where("OrderID = ?", documentoNo).Find(&allDetails).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 查询退货单所有商品明细失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "查询退货单所有商品明细失败",
		})
		return
	}

	var totalSubTotal float64 = 0
	for _, detail := range allDetails {
		totalSubTotal += detail.Quantity * detail.UnitPrice
	}
	if totalSubTotal > 0 {
		totalSubTotal = -totalSubTotal
	}

	// 更新退货单总金额
	if err := tx.Model(&purchase).Where("OrderID = ?", documentoNo).Updates(map[string]interface{}{
		"SubTotal":       totalSubTotal,
		"Amount":         totalSubTotal,
		"AccountPayable": totalSubTotal,
		"Cash":           totalSubTotal,
		"Paid":           totalSubTotal,
	}).Error; err != nil {
		tx.Rollback()
		log.Printf("错误: 更新退货单总金额失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "更新退货单总金额失败",
		})
		return
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("错误: 提交事务失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败",
		})
		return
	}

	log.Printf("成功修改退货单商品信息: documentoNo=%s, ProductID=%s, 新数量=%.3f, 新进价=%.4f",
		documentoNo, code, newAmount, newPrecio)

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "成功修改退货单商品信息",
		"data":    "",
	})
}

func (re *ReturnController) GetTopPurchases() ([]models.PurchaseWithSupplier, error) {
	var orderIDs []int
	if err := re.db.Table("Purchase AS p").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Select("TOP 30 p.OrderID").
		Order("p.AlbaranID DESC").
		Pluck("p.OrderID", &orderIDs); err.Error != nil {
		return nil, err.Error
	}

	if len(orderIDs) == 0 {
		return []models.PurchaseWithSupplier{}, nil
	}

	var results []models.PurchaseWithSupplier
	if err := re.db.Table("Purchase AS p").
		Joins("LEFT JOIN Suppliers s ON p.SupplierID = s.SupplierID").
		Where("p.OrderID IN ?", orderIDs).
		Select("p.*, s.CompanyName AS supplier_company_name").
		Order("p.AlbaranID DESC").
		Scan(&results); err.Error != nil {
		return nil, err.Error
	}

	return results, nil
}

func (re *ReturnController) GetNewLatestAlbaranID() (*models.Purchase, error) {
	var results models.Purchase
	if err := re.db.Raw("select Top 1 * from Purchase Order by AlbaranID DESC").Scan(&results); err.Error != nil {
		return nil, err.Error
	}
	return &results, nil
}

func (re *ReturnController) GetNewLatestOrderId() (*models.Purchase, error) {
	var results models.Purchase
	if err := re.db.Raw("select Top 1 * from Purchase Order by OrderID DESC").Scan(&results); err.Error != nil {
		return nil, err.Error
	}
	return &results, nil
}
