// Package controllers 包含所有的HTTP控制器
// 负责处理HTTP请求和响应，实现RESTful API接口
package controllers

import (
	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
)

// RegisterRoutes 注册所有API路由
// 将所有控制器的路由注册到Gin路由器中
// 参数: r - Gin路由器实例, db - 数据库连接实例
func RegisterRoutes(r *gin.Engine, db *gorm.DB) {
	// 创建各个控制器的实例
	purchaseController := NewPurchaseController(db)
	purchaseDetailsController := NewPurchaseDetailsController(db)
	productController := NewProductController(db)
	inventoryController := NewInventoryController(db)
	scanController := NewScanController(db)
	returnController := NewReturnController(db)

	// 退货相关路由
	returnGroup := r.Group("/returnGoods")
	{
		returnGroup.POST("/submitReturnGoods", returnController.SubmitReturnGoods)           // 新增退货单
		returnGroup.POST("/listReturnGoods", returnController.ReturnGoodsList)               // 新增退货单
		returnGroup.GET("/listReturnedArticulos", returnController.ReturnGoodsProductList)   // 退货订单内商品数据
		returnGroup.POST("/addReturnedArticulo", returnController.AddReturnedArticulo)       // 添加商品到指定退货单
		returnGroup.POST("/bloqueado", returnController.Bloqueado)                           // 更新退货单审核状态
		returnGroup.POST("/deleteReturnedArticulo", returnController.DeleteReturnedArticulo) // 删除退货单中的商品
		returnGroup.POST("/updateReturnedArticulo", returnController.UpdateReturnedArticulo) // 修改退货单中的商品信息
	}
	// 采购相关路由组 (/purchase)
	// 处理采购单管理的API接口
	purchaseGroup := r.Group("/purchase")
	{
		// 基础CRUD操作
		purchaseGroup.GET("/:orderId", purchaseController.GetByOrderId)             // 根据订单ID获取采购单详情
		purchaseGroup.GET("/all", purchaseController.GetAllPurchases)               // 获取所有采购单列表
		purchaseGroup.POST("/add", purchaseController.AddPurchase)                  // 添加新采购单
		purchaseGroup.PUT("/update", purchaseController.UpdatePurchase)             // 更新采购单信息
		purchaseGroup.DELETE("/delete/:orderId", purchaseController.DeletePurchase) // 删除采购单

		// 特殊字段更新操作
		purchaseGroup.POST("/updateRemark", purchaseController.UpdateRemark)     // 更新备注字段（审核状态）
		purchaseGroup.POST("/updateSubTotal", purchaseController.UpdateSubTotal) // 更新小计字段
	}

	// 采购明细相关路由组 (/purchaseDetails)
	// 处理采购单明细管理的API接口
	purchaseDetailsGroup := r.Group("/purchaseDetails")
	{
		// 基础CRUD操作
		purchaseDetailsGroup.GET("/all", purchaseDetailsController.GetAllPurchaseDetails)           // 获取所有采购明细
		purchaseDetailsGroup.GET("/byOrderId", purchaseDetailsController.GetByOrderId)              // 根据订单ID获取采购明细
		purchaseDetailsGroup.POST("/add", purchaseDetailsController.AddPurchaseDetails)             // 添加采购明细
		purchaseDetailsGroup.PUT("/update", purchaseDetailsController.UpdatePurchaseDetails)        // 更新采购明细
		purchaseDetailsGroup.DELETE("/delete/:id", purchaseDetailsController.DeletePurchaseDetails) // 删除采购明细

		// 查询操作
		purchaseDetailsGroup.POST("/getByBarCode", purchaseDetailsController.GetByBarCode)   // 根据条形码查询（包含产品信息）
		purchaseDetailsGroup.POST("/getByUserCode", purchaseDetailsController.GetByUserCode) // 根据用户编码查询

		// 更新操作
		purchaseDetailsGroup.POST("/updateProductUnitPrice2", purchaseDetailsController.UpdateProductUnitPrice2)         // 更新产品单价2
		purchaseDetailsGroup.POST("/updateProductUnitPrice", purchaseDetailsController.UpdateProductUnitPrice)           // 更新产品单价
		purchaseDetailsGroup.POST("/updateUnitPrice", purchaseDetailsController.UpdateUnitPrice)                         // 更新单价
		purchaseDetailsGroup.POST("/updateEntrada", purchaseDetailsController.UpdateEntrada)                             // 更新入库数量
		purchaseDetailsGroup.POST("/updateQuantity", purchaseDetailsController.UpdateQuantity)                           // 更新数量
		purchaseDetailsGroup.POST("/updateEntradaById", purchaseDetailsController.UpdateEntradaById)                     // 根据ID更新入库数量
		purchaseDetailsGroup.POST("/updateUnitPriceShenHe", purchaseDetailsController.UpdateUnitPriceShenHe)             // 更新单价审核
		purchaseDetailsGroup.POST("/updateUnitPriceShenHeBatch", purchaseDetailsController.UpdateUnitPriceShenHeBatch)   // 更新单价审核
		purchaseDetailsGroup.POST("/batchUpdateStockAndSubTotal", purchaseDetailsController.BatchUpdateStockAndSubTotal) // 更新单价审核
	}

	// 产品相关路由组 (/articulo)
	// 处理商品管理相关的所有API接口
	productGroup := r.Group("/articulo")
	{
		productGroup.POST("/add", productController.Add)                               // 添加新商品
		productGroup.POST("/scan", productController.Scan)                             // 扫描商品（POST方式）
		productGroup.GET("/scan6", productController.Scan6)                            // 扫描商品（GET方式）
		productGroup.POST("/update", productController.Update)                         // 更新商品信息
		productGroup.POST("/updateUnitsInStock", productController.UpdateUnitsInStock) // 更新商品库存

		// 分类和供应商接口
		productGroup.POST("/getClass", productController.GetClass)    // 获取商品类别列表
		productGroup.POST("/supplier", productController.GetSupplier) // 获取供应商列表
	}

	// 库存相关路由组 (/inventory)
	// 处理库存管理和盘点相关的API接口
	inventoryGroup := r.Group("/inventory")
	{
		inventoryGroup.GET("/getPurchaseListByNodeCash", inventoryController.GetPurchaseListByNodeCash)            // 获取盘点单列表
		inventoryGroup.POST("/updateProductAndInsertPurchase", inventoryController.UpdateProductAndInsertPurchase) // 新增盘点单并更新库存
	}

	// 扫描相关路由组 (/scan)
	// 处理Excel数据解析和导入相关的API接口
	scanGroup := r.Group("/scan")
	{
		scanGroup.GET("/getExcelData", scanController.GetExcelData)        // 获取Excel数据（解析不同格式的文件）
		scanGroup.POST("/insertExcelData", scanController.InsertExcelData) // 插入Excel数据到数据库
	}
}
