package controllers

import (
	"fmt"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/errors"
	"github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/common/profile"
	tmt "github.com/tencentcloud/tencentcloud-sdk-go/tencentcloud/tmt/v20180321"
	"log"
	"net/http"
	"net/url"
	"shopbootx-go/internal"
	"strconv"
	"strings"
	"sync"
	"time"

	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"shopbootx-go/models"
)

// ScanController 扫描控制器
// 负责处理所有与扫描相关的HTTP请求，包括Excel数据解析、数据导入等功能
type ScanController struct {
	db *gorm.DB // 数据库连接实例
}

// NewScanController 创建新的扫描控制器实例
// 参数: db - 数据库连接实例
// 返回: ScanController实例
func NewScanController(db *gorm.DB) *ScanController {
	log.Println("创建扫描控制器实例")
	return &ScanController{db: db}
}

// GetExcelData 获取Excel数据接口
// 根据不同的URL解析Excel、PDF或CSV文件，返回结构化数据
// 路由: GET /scan/getExcelData
// 查询参数: url - Excel文件URL, scanUrl - 扫描URL（可选）
func (sc *ScanController) GetExcelData(c *gin.Context) {
	log.Println("=== 处理获取Excel数据请求 ===")

	// 获取URL参数
	urlParam := c.Query("url")
	scanUrl := c.Query("scanUrl")
	isSaveImage := c.Query("isSaveImage")

	log.Printf("接收Excel数据请求参数: url=%s, scanUrl=%s", urlParam, scanUrl)

	// 验证URL参数是否为空
	if urlParam == "" {
		log.Println("错误: URL参数为空")
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "URL参数不能为空",
		})
		return
	}

	// 解码URL参数
	log.Println("正在解码URL参数...")
	decodedURL, err := url.QueryUnescape(urlParam)
	if err != nil {
		log.Printf("错误: URL解码失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "URL解码失败",
		})
		return
	}
	log.Printf("URL解码成功: %s", decodedURL)

	// 解码扫描URL参数（如果提供）
	var scanURLDecoded string
	if scanUrl != "" {
		log.Println("正在解码扫描URL参数...")
		scanURLDecoded, err = url.QueryUnescape(scanUrl)
		if err != nil {
			log.Printf("错误: 扫描URL解码失败: %v", err)
			c.JSON(http.StatusBadRequest, gin.H{
				"code":    400,
				"message": "扫描URL解码失败",
			})
			return
		}
		log.Printf("扫描URL解码成功: %s", scanURLDecoded)
	}

	var excelData []models.ExcelData

	// 根据不同的URL进行不同的解析处理
	log.Println("开始解析Excel数据...")
	if strings.Contains(decodedURL, "https://densen.es") {
		log.Println("检测到densen.es厂家，使用Excel解析方式")
		excelData, err = internal.ParseExcel(decodedURL, isSaveImage)
		if err != nil {
			c.JSON(http.StatusOK, gin.H{
				"code":    500,
				"message": "解析异常",
			})
		}
	} else if strings.Contains(decodedURL, "https://s7star.online") {
		log.Println("检测到s7star.online厂家，使用PDF解析方式")
		excelData, err = internal.ParsePdfByImg(decodedURL)
		if err != nil {
			c.JSON(http.StatusOK, gin.H{
				"code":    500,
				"message": "解析异常",
			})
		}
	} else if strings.Contains(decodedURL, "casacado.com") {
		log.Println("检测到casacado.com厂家，使用CSV解析方式")
		excelData, err = internal.ParseCsvByCasacado(decodedURL, scanURLDecoded, isSaveImage)
		if err != nil {
			c.JSON(http.StatusOK, gin.H{
				"code":    500,
				"message": "解析异常",
			})
		}
	}

	log.Printf("Excel数据解析完成，共解析 %d 条记录", len(excelData))
	// 返回解析结果
	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "解析成功",
		"data":    excelData,
	})
}

// InsertExcelData 插入Excel数据
func (sc *ScanController) InsertExcelData(c *gin.Context) {
	var excelDataList []models.ExcelDataRequest
	if err := c.ShouldBindJSON(&excelDataList); err != nil {
		log.Printf("错误: Excel数据请求参数绑定失败: %v", err)
		c.JSON(http.StatusBadRequest, gin.H{
			"code":    400,
			"message": "请求参数错误",
		})
		return
	}

	log.Printf("接收Excel数据插入请求，数据条数: %d", len(excelDataList))

	// 获取查询参数 供应商Id
	supplierIdStr := c.Query("supplierId")
	// 商品品类
	categoryIDStr := c.Query("categoryID")
	//商品检测式 1只编码 2只条码 3编码条码
	proTypeStr := c.Query("proType")
	//是否更新销售价
	isUpdateStr := c.Query("isUpdate")
	shipAddress := c.Query("shipAddress")
	//是否只更新销售价（对应app的更新和入库两个操作）
	isOnlyUpdatePriceStr := c.Query("isOnlyUpdatePrice")

	log.Printf("查询参数: supplierId=%s, isUpdate=%s, shipAddress=%s, isOnlyUpdatePrice=%s, categoryID=%s, proType=%s",
		supplierIdStr, isUpdateStr, shipAddress, isOnlyUpdatePriceStr, categoryIDStr, proTypeStr)
	// 解析参数
	supplierId, _ := strconv.Atoi(supplierIdStr)
	isUpdate := isUpdateStr == "true"
	isOnlyUpdatePrice := isOnlyUpdatePriceStr == "true"
	categoryID, _ := strconv.Atoi(categoryIDStr)
	proType, _ := strconv.Atoi(proTypeStr)

	log.Printf("解析后的参数: supplierId=%d, isUpdate=%t, isOnlyUpdatePrice=%t, categoryID=%d, proType=%d",
		supplierId, isUpdate, isOnlyUpdatePrice, categoryID, proType)

	// 开始事务
	log.Println("开始数据库事务...")
	tx := sc.db.Begin()

	// 只更新销售价模式
	if isOnlyUpdatePrice {
		log.Println("进入只更新销售价模式")
		for i, data := range excelDataList {
			log.Printf("处理第 %d 条数据: UserCode=%s, BarCode=%s", i+1, data.UserCode, data.Barcode)
			var products []models.Products
			query := tx.Model(&models.Products{})

			// 根据proType确定查询条件
			if proType == 1 { // 只编码
				query = query.Where("UserCode = ?", data.UserCode)
				log.Printf("使用UserCode查询: %s", data.UserCode)
			} else if proType == 2 { // 只条码
				query = query.Where("BarCode = ?", data.Barcode)
				log.Printf("使用BarCode查询: %s", data.Barcode)
			} else if proType == 3 { // 编码条码
				query = query.Where("UserCode = ? AND BarCode = ?", data.UserCode, data.Barcode)
				log.Printf("使用UserCode和BarCode查询: %s, %s", data.UserCode, data.Barcode)
			} else {
				query = query.Where("BarCode = ?", data.Barcode)
				log.Printf("使用默认BarCode查询: %s", data.Barcode)
			}

			if err := query.Find(&products).Error; err != nil {
				log.Printf("query Find err: %s", err.Error())
			}

			if len(products) > 0 {
				product := products[0]
				log.Printf("找到商品: ID=%d, UserCode=%s", product.ProductId, product.UserCode)

				// 更新销售价
				if data.PricePurchase != "" {
					num, err := strconv.ParseFloat(data.PricePurchase, 64)
					if err != nil {
						log.Printf("更新销售价 ParseFloat err: %s", err.Error())
					}
					log.Printf("条码:%v 更新销售价 原销售价为: %f,新销售价为:%f", product.BarCode, product.PricePurchase, num)
					product.UnitPrice = num
				}

				// 更新供应商ID
				if supplierId > 0 {
					log.Printf("更新供应商ID: %d", supplierId)
					product.SupplierId = supplierId
				}

				// 更新类别ID
				if categoryID > 0 {
					log.Printf("更新类别ID: %d", categoryID)
					product.CategoryId = categoryID
				}

				// 更新税率
				if data.Tax != "" {
					if tax, err := strconv.ParseFloat(data.Tax, 32); err == nil {
						log.Printf("更新税率: %f", tax)
						product.Tax = float32(tax)
					} else {
						log.Printf("税率解析失败，使用默认税率: 0.21")
						product.Tax = 0.21 // 默认税率
					}
				} else {
					log.Printf("使用默认税率: 0.21")
					product.Tax = 0.21 // 默认税率
				}

				if err := tx.Save(&product).Error; err != nil {
					log.Printf("错误: 保存商品失败: %v", err)
					tx.Rollback()
					c.JSON(http.StatusInternalServerError, gin.H{
						"code":    500,
						"message": "保存商品失败",
					})
					return
				} else {
					log.Printf("商品更新成功: ID=%d", product.ProductId)
				}
			} else {
				log.Printf("警告: 未找到商品，UserCode=%s, BarCode=%s", data.UserCode, data.Barcode)
			}
		}

		log.Println("提交事务...")
		if err := tx.Commit().Error; err != nil {
			log.Printf("错误: 提交事务失败: %v", err)
			c.JSON(http.StatusInternalServerError, gin.H{
				"code":    500,
				"message": "提交事务失败",
			})
			return
		}
		log.Println("只更新销售价模式处理完成")
		c.JSON(http.StatusOK, gin.H{
			"code":    200,
			"message": "解析成功",
		})
		return
	}
	//TODO:先新增总进货单  然后分别处理商品和进货单明细
	log.Println("进入完整数据插入模式")

	// 获取最新albaranid
	albaranId := 1
	purchases, err := sc.GetPurchasesAll(nil, nil)
	if err != nil {
		log.Printf("错误: 获取采购单失败: %v", err)
		tx.Rollback()
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "获取采购单失败",
		})
		return
	}
	if len(purchases) > 0 {
		albaranId = *purchases[0].AlbaranId + 1
		log.Printf("获取最新albaranId: %d", albaranId)
	} else {
		log.Println("使用默认albaranId: 1")
	}

	// 计算总金额
	var totalCash float64
	for _, data := range excelDataList {
		if subTotal, _ := strconv.ParseFloat(data.SubTotal, 64); err == nil {
			totalCash += subTotal
		}
	}
	log.Printf("计算总金额: %f", totalCash)

	// 创建采购单
	now := time.Now()
	purchase := models.Purchase{
		SupplierId:   &supplierId,
		AlbaranId:    &albaranId,
		EmployeeId:   &[]int{1}[0],
		ExchangeRate: 8.0,
		FacturaId:    &[]int{0}[0],
		OrderDate:    &now,
		OrderType:    &[]int{0}[0],
		Paid:         &totalCash,
		Status:       "已结账",
		PayMode:      "Dinero",
		Remark:       "",
		TaxInclude:   &[]bool{true}[0],
		Final:        "1",
		ShipAddress:  shipAddress,
		Cash:         &totalCash,
		SubTotal:     totalCash,
	}

	if !isUpdate {
		if err := tx.Create(&purchase).Error; err != nil {
			log.Printf("错误: 创建采购单失败: %v", err)
			tx.Rollback()
			c.JSON(http.StatusInternalServerError, gin.H{
				"code":    500,
				"message": "创建采购单失败",
			})
			return
		}
	}

	// 处理商品和采购单明细
	if len(excelDataList) > 0 {
		// 多线程处理
		var wg sync.WaitGroup
		errorChan := make(chan error, len(excelDataList))

		for _, data := range excelDataList {
			wg.Add(1)
			go func(data models.ExcelDataRequest) {
				defer wg.Done()

				if proType == 3 { // 编码条码
					// 判断条码、编码、条码编码三种情况，有一种以上返回false
					products, err := sc.parseBefore(data, 1, supplierId, isUpdate, categoryID, &purchase)
					if err != nil {
						errorChan <- err
						return
					}
					userCodeSize := len(products)

					products1, err := sc.parseBefore(data, 2, supplierId, isUpdate, categoryID, &purchase)
					if err != nil {
						errorChan <- err
						return
					}
					barCodeSize := len(products1)

					if userCodeSize > 0 && barCodeSize > 0 && (products[0].ProductId != products1[0].ProductId) {
						// 都有重复，返回错误
						errorChan <- fmt.Errorf("重复率过高")
						return
					} else if userCodeSize > 0 { // 编码有重复的
						if err := sc.parse(data, 3, supplierId, isUpdate, categoryID, &purchase); err != nil {
							errorChan <- err
							return
						}
					} else if barCodeSize > 0 { // 条码有重复的
						if err := sc.parse(data, 4, supplierId, isUpdate, categoryID, &purchase); err != nil {
							errorChan <- err
							return
						}
					} else { // 不存在重复的，走一条去新增就好了
						if err := sc.parse(data, 3, supplierId, isUpdate, categoryID, &purchase); err != nil {
							errorChan <- err
							return
						}
					}
				} else if proType == 1 {
					if err := sc.parse(data, 1, supplierId, isUpdate, categoryID, &purchase); err != nil {
						errorChan <- err
						return
					}
				} else if proType == 2 {
					if err := sc.parse(data, 2, supplierId, isUpdate, categoryID, &purchase); err != nil {
						errorChan <- err
						return
					}
				}
			}(data)
		}

		// 等待所有goroutine完成
		wg.Wait()
		close(errorChan)

		// 检查是否有错误
		for err := range errorChan {
			if err != nil {
				if err.Error() == "重复率过高" {
					tx.Rollback()
					c.JSON(http.StatusInternalServerError, gin.H{
						"code":    601,
						"message": "重复率过高，已终止，建议电脑输入",
					})
					return
				} else {
					tx.Rollback()
					c.JSON(http.StatusInternalServerError, gin.H{
						"code":    500,
						"message": "解析失败",
					})
					return
				}
			}
		}
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("错误: 提交事务失败: %v", err)
		c.JSON(http.StatusInternalServerError, gin.H{
			"code":    500,
			"message": "提交事务失败",
		})
		return
	}

	c.JSON(http.StatusOK, gin.H{
		"code":    200,
		"message": "解析成功",
	})
}

func (sc *ScanController) GetPurchasesAll(albaranId *int, supplierCompanyName *string) ([]models.Purchase, error) {
	var orderIDs []int

	q := sc.db.Table("purchase AS p").
		Select("TOP 30 p.OrderID").
		Joins("LEFT JOIN suppliers s ON p.SupplierID = s.SupplierID")

	if albaranId != nil {
		q = q.Where("p.AlbaranID = ?", *albaranId)
	}

	if supplierCompanyName != nil && *supplierCompanyName != "" {
		q = q.Where("s.companyname LIKE ?", "%"+*supplierCompanyName+"%")
	}

	if err := q.Order("p.AlbaranID DESC").Pluck("p.OrderID", &orderIDs).Error; err != nil {
		return nil, err
	}

	var purchases []models.Purchase
	if len(orderIDs) == 0 {
		return purchases, nil
	}

	err := sc.db.Table("purchase AS p").
		Select("p.*, s.companyname AS supplierCompanyName").
		Joins("LEFT JOIN suppliers s ON p.SupplierID = s.SupplierID").
		Where("p.OrderID IN ?", orderIDs).
		Order("p.AlbaranID DESC").
		Find(&purchases).Error

	return purchases, err
}

// parseBefore 检查重复商品
// 参数: data - Excel数据, type - 检查类型(1:只编码, 2:只条码), supplierId - 供应商ID, isUpdate - 是否更新, categoryID - 类别ID, purchase - 采购单
// 返回: 重复的商品列表
func (sc *ScanController) parseBefore(data models.ExcelDataRequest, typeCheck int, supplierId int, isUpdate bool, categoryID int, purchase *models.Purchase) ([]models.Products, error) {
	var products []models.Products
	query := sc.db.Model(&models.Products{})

	if typeCheck == 1 {
		query = query.Where("UserCode = ?", data.UserCode)
	} else if typeCheck == 2 {
		query = query.Where("BarCode = ?", data.Barcode)
	}

	if err := query.Find(&products).Error; err != nil {
		return nil, fmt.Errorf("查询重复商品失败: %v", err)
	}

	return products, nil
}

// parse 主要的解析方法，处理商品的新增和更新逻辑
// 参数: data - Excel数据, type - 处理类型, supplierId - 供应商ID, isUpdate - 是否更新, categoryID - 类别ID, purchase - 采购单
func (sc *ScanController) parse(data models.ExcelDataRequest, typeProcess int, supplierId int, isUpdate bool, categoryID int, purchase *models.Purchase) error {
	var oldPro *models.Products // 处理后的商品
	var unitPrice *float64      // 销售价

	if data.PricePurchase != "" {
		num, err := strconv.ParseFloat(data.PricePurchase, 64)
		if err != nil {
			log.Printf("parse ParseFloat err:%v", err)
		}
		unitPrice = &num
	}

	query := sc.db.Model(&models.Products{})
	products := models.Products{}

	// 根据类型设置查询条件和商品信息
	if typeProcess == 1 { // 只编码
		query = query.Where("UserCode = ?", data.UserCode)
		products.BarCode = data.UserCode
		products.UserCode = data.UserCode
	} else if typeProcess == 2 { // 只条码
		query = query.Where("BarCode = ?", data.Barcode)
		products.BarCode = data.Barcode
		products.UserCode = data.Barcode
	} else if typeProcess == 3 { // 编码条码
		query = query.Where("UserCode = ?", data.UserCode)
		products.BarCode = data.Barcode
		products.UserCode = data.UserCode
	} else if typeProcess == 4 { // 条码编码
		query = query.Where("BarCode = ?", data.Barcode)
		products.BarCode = data.Barcode
		products.UserCode = data.UserCode
	}

	var productsList []models.Products
	if err := query.Find(&productsList).Error; err != nil {
		return fmt.Errorf("查询商品失败: %v", err)
	}

	if len(productsList) > 0 {
		// 修改库存逻辑
		oldPro = sc.updatePro(data, productsList, supplierId, categoryID, unitPrice, isUpdate, products)
	} else {
		// 新增逻辑
		// 查重
		queryOld := sc.db.Model(&models.Products{})

		isNeedInsert := true

		if typeProcess == 1 { // 只编码，查重条码
			queryOld = queryOld.Where("BarCode = ?", data.Barcode)
			var queryOldList []models.Products
			if err := queryOld.Find(&queryOldList).Error; err != nil {
				return fmt.Errorf("查询条码重复失败: %v", err)
			}

			// 检查是否存在条码入库那种数据
			sec := sc.db.Model(&models.Products{}).Where("BarCode = ? AND UserCode = ?", data.Barcode, data.Barcode)
			var secProList []models.Products
			if err := sec.Find(&secProList).Error; err != nil {
				return fmt.Errorf("查询条码编码重复失败: %v", err)
			}

			if len(queryOldList) > 0 && len(secProList) == 0 {
				// 有重复的，走修改逻辑
				oldPro = sc.updatePro(data, queryOldList, supplierId, categoryID, unitPrice, isUpdate, products)
				isNeedInsert = false
			}
		} else if typeProcess == 2 { // 只条码，查重编码
			queryOld = queryOld.Where("UserCode = ?", data.UserCode)
			var queryOldList []models.Products
			if err := queryOld.Find(&queryOldList).Error; err != nil {
				return fmt.Errorf("查询编码重复失败: %v", err)
			}

			// 检查是否存在条码编码都相同的数据
			sec := sc.db.Model(&models.Products{}).Where("BarCode = ? AND UserCode = ?", data.UserCode, data.UserCode)
			var secProList []models.Products
			if err := sec.Find(&secProList).Error; err != nil {
				return fmt.Errorf("查询编码条码重复失败: %v", err)
			}

			if len(queryOldList) > 0 && len(secProList) == 0 {
				// 有重复的，走修改逻辑
				oldPro = sc.updatePro(data, queryOldList, supplierId, categoryID, unitPrice, isUpdate, products)
				isNeedInsert = false
			}
		}

		// 需要新增
		if isNeedInsert {
			products.SupplierId = supplierId
			products.CategoryId = categoryID
			products.ProductName = data.ProductName
			products.ProductName2 = data.ProductName2
			products.ProductName3 = time.Now().Format("2006-01-02")

			if unitPrice1, err := strconv.ParseFloat(data.UnitPrice, 64); err == nil {
				products.PricePurchase = unitPrice1
			}

			if quantity, err := strconv.ParseFloat(data.Quantity, 64); err == nil {
				products.UnitsInStock = quantity
			}

			products.QuantityPerUnit = 0

			if unitPrice != nil {
				products.UnitPrice = *unitPrice // 销售价
			} else {
				products.UnitPrice = 0 // 销售价
			}

			// 税率为空，默认0.21
			if data.Tax != "" {
				if tax, err := strconv.ParseFloat(data.Tax, 32); err == nil {
					products.Tax = float32(tax)
				} else {
					products.Tax = 0.21
				}
			} else {
				products.Tax = 0.21
			}

			//INSERT INTO "Products" ("BarCode","UserCode","ProductName","ProductName2","ProductName3","ProductName4","ProductName5","PricePurchase","UnitPrice","UnitPrice2","UnitPrice3","UnitPrice4","UnitPrice5","UnitPrice6","UnitPrice7","UnitPrice8","UnitPriceAVG","UnitPriceLast","Tax","SupplierID","CategoryID","QuantityPerUnit","QuantityPerUnit2nd","UnitsInStock","UnitsOnOrder","UnitsOnPurchase","MarketingNO","Discontinued","Discount","DtoStartDate","DtoExipDate","Remark","ReorderLevel","StorageLocation","timestamp","Picture","fjson","fjson2","hide","Leng2ProductName","Leng3ProductName","Leng4ProductName","Leng5ProductName","Leng2Remark","Leng3Remark","Leng4Remark","Leng5Remark","BigProductID","colorsize","PriceFactory","PriceTransport","QuantityTotal","height","width","length","weight","MaxBultos2nd","Option1","UnboxLevel","Image","Views","UUID","SellerID","Color","limit","limitQty","Parent","Step","Option2","Option3","Option4","Option5","Option6","Option7","Option8","OptionS1","OptionS2","UnitPriceTaxExcluded","InventoryLimit","ForeColor") OUTPUT INSERTED."ProductID" VALUES ('W30621-1','W30621-1','波浪纹床盖 米色 230X260CM','','2025-08-06','','',8.99,8.99,NULL,NULL,NULL,NULL,NULL,NULL,NULL,0,NULL,0.21,1,1,1,1,2,0,0,0,NULL,0,NULL,NULL,'',NULL,'','','','','',NULL,'','','','','','','','','','',NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,0,NULL,'',NULL,'',NULL,'',NULL,NULL,'',NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,'','',NULL,0,'');
			insertSQL := `DECLARE @OutputTable TABLE (ProductID INT);
				INSERT INTO "Products" (
				"BarCode", "UserCode", "ProductName", "ProductName2","ProductName3",
				"PricePurchase", "UnitPrice", "Tax", "SupplierID", "CategoryID",
				 "UnitsInStock", "QuantityPerUnit"
			)
			OUTPUT INSERTED."ProductID" INTO @OutputTable
			VALUES (
				?, ?, ?, ?, ?, ?,?, ?,
				?, ?, ?, ?
			);
		
			SELECT ProductID FROM @OutputTable;`

			// 准备所有参数（与SQL语句中的?一一对应）
			params := []interface{}{
				products.BarCode, products.UserCode, products.ProductName, products.ProductName2, products.ProductName3, products.PricePurchase,
				products.UnitPrice, products.Tax, products.SupplierId, products.CategoryId, products.UnitsInStock, products.QuantityPerUnit,
			}

			// 执行SQL并获取插入的ProductID
			var productId int64
			err := sc.db.Raw(insertSQL, params...).Scan(&productId).Error
			if err != nil {
				log.Fatalf("插入数据失败: %v", err)
			}
			//if err := sc.db.Create(&products).Error; err != nil {
			//	return fmt.Errorf("创建商品失败: %v", err)
			//}
			products.ProductId = productId
			// 获取最后插入的ID
			//var lastInsertId int64
			//if err := sc.db.Raw("SELECT LAST_INSERT_ID()").Scan(&lastInsertId).Error; err != nil {
			//	log.Printf("获取最后插入ID失败: %v", err)
			//} else {
			//	products.ProductId = lastInsertId
			//}

			log.Printf("条码%s编码%s商品不存在，新增商品", data.Barcode, data.UserCode)
			oldPro = &products
		}
	}

	// 添加进货单明细
	if !isUpdate {
		purchaseDetails := models.PurchaseDetails{
			OrderId:     &purchase.OrderId,
			ProductId:   int(oldPro.ProductId),
			UserCode:    &oldPro.UserCode,
			BarCode:     &oldPro.BarCode,
			ProductName: &oldPro.ProductName,
		}

		if quantity, err := strconv.ParseFloat(data.Quantity, 64); err == nil {
			purchaseDetails.Quantity = quantity
		}

		if discount, err := strconv.Atoi(data.Discount); err == nil {
			purchaseDetails.Discount = &discount
		}

		purchaseDetails.Tax = &oldPro.Tax

		if unitPrice, err := strconv.ParseFloat(data.UnitPrice, 64); err == nil {
			purchaseDetails.UnitPrice = unitPrice
		}

		if unitPrice != nil {
			purchaseDetails.PriceSale = unitPrice
		}

		// 设置默认值
		quantityPerUnit := 1
		bultos := 1
		quantityPerUnit2nd := 1
		bultos2nd := 0
		option1 := 0
		entrada := 0.0

		purchaseDetails.QuantityPerUnit = &quantityPerUnit
		purchaseDetails.Bultos = &bultos
		purchaseDetails.QuantityPerUnit2nd = &quantityPerUnit2nd
		purchaseDetails.Bultos2nd = &bultos2nd
		purchaseDetails.Option1 = &option1
		purchaseDetails.Entrada = &entrada
		purchaseDetails.Fjson = "{\"StockID\":1,\"StockName\":\"\",\"OrgQtq\":\"\",\"SQL_QtyOrg\":null,\"check\":\"\",\"strPanDian\":null,\"StorageLocation\":null,\"Moreinfo1\":null,\"Moreinfo2\":null}"

		if err := sc.db.Create(&purchaseDetails).Error; err != nil {
			return fmt.Errorf("创建采购单明细失败: %v", err)
		}

		log.Printf("新增进货单明细：条码%s", oldPro.UserCode)
	}

	return nil
}

// updatePro 修改商品
// 参数: data - Excel数据, productsList - 现有商品列表, supplierId - 供应商ID, categoryID - 类别ID, unitPrice - 销售价, isUpdate - 是否更新, products - 商品信息
// 返回: 更新后的商品
func (sc *ScanController) updatePro(data models.ExcelDataRequest, productsList []models.Products, supplierId int, categoryID int, unitPrice *float64, isUpdate bool, products models.Products) *models.Products {
	productsOld := productsList[0]

	// 更新库存
	if quantity, err := strconv.ParseFloat(data.Quantity, 64); err == nil {
		newUnitsInStock := productsOld.UnitsInStock + quantity
		log.Printf("条码%s商品已存在，原有库存:%.2f 新库存：%.2f", productsOld.BarCode, productsOld.UnitsInStock, newUnitsInStock)
		productsOld.UnitsInStock = newUnitsInStock
	}

	productsOld.SupplierId = supplierId
	productsOld.CategoryId = categoryID
	productsOld.BarCode = products.BarCode
	productsOld.UserCode = products.UserCode
	productsOld.ProductName = data.ProductName
	productsOld.ProductName2 = data.ProductName2

	if unitPrice1, err := strconv.ParseFloat(data.UnitPrice, 64); err == nil {
		productsOld.PricePurchase = unitPrice1
	}

	// 税率为空，默认0.21
	if data.Tax != "" {
		if tax, err := strconv.ParseFloat(data.Tax, 32); err == nil {
			productsOld.Tax = float32(tax)
		} else {
			productsOld.Tax = 0.21
		}
	} else {
		productsOld.Tax = 0.21
	}

	if isUpdate && unitPrice != nil {
		log.Printf("条码%s 更新销售价 原销售价为：%.2f 新销售价为:%.2f", productsOld.BarCode, productsOld.PricePurchase, unitPrice)
		productsOld.UnitPrice = *unitPrice
	}

	if err := sc.db.Save(&productsOld).Error; err != nil {
		log.Printf("更新商品失败: %v", err)
	} else {
		log.Printf("条码%s 更新信息", productsOld.BarCode)
	}

	return &productsOld
}

func TranslateBatch(contextString []string) []string {
	// 实例化一个认证对象，入参需要传入腾讯云账户 SecretId 和 SecretKey，此处还需注意密钥对的保密
	// 代码泄露可能会导致 SecretId 和 SecretKey 泄露，并威胁账号下所有资源的安全性
	// 以下代码示例仅供参考，建议采用更安全的方式来使用密钥
	// 请参见：https://cloud.tencent.com/document/product/1278/85305
	// 密钥可前往官网控制台 https://console.cloud.tencent.com/cam/capi 进行获取
	var result []string
	credential := common.NewCredential(
		"AKIDRmy0pS1fLyT2TiiUBg0AAD5lGYsoZAcv",
		"OeiaIq6NWWNMH1hGuSbnoeQAva7Rcnrm",
	)
	// 使用临时密钥示例
	// credential := common.NewTokenCredential("SecretId", "SecretKey", "Token")
	// 实例化一个client选项，可选的，没有特殊需求可以跳过
	cpf := profile.NewClientProfile()
	cpf.HttpProfile.Endpoint = "tmt.tencentcloudapi.com"
	// 实例化要请求产品的client对象,clientProfile是可选的
	client, _ := tmt.NewClient(credential, "ap-beijing", cpf)

	// 实例化一个请求对象,每个接口都会对应一个request对象
	request := tmt.NewTextTranslateBatchRequest()
	request.Source = common.StringPtr("zh")
	request.Target = common.StringPtr("es")
	request.ProjectId = common.Int64Ptr(0)
	request.SourceTextList = common.StringPtrs(contextString)
	// 返回的resp是一个TextTranslateBatchResponse的实例，与请求对象对应
	response, err := client.TextTranslateBatch(request)
	if _, ok := err.(*errors.TencentCloudSDKError); ok {
		fmt.Printf("An API error has returned: %s", err)
		return nil
	}
	if err != nil {
		fmt.Printf("An API error has returned: %s", err.Error())
		return nil
	}
	//fmt.Printf("TargetTextList: %q\n", response.Response.TargetTextList)
	for _, text := range response.Response.TargetTextList {
		//fmt.Printf("[%d] %s\n", i, *text)
		result = append(result, *text)
	}
	return result
}
