package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
)

// SubmitReturnGoodsRequest 提交退货单请求结构体
type SubmitReturnGoodsRequest struct {
	ArticuloList []ArticuloItem `json:"articuloList"` // 商品列表
	EmpresaId    string         `json:"empresaId"`    // 企业ID
}

// ArticuloItem 商品项结构体
type ArticuloItem struct {
	Amount float64 `json:"amount"` // 退货数量（负数）
	Nombre string  `json:"nombre"` // 商品名称
	Precio float64 `json:"precio"` // 价格
	Type   int     `json:"type"`   // 类型：0-商品ID, 1-条码
	Value  string  `json:"value"`  // 商品ID或条码
}

func main() {
	// 示例请求数据
	request := SubmitReturnGoodsRequest{
		ArticuloList: []ArticuloItem{
			{
				Amount: -1,                  // 退货数量（负数）
				Nombre: "Sopa de costillas", // 商品名称
				Precio: 33.3,                // 价格
				Type:   0,                   // 0: 商品ID
				Value:  "1",                 // 商品ID
			},
			{
				Amount: -10,        // 退货数量（负数）
				Nombre: "Articulo", // 商品名称
				Precio: 50.0,       // 价格
				Type:   0,          // 0: 商品ID
				Value:  "0",        // 商品ID
			},
		},
		EmpresaId: "2", // 企业ID
	}

	// 将请求数据转换为JSON
	jsonData, err := json.Marshal(request)
	if err != nil {
		fmt.Printf("JSON序列化失败: %v\n", err)
		return
	}

	fmt.Printf("请求数据: %s\n", string(jsonData))

	// 发送HTTP请求
	url := "http://localhost:8080/returnGoods/submitReturnGoods"
	resp, err := http.Post(url, "application/json", bytes.NewBuffer(jsonData))
	if err != nil {
		fmt.Printf("HTTP请求失败: %v\n", err)
		return
	}
	defer resp.Body.Close()

	// 读取响应
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		fmt.Printf("读取响应失败: %v\n", err)
		return
	}

	fmt.Printf("响应状态码: %d\n", resp.StatusCode)
	fmt.Printf("响应内容: %s\n", string(body))
}

// 使用条码的示例
func exampleWithBarcode() {
	request := SubmitReturnGoodsRequest{
		ArticuloList: []ArticuloItem{
			{
				Amount: -5,          // 退货数量（负数）
				Nombre: "商品名称",      // 商品名称
				Precio: 25.5,        // 价格
				Type:   1,           // 1: 条码
				Value:  "123456789", // 条码
			},
		},
		EmpresaId: "1", // 企业ID
	}

	jsonData, _ := json.Marshal(request)
	fmt.Printf("条码查询示例: %s\n", string(jsonData))
}
