package internal

import (
	"bytes"
	"encoding/csv"
	"fmt"
	"github.com/PuerkitoBio/goquery"
	"github.com/unidoc/unipdf/v3/extractor"
	"github.com/unidoc/unipdf/v3/model"
	"github.com/xuri/excelize/v2"
	"golang.org/x/text/encoding/simplifiedchinese"
	"golang.org/x/text/transform"
	"image"
	"image/jpeg"
	"io"
	"log"
	"math"
	"net/http"
	"net/url"
	"os"
	"path/filepath"
	"shopbootx-go/models"
	"strconv"
	"strings"
	"sync"
	"time"
)

// ParseExcel 从 URL 中读取 Excel 并解析
// 对应Java版本中的parseExcel方法
func ParseExcel(rawURL string, isSaveImage string) ([]models.ExcelData, error) {
	// 解析 URL
	u, err := url.Parse(rawURL)
	if err != nil {
		return nil, fmt.Errorf("解析URL失败: %v", err)
	}

	// 获取query参数
	queryParams := u.Query()
	fileName := queryParams.Get("file")
	if fileName == "" {
		return nil, fmt.Errorf("file 参数为空")
	}

	// 拼接文件下载地址 - 与Java版本保持一致
	fileURL := fmt.Sprintf("https://app.obs.eu-west-101.myhuaweicloud.eu/%s.xlsx", fileName)
	log.Printf("开始下载Excel文件: %s", fileURL)

	// 下载 Excel 文件
	resp, err := http.Get(fileURL)
	if err != nil {
		return nil, fmt.Errorf("下载Excel文件失败: %v", err)
	}
	defer resp.Body.Close()

	// 检查HTTP状态码
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("下载Excel文件失败，HTTP状态码: %d", resp.StatusCode)
	}

	// 读取响应体为字节流
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("读取Excel文件内容失败: %v", err)
	}

	// 通过 xuri/excelize 读取 excel
	f, err := excelize.OpenReader(strings.NewReader(string(body)))
	if err != nil {
		return nil, fmt.Errorf("打开Excel文件失败: %v", err)
	}
	defer f.Close()

	// 获取第一个工作表
	sheetName := f.GetSheetName(0)
	rows, err := f.GetRows(sheetName)
	if err != nil {
		return nil, fmt.Errorf("读取Excel工作表失败: %v", err)
	}

	// 检查是否有数据
	if len(rows) == 0 {
		return nil, fmt.Errorf("Excel文件无数据")
	}

	// 解析标题行，创建列名到索引的映射
	var headerMap = make(map[string]int)
	for idx, colName := range rows[0] {
		headerMap[strings.TrimSpace(colName)] = idx
	}

	// 解析数据行 - 对应Java版本中的AnalysisEventListener逻辑
	var result []models.ExcelData
	for _, row := range rows[1:] {
		// 跳过空行
		if len(row) == 0 {
			continue
		}

		data := models.ExcelData{
			UserCode:     getColumnValue(row, headerMap, "Referencia"),
			Barcode:      getColumnValue(row, headerMap, "Código"),
			ProductName:  getColumnValue(row, headerMap, "Articulo"),
			ProductName2: getColumnValue(row, headerMap, "Nombre2"),
			Lote:         getColumnValue(row, headerMap, "Lote"),
			Quantity:     getColumnValue(row, headerMap, "Cantidad"),
			UnitPrice:    getColumnValue(row, headerMap, "Precio"),
			Discount:     getColumnValue(row, headerMap, "Dto%"),
			Tax:          getColumnValue(row, headerMap, "IVA"),
			FinalPrice:   getColumnValue(row, headerMap, "Precio Final"),
			SubTotal:     getColumnValue(row, headerMap, "Subtotal"),
		}

		// 记录解析到的数据 - 对应Java版本中的日志输出
		log.Printf("解析到一条数据: %+v", data)
		result = append(result, data)
	}

	log.Printf("Excel解析完成，共解析 %d 条记录", len(result))
	return result, nil
}

// getColumnValue 安全获取列的值
func getColumnValue(row []string, headerMap map[string]int, col string) string {
	if idx, ok := headerMap[col]; ok {
		if idx < len(row) {
			return strings.TrimSpace(row[idx])
		}
	}
	return ""
}

// ParsePdfByImg 解析PDF文件，对应Java版本的parsePdfByImg方法
func ParsePdfByImg(url string) ([]models.ExcelData, error) {
	log.Printf("开始解析PDF文件: %s", url)

	// 建立读取文件流 - 对应Java版本的URL urls = new URL(url); InputStream inputStream = urls.openStream();
	resp, err := http.Get(url)
	if err != nil {
		return nil, fmt.Errorf("下载PDF文件失败: %v", err)
	}
	defer resp.Body.Close()

	// 读取PDF内容
	buf := new(bytes.Buffer)
	_, err = io.Copy(buf, resp.Body)
	if err != nil {
		return nil, fmt.Errorf("读取PDF内容失败: %v", err)
	}

	// 解析PDF - 对应Java版本的PDDocument document = PDDocument.load(inputStream)
	pdfReader, err := model.NewPdfReader(bytes.NewReader(buf.Bytes()))
	if err != nil {
		return nil, fmt.Errorf("创建PDF读取器失败: %v", err)
	}

	numPages, err := pdfReader.GetNumPages()
	if err != nil {
		return nil, fmt.Errorf("获取PDF页数失败: %v", err)
	}

	log.Printf("PDF共有 %d 页", numPages)

	// 对应Java版本的List<ExcelData> rows = new ArrayList<>();
	var rows []models.ExcelData

	// 对应Java版本的for (int pageIndex = 0; pageIndex < document.getNumberOfPages(); pageIndex++)
	for pageIndex := 1; pageIndex <= numPages; pageIndex++ {
		log.Printf("正在处理第 %d 页", pageIndex)

		// 对应Java版本的PDPage page = document.getPage(pageIndex);
		page, err := pdfReader.GetPage(pageIndex)
		if err != nil {
			return nil, fmt.Errorf("获取第 %d 页失败: %v", pageIndex, err)
		}

		// 解析文本 - 对应Java版本的List<ExcelData> pageRows = PdfParser.parseText(document, page, pageIndex);
		pageRows, err := parseText(page, pageIndex-1)
		if err != nil {
			return nil, fmt.Errorf("解析第 %d 页文本失败: %v", pageIndex, err)
		}
		rows = append(rows, pageRows...)

		// 解析图片并关联 - 对应Java版本的PdfParser.processImages(page, pageRows);
		err = processImages(page, pageRows)
		if err != nil {
			log.Printf("处理第 %d 页图片时出现警告: %v", pageIndex, err)
		}
	}

	log.Printf("PDF解析完成，共解析 %d 条记录", len(rows))
	return rows, nil
}

// parseText 解析PDF文本内容，对应Java版本的parseText方法
func parseText(page *model.PdfPage, pageIndex int) ([]models.ExcelData, error) {
	// 对应Java版本的PDFTextStripper实现
	ext, err := extractor.New(page)
	if err != nil {
		return nil, fmt.Errorf("创建文本提取器失败: %v", err)
	}

	textData, err := ext.ExtractText()
	if err != nil {
		return nil, fmt.Errorf("提取文本失败: %v", err)
	}

	// 对应Java版本的List<ExcelData> rows = new ArrayList<>();
	var rows []models.ExcelData

	// 对应Java版本的private ExcelData currentRow;
	var currentRow *models.ExcelData

	// 对应Java版本的private float lastY = -1; private float lastMinY = -1;
	var lastY float64 = -1
	var lastMinY float64 = -1

	// 对应Java版本的private StringBuilder currentLine = new StringBuilder();
	//var currentLine strings.Builder

	// 按行分割文本，模拟Java版本的writeString方法
	lines := strings.Split(textData, "\n")

	for _, line := range lines {
		line = strings.TrimSpace(line)
		if line == "" {
			continue
		}

		// 对应Java版本的processLine方法
		processLine(line, &rows, &currentRow, pageIndex, lastY, lastMinY)
	}

	// 对应Java版本的flushRemainingText方法
	if currentRow != nil {
		rows = append(rows, *currentRow)
	}

	return rows, nil
}

// processLine 处理单行文本，对应Java版本的processLine方法
func processLine(line string, rows *[]models.ExcelData, currentRow **models.ExcelData, pageIndex int, lastY, lastMinY float64) {
	// 对应Java版本的String[] parts = line.split("\\|");
	parts := strings.Split(line, "|")

	if len(parts) >= 9 {
		// 对应Java版本的if (!(parts[0].equals("Articulo") && parts[1].equals("Caja")))
		if !(parts[0] == "Articulo" && parts[1] == "Caja") {
			// 对应Java版本的if (null != currentRow) { rows.add(currentRow); currentRow = new ExcelData(); }
			if *currentRow != nil {
				*rows = append(*rows, **currentRow)
			}

			// 对应Java版本的String Articulo = parts[0].trim().substring(0, parts[0].length() - 1);
			articulo := strings.TrimSpace(parts[0])
			if len(articulo) > 0 {
				articulo = articulo[:len(articulo)-1] // 去掉最后一个字符
			}

			// 对应Java版本的BigDecimal Precio = BigDecimal.valueOf(Double.parseDouble(parts[3]));
			precio, _ := strconv.ParseFloat(parts[3], 64)

			// 对应Java版本的折扣计算
			do, _ := strconv.ParseFloat(parts[4], 64)
			dps, _ := strconv.ParseFloat(parts[5], 64)
			dpm, _ := strconv.ParseFloat(parts[6], 64)
			precios2, _ := strconv.ParseFloat(parts[7], 64)

			// 对应Java版本的折扣公式: ((100-DO)/100)*((100-DPS)/100)*((100-DPM)/100)
			doRate := (100.0 - do) / 100.0
			dpsRate := (100.0 - dps) / 100.0
			dpmRate := (100.0 - dpm) / 100.0
			discount := doRate * dpsRate * dpmRate
			finalDiscount := 100.0 - (discount * 100.0)

			// 对应Java版本的currentRow = new ExcelData();
			*currentRow = &models.ExcelData{
				UserCode:   articulo,
				ImgPath:    articulo + ".jpg", // 对应Java版本的currentRow.setImgPath(Articulo + ".jpg");
				UnitPrice:  fmt.Sprintf("%.2f", precio),
				FinalPrice: fmt.Sprintf("%.2f", precios2),
				Quantity:   parts[8], // 对应Java版本的String Cant = parts[8];
				SubTotal:   parts[9], // 对应Java版本的String Total = parts[9];
				Discount:   fmt.Sprintf("%.0f", finalDiscount),
				PageIndex:  pageIndex,
				MaxY:       lastY,
				MinY:       lastMinY,
			}
		}
	} else {
		// 对应Java版本的else分支：是无关的或者同一行的数据
		if len(parts) == 1 && parts[0] != "ALBARAN" && !strings.Contains(parts[0], "Nº") {
			if *currentRow != nil {
				value := strings.TrimSpace(parts[0])
				if len(value) > 0 {
					value = value[:len(value)-1] // 去掉最后一个字符
				}

				// 对应Java版本的if (null == currentRow.getBarcode()) {//条码
				if (*currentRow).Barcode == "" {
					// 条码
					(*currentRow).Barcode = value
				} else {
					// 商品名称 - 对应Java版本的currentRow.setProductName((currentRow.getProductName() == null ? "" : currentRow.getProductName() + " ") + parts[0].trim().substring(0, parts[0].length() - 1));
					if (*currentRow).ProductName == "" {
						(*currentRow).ProductName = value
					} else {
						(*currentRow).ProductName += " " + value
					}
				}
			}
		}
	}
}

// processImages 处理PDF中的图片，对应Java版本的processImages方法
func processImages(page *model.PdfPage, rows []models.ExcelData) error {
	// 对应Java版本的PDResources resources = page.getResources();
	resources := page.Resources
	if resources == nil {
		log.Printf("页面资源为空，跳过图片处理")
		return nil
	}

	// 对应Java版本的float pageHeight = page.getMediaBox().getHeight();
	pageHeight := float64(page.MediaBox.Height())
	log.Printf("页面高度: %.2f", pageHeight)

	// 对应Java版本的ImagePositionTracker imagePositionTracker = new ImagePositionTracker();
	imagePositionTracker := NewImagePositionTracker()

	// 对应Java版本的imagePositionTracker.processPage(page);
	err := imagePositionTracker.ProcessPage(page)
	if err != nil {
		log.Printf("处理页面图片位置失败: %v", err)
	}

	// 对应Java版本的int imageCounter = 0;
	//imageCounter := 0

	// 对应Java版本的for (COSName name : resources.getXObjectNames())
	// 由于unipdf库的API限制，这里简化实现
	log.Printf("开始处理页面图片...")

	// 尝试提取图片并关联
	tryExtractImages(page, rows, pageHeight, imagePositionTracker)

	log.Printf("图片处理完成")
	return nil
}

// tryExtractImages 尝试提取PDF中的图片，对应Java版本的图片处理循环
func tryExtractImages(page *model.PdfPage, rows []models.ExcelData, pageHeight float64, tracker *ImagePositionTracker) {
	// 对应Java版本的for (COSName name : resources.getXObjectNames())循环
	// 由于unipdf库的API限制，这里提供基础实现

	log.Printf("尝试提取页面图片...")

	// 获取图片绘制信息
	imageDrawInfos := tracker.GetImageDrawInfos()

	// 对应Java版本的for (ImagePositionTracker.ImageDrawInfo imageDrawInfo : imageDrawInfos)循环
	for _, imageDrawInfo := range imageDrawInfos {
		// 对应Java版本的Matrix ctm = imageDrawInfo.ctm;
		//ctm := imageDrawInfo.CTM

		// 对应Java版本的int height = imageDrawInfo.image.getHeight();
		// 这里简化处理，实际应该获取图片高度

		// 对应Java版本的float imgHeight = pageHeight - imageDrawInfo.y;
		imgHeight := pageHeight - imageDrawInfo.Y

		// 对应Java版本的ExcelData matchedRow = findNearestRow(rows, imgHeight);
		matchedRow := findNearestRow(rows, imgHeight)

		// 对应Java版本的if (matchedRow != null) { saveProductImage(image, matchedRow); }
		if matchedRow != nil {
			err := saveProductImage(imageDrawInfo.Image, matchedRow)
			if err != nil {
				log.Printf("保存商品图片失败: %v", err)
			}
		}
	}

	log.Printf("图片提取完成")
}

// findNearestRow 坐标匹配算法，对应Java版本的findNearestRow方法
func findNearestRow(rows []models.ExcelData, imageY float64) *models.ExcelData {
	// 对应Java版本的ExcelData closest = null; float minDiff = Float.MAX_VALUE;
	var closest *models.ExcelData
	//minDiff := math.MaxFloat64

	// 对应Java版本的List<Map<String, Object>> mapList = new ArrayList<>();
	type rowDiff struct {
		row  *models.ExcelData
		diff float64
	}

	var rowDiffs []rowDiff

	// 对应Java版本的for (ExcelData row : rows)循环
	for i := range rows {
		row := &rows[i]
		// 对应Java版本的float diff = Math.abs((row.getMinY() + row.getMaxY()) / 2 - imageY);
		rowCenterY := (row.MinY + row.MaxY) / 2
		diff := math.Abs(rowCenterY - imageY)

		// 对应Java版本的Map<String, Object> map = new HashMap<>(); map.put("row", row); map.put("diff", diff); mapList.add(map);
		rowDiffs = append(rowDiffs, rowDiff{
			row:  row,
			diff: diff,
		})
	}

	// 对应Java版本的排序逻辑
	// List<Map<String, Object>> collect = mapList.stream().sorted(Comparator.comparingDouble(map -> { ... })).collect(Collectors.toList());
	for i := 0; i < len(rowDiffs)-1; i++ {
		for j := i + 1; j < len(rowDiffs); j++ {
			if rowDiffs[i].diff > rowDiffs[j].diff {
				rowDiffs[i], rowDiffs[j] = rowDiffs[j], rowDiffs[i]
			}
		}
	}

	// 对应Java版本的if (collect.size() > 0) { Map<String, Object> map = collect.get(0); ... }
	if len(rowDiffs) > 0 {
		// 对应Java版本的Float diff = (Float) map.get("diff"); ExcelData row = (ExcelData) map.get("row");
		//diff := rowDiffs[0].diff
		row := rowDiffs[0].row

		// 对应Java版本的minDiff = diff; closest = row;
		//minDiff = diff
		closest = row
	}

	return closest
}

// saveProductImage 保存商品图片，对应Java版本的saveProductImage方法
func saveProductImage(img image.Image, excelData *models.ExcelData) error {
	// 对应Java版本的BufferedImage bufferedImage = image.getImage();
	bufferedImage := img

	// 对应Java版本的File outputFile = new File(staticPath + excelData.getUserCode().trim() + ".jpg");
	// 注意：这里使用相对路径，实际项目中应该配置staticPath
	outputDir := "./images" // 对应Java版本的staticPath
	outputPath := filepath.Join(outputDir, excelData.UserCode+".jpg")

	// 对应Java版本的while (outputFile.exists()) { if (!outputFile.delete()) { throw new RuntimeException("覆盖文件失败: " + outputFile.getAbsolutePath()); } }
	if _, err := os.Stat(outputPath); err == nil {
		if err := os.Remove(outputPath); err != nil {
			return fmt.Errorf("覆盖文件失败: %s", outputPath)
		}
	}

	// 确保输出目录存在
	if err := os.MkdirAll(outputDir, 0755); err != nil {
		return fmt.Errorf("创建输出目录失败: %v", err)
	}

	// 创建输出文件
	outputFile, err := os.Create(outputPath)
	if err != nil {
		return fmt.Errorf("创建输出文件失败: %v", err)
	}
	defer outputFile.Close()

	// 对应Java版本的ImageIO.write(bufferedImage, "jpg", outputFile);
	err = jpeg.Encode(outputFile, bufferedImage, &jpeg.Options{Quality: 90})
	if err != nil {
		return fmt.Errorf("保存图片失败: %v", err)
	}

	log.Printf("成功保存商品图片: %s", outputPath)
	return nil
}

// ImagePositionTracker 图片位置跟踪器，对应Java版本的ImagePositionTracker类
type ImagePositionTracker struct {
	graphicsStateStack   []GraphicsState
	imageDrawInfos       []ImageDrawInfo
	currentGraphicsState GraphicsState
}

// GraphicsState 图形状态，对应Java版本的GraphicsState类
type GraphicsState struct {
	CTM      Matrix
	CurrentX float64
	CurrentY float64
}

// Matrix 变换矩阵，对应Java版本的Matrix类
type Matrix struct {
	A, B, C, D, E, F float64
}

// ImageDrawInfo 图片绘制信息，对应Java版本的ImageDrawInfo类
type ImageDrawInfo struct {
	Image image.Image
	CTM   Matrix
	X, Y  float64
}

// NewImagePositionTracker 创建新的图片位置跟踪器
func NewImagePositionTracker() *ImagePositionTracker {
	return &ImagePositionTracker{
		graphicsStateStack: make([]GraphicsState, 0),
		imageDrawInfos:     make([]ImageDrawInfo, 0),
		currentGraphicsState: GraphicsState{
			CTM: Matrix{A: 1, D: 1}, // 单位矩阵
		},
	}
}

// ProcessPage 处理页面，提取图片位置信息，对应Java版本的processPage方法
func (ipt *ImagePositionTracker) ProcessPage(page *model.PdfPage) error {
	// 对应Java版本的imagePositionTracker.processPage(page);
	// 由于unipdf库的API限制，这里提供基础实现

	log.Printf("图片位置跟踪器处理页面")

	// 在实际项目中，这里应该：
	// 1. 解析PDF内容流
	// 2. 跟踪图形状态
	// 3. 记录图片绘制信息
	// 4. 计算图片位置坐标

	// 目前提供基础框架，可以根据需要扩展
	return nil
}

// GetImageDrawInfos 获取图片绘制信息
func (ipt *ImagePositionTracker) GetImageDrawInfos() []ImageDrawInfo {
	return ipt.imageDrawInfos
}

// ParseCsvByCasacado 解析casacado.com的CSV文件
// 对应Java版本的parseCsvByCasacado方法
// 参数: url - CSV文件URL, scanUrl - 扫描URL
// 返回: ExcelData列表和错误
func ParseCsvByCasacado(url, scanUrl, isSaveImage string) ([]models.ExcelData, error) {
	log.Printf("开始解析CSV文件: %s", url)
	// 下载CSV文件
	resp, err := http.Get(url)
	if err != nil {
		return nil, fmt.Errorf("下载CSV文件失败: %v", err)
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("HTTP请求失败，状态码: %d", resp.StatusCode)
	}

	// 解析CSV
	var excelDataList []models.ExcelData
	var productNames []string
	utf8Reader := transform.NewReader(resp.Body, simplifiedchinese.GBK.NewDecoder())
	// 创建CSV读取器，跳过第一行（标题行）
	reader := csv.NewReader(utf8Reader)
	reader.FieldsPerRecord = -1 // 允许变长记录

	// 跳过标题行
	_, err = reader.Read()
	if err != nil {
		return nil, fmt.Errorf("读取CSV标题行失败: %v", err)
	} // 转换 GBK → UTF-8

	// 读取CSV数据
	for {
		line, err := reader.Read()
		if err != nil {
			if err.Error() == "EOF" {
				break
			}
			log.Printf("警告: 读取CSV行失败: %v", err)
			continue
		}

		// 确保行有足够的列
		if len(line) < 12 {
			log.Printf("警告: CSV行列数不足，跳过: %v", line)
			continue
		}

		usercode := line[6]
		barcode := line[7]
		productName := line[8]
		quantity := line[9]
		unitPrice := line[10]
		discount := line[11]

		excelData := models.ExcelData{
			UserCode: usercode,
			Barcode:  barcode,
		}

		excelData.ProductName = productName
		productNames = append(productNames, productName)
		// 解析数量
		quantityFloat, err := strconv.ParseFloat(quantity, 64)
		if err != nil {
			log.Printf("警告: 解析数量失败: %s, 错误: %v", quantity, err)
			quantityFloat = 0
		}
		excelData.Quantity = strconv.FormatFloat(quantityFloat, 'f', 0, 64)

		// 解析单价
		unitPriceFloat, err := strconv.ParseFloat(unitPrice, 64)
		if err != nil {
			log.Printf("警告: 解析单价失败: %s, 错误: %v", unitPrice, err)
			unitPriceFloat = 0
		}
		excelData.UnitPrice = strconv.FormatFloat(unitPriceFloat, 'f', 2, 64)

		// 解析折扣
		discountFloat, err := strconv.ParseFloat(discount, 64)
		if err != nil {
			log.Printf("警告: 解析折扣失败: %s, 错误: %v", discount, err)
			discountFloat = 0
		}
		excelData.Discount = strconv.FormatFloat(discountFloat, 'f', 0, 64)

		// 计算折后价
		discountRate := (100 - discountFloat) / 100
		finalPrice := unitPriceFloat * discountRate
		excelData.FinalPrice = strconv.FormatFloat(finalPrice, 'f', 2, 64)

		// 计算合计（折后价 * 数量）
		subTotal := finalPrice * quantityFloat
		excelData.SubTotal = strconv.FormatFloat(subTotal, 'f', 2, 64)

		excelDataList = append(excelDataList, excelData)
	}

	log.Printf("CSV解析完成，共解析 %d 条记录", len(excelDataList))

	// TODO: 翻译处理 - 由于需要腾讯云翻译服务，这里暂时跳过
	// 在实际项目中，需要集成腾讯云翻译API
	log.Println("跳过翻译处理（需要集成腾讯云翻译服务）")
	//productNamesEs := TranslateBatch(productNames)

	// 爬取网页图片
	if scanUrl != "" && isSaveImage == "true" {
		log.Printf("开始爬取网页图片: %s", scanUrl)
		parseList, err := scrapeImages(scanUrl)
		if err != nil {
			log.Printf("警告: 爬取图片失败: %v", err)
		} else {
			// 匹配图片并保存
			saveImages(excelDataList, parseList)
		}
	}

	return excelDataList, nil
}

// scrapeImages 爬取网页图片信息
func scrapeImages(scanUrl string) ([]models.ExcelData, error) {
	// 创建HTTP客户端
	uri, err := url.Parse(scanUrl)
	if err != nil {
		log.Println(fmt.Sprintf("scrapeImages Parse scanUrl:%v error,err:%v", scanUrl, err.Error()))
		return nil, err
	}
	resp, err := http.Get(scanUrl)
	if err != nil {
		log.Printf("scrapeImages Get scanUrl:%v error,err:%v", scanUrl, err.Error())
		return nil, err
	}
	defer resp.Body.Close()

	// 解析 HTML
	doc, err := goquery.NewDocumentFromReader(resp.Body)
	if err != nil {
		log.Fatal("解析 HTML 失败:", err)
	}
	var parseList []models.ExcelData
	doc.Find("tbody tr").Each(func(i int, s *goquery.Selection) {
		item := models.ExcelData{}
		s.Find("td").Each(func(i int, s *goquery.Selection) {
			// 获取原始 HTML 内容
			tdHtml, _ := s.Html()
			// 替换 <br> 或 <br/> 为换行
			tdText := strings.ReplaceAll(tdHtml, "<br>", "\n")
			tdText = strings.ReplaceAll(tdText, "<br/>", "\n")
			// 清理多余空白字符（可选）
			tdText = strings.TrimSpace(tdText)
			if i == 1 {
				// 拆分为数组
				lines := strings.Split(tdText, "\n")
				// 去除空白字符
				for index, line := range lines {
					if index == 0 {
						item.UserCode = line
					}
					if index == 1 {
						item.Barcode = line
					}
				}
			}
		})

		// 查找并提取img标签
		img := s.Find("img")
		if img.Length() > 0 {
			src, exists := img.Attr("src")
			if exists {
				u, err := url.Parse(src)
				if err != nil {
					fmt.Println("Error parsing URL:", err)
					return
				}
				// 创建一个新的URL，只包含特定的查询参数
				newURL := &url.URL{
					Scheme:   uri.Scheme,
					Host:     uri.Host,
					Path:     u.Path,
					RawQuery: "upload_id=" + u.Query().Get("upload_id"), // 只保留upload_id参数
				}
				// 输出新的URL
				item.ImgPath = newURL.String()
			}
		}
		parseList = append(parseList, item)
	})
	log.Printf("爬取到 %d 条图片信息", len(parseList))
	return parseList, nil
}

// saveImages 保存图片
// 对应Java版本的图片保存部分
func saveImages(excelDataList []models.ExcelData, parseList []models.ExcelData) {
	log.Println("开始保存图片...")

	// 创建输出目录
	outputDir := "./images"
	if err := os.MkdirAll(outputDir, 0755); err != nil {
		log.Printf("警告: 创建图片目录失败: %v", err)
		return
	}

	// 使用goroutine并发保存图片
	var wg sync.WaitGroup
	semaphore := make(chan struct{}, 100) // 限制并发数

	for i := range excelDataList {
		// 查找匹配的图片信息
		var matchedData *models.ExcelData
		for _, parseData := range parseList {
			if parseData.Barcode == excelDataList[i].Barcode && parseData.UserCode == excelDataList[i].UserCode {
				matchedData = &parseData
				break
			}
		}

		if matchedData != nil && matchedData.ImgPath != "" {
			wg.Add(1)
			go func(data models.ExcelData, imgPath string) {
				defer wg.Done()
				semaphore <- struct{}{}        // 获取信号量
				defer func() { <-semaphore }() // 释放信号量

				// 下载并保存图片
				if err := downloadAndSaveImage(imgPath, data.Barcode, outputDir); err != nil {
					log.Printf("保存图片失败: %s, 错误: %v", data.Barcode, err)
				} else {
					log.Printf("成功保存图片: %s", data.Barcode)
				}
			}(excelDataList[i], matchedData.ImgPath)

			// 设置图片路径
			excelDataList[i].ImgPath = strings.TrimSpace(excelDataList[i].Barcode) + ".jpg"
		}
	}

	wg.Wait()
	log.Println("图片保存完成")
}

// downloadAndSaveImage 下载并保存图片
func downloadAndSaveImage(imgUrl, barcode, outputDir string) error {
	// 创建HTTP客户端
	client := &http.Client{
		Timeout: 30 * time.Second,
	}

	// 下载图片
	resp, err := client.Get(imgUrl)
	if err != nil {
		return fmt.Errorf("下载图片失败: %v", err)
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("下载图片失败，状态码: %d", resp.StatusCode)
	}

	// 解码图片
	img, _, err := image.Decode(resp.Body)
	if err != nil {
		return fmt.Errorf("解码图片失败: %v", err)
	}

	// 构建输出文件路径
	outputPath := filepath.Join(outputDir, strings.TrimSpace(barcode)+".jpg")

	// 如果文件已存在，先删除
	if _, err := os.Stat(outputPath); err == nil {
		if err := os.Remove(outputPath); err != nil {
			return fmt.Errorf("删除已存在的文件失败: %v", err)
		}
	}

	// 创建输出文件
	outputFile, err := os.Create(outputPath)
	if err != nil {
		return fmt.Errorf("创建输出文件失败: %v", err)
	}
	defer outputFile.Close()

	// 保存图片
	err = jpeg.Encode(outputFile, img, &jpeg.Options{Quality: 90})
	if err != nil {
		return fmt.Errorf("保存图片失败: %v", err)
	}

	return nil
}
