package internal

import (
	"log"
	"shopbootx-go/models"
	"testing"
)

// TestParsePdfByImg 测试PDF解析功能
func TestParsePdfByImg(t *testing.T) {
	// 测试URL - 这里使用一个示例URL，实际使用时需要替换为真实的PDF URL
	testURL := "https://s7star.online/example.pdf"

	log.Printf("开始测试PDF解析: %s", testURL)

	// 调用解析方法
	result, err := ParsePdfByImg(testURL)
	if err != nil {
		log.Printf("PDF解析失败: %v", err)
		t.Fatalf("PDF解析失败: %v", err)
	}

	// 验证结果
	log.Printf("PDF解析成功，共解析 %d 条记录", len(result))

	// 打印前几条记录用于验证
	for i, data := range result {
		if i >= 3 { // 只打印前3条记录
			break
		}
		log.Printf("记录 %d: UserCode=%s, Barcode=%s, ProductName=%s, Quantity=%s, UnitPrice=%s, SubTotal=%s",
			i+1, data.UserCode, data.Barcode, data.ProductName, data.Quantity, data.UnitPrice, data.SubTotal)
	}

	// 验证数据完整性
	for i, data := range result {
		if data.UserCode == "" {
			t.Errorf("记录 %d: UserCode为空", i+1)
		}
		if data.UnitPrice == "" {
			t.Errorf("记录 %d: UnitPrice为空", i+1)
		}
		if data.Quantity == "" {
			t.Errorf("记录 %d: Quantity为空", i+1)
		}
	}
}

// TestParseText 测试文本解析功能
func TestParseText(t *testing.T) {
	// 模拟PDF文本内容
	textContent := `Articulo|Caja|Paq|Precio|DO|DPS|DPM|Precios2|Cant|Total
123456|1|1|10.50|5.00|3.00|2.00|9.50|5|47.50
789012|2|1|15.75|10.00|5.00|3.00|13.50|3|40.50
345678|1|1|8.25|0.00|0.00|0.00|8.25|10|82.50`

	log.Printf("开始测试文本解析")

	// 这里应该调用实际的parseText方法
	// 由于parseText是私有方法，这里只是示例
	log.Printf("文本内容: %s", textContent)

	// 验证解析逻辑
	lines := []string{
		"123456|1|1|10.50|5.00|3.00|2.00|9.50|5|47.50",
		"789012|2|1|15.75|10.00|5.00|3.00|13.50|3|40.50",
		"345678|1|1|8.25|0.00|0.00|0.00|8.25|10|82.50",
	}

	for i, line := range lines {
		parts := []string{line}
		if len(parts) >= 9 {
			log.Printf("解析行 %d: %s", i+1, parts[0])
		}
	}
}

// TestFindNearestRow 测试坐标匹配算法
func TestFindNearestRow(t *testing.T) {
	// 创建测试数据
	rows := []models.ExcelData{
		{UserCode: "123456", MinY: 100, MaxY: 120},
		{UserCode: "789012", MinY: 200, MaxY: 220},
		{UserCode: "345678", MinY: 300, MaxY: 320},
	}

	// 测试图片Y坐标
	imageY := 210.0

	// 调用坐标匹配算法
	closest := findNearestRow(rows, imageY)

	if closest == nil {
		t.Fatal("未找到匹配的行")
	}

	log.Printf("找到最近的行: UserCode=%s, 中心Y坐标=%.2f",
		closest.UserCode, (closest.MinY+closest.MaxY)/2)

	// 验证结果
	expectedUserCode := "789012"
	if closest.UserCode != expectedUserCode {
		t.Errorf("期望UserCode=%s, 实际UserCode=%s", expectedUserCode, closest.UserCode)
	}
}

// TestSaveProductImage 测试图片保存功能
func TestSaveProductImage(t *testing.T) {
	// 创建测试数据
	excelData := &models.ExcelData{
		UserCode:    "123456",
		Barcode:     "123456789",
		ProductName: "测试商品",
	}

	// 创建一个简单的测试图片
	// 这里简化处理，实际应该从PDF中提取真实图片
	log.Printf("测试保存商品图片: UserCode=%s", excelData.UserCode)

	// 验证输出目录是否存在
	outputDir := "./images"
	log.Printf("检查输出目录: %s", outputDir)

	// 这里只是测试，不实际保存图片
	log.Printf("图片保存测试完成")
}

// TestImagePositionTracker 测试图片位置跟踪器
func TestImagePositionTracker(t *testing.T) {
	// 创建图片位置跟踪器
	tracker := NewImagePositionTracker()
	if tracker == nil {
		t.Fatal("创建图片位置跟踪器失败")
	}

	log.Printf("图片位置跟踪器创建成功")

	// 测试获取图片绘制信息
	imageDrawInfos := tracker.GetImageDrawInfos()
	if imageDrawInfos == nil {
		t.Fatal("获取图片绘制信息失败")
	}

	log.Printf("图片位置跟踪器测试完成")
}

// TestCompilation 测试代码编译
func TestCompilation(t *testing.T) {
	// 这个测试确保所有必要的函数都存在并且可以调用
	log.Printf("开始编译测试")

	// 测试ParsePdfByImg函数存在
	_ = ParsePdfByImg

	// 测试findNearestRow函数存在
	_ = findNearestRow

	// 测试saveProductImage函数存在
	_ = saveProductImage

	// 测试NewImagePositionTracker函数存在
	_ = NewImagePositionTracker

	log.Printf("编译测试通过")
}

// BenchmarkParsePdfByImg 性能测试
func BenchmarkParsePdfByImg(b *testing.B) {
	testURL := "https://s7star.online/example.pdf"

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		_, err := ParsePdfByImg(testURL)
		if err != nil {
			b.Fatalf("PDF解析失败: %v", err)
		}
	}
}

// TestParseTdSegments 测试parseTdSegments函数
func TestParseTdSegments(t *testing.T) {
	// 这个测试需要goquery的Selection对象，在实际项目中应该使用真实的HTML
	// 这里只是验证函数存在
	t.Run("测试函数存在", func(t *testing.T) {
		// 由于parseTdSegments需要goquery.Selection，这里只验证函数定义
		// 在实际测试中，应该创建mock的goquery.Selection对象
		t.Log("parseTdSegments函数已定义")
	})
}

// TestDownloadAndSaveImage 测试downloadAndSaveImage函数
func TestDownloadAndSaveImage(t *testing.T) {
	t.Run("测试无效URL", func(t *testing.T) {
		err := downloadAndSaveImage("http://invalid-url", "test-barcode", "./test-images")
		if err == nil {
			t.Error("期望函数返回错误，但没有错误")
		}
	})
}
