package main

import (
	"database/sql"
	"gorm.io/gorm/schema"
	"log"
	"os"
	"time"

	"github.com/gin-gonic/gin"
	"gorm.io/driver/sqlserver"
	"gorm.io/gorm"
	"shopbootx-go/config"
	"shopbootx-go/controllers"
	"shopbootx-go/middleware"
	"shopbootx-go/utils"
)

// initLogger 初始化日志配置
func initLogger(cfg *config.Config) (*utils.Logger, error) {
	// 创建日志目录
	logDir := "./logs"
	if err := os.MkdirAll(logDir, 0755); err != nil {
		log.Printf("警告: 无法创建日志目录 %s: %v", logDir, err)
	}

	// 设置日志格式
	log.SetFlags(log.LstdFlags | log.Lshortfile)

	// 创建自定义日志管理器
	logger, err := utils.NewLogger(logDir, "app.log")
	if err != nil {
		log.Printf("警告: 无法创建日志管理器: %v", err)
		return nil, err
	}

	// 如果配置了日志文件，则同时输出到文件
	if cfg.Log.File != "" {
		logFile, err := os.OpenFile(cfg.Log.File, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0666)
		if err != nil {
			log.Printf("警告: 无法打开日志文件 %s: %v", cfg.Log.File, err)
		} else {
			log.SetOutput(logFile)
		}
	}

	return logger, nil
}

func main() {
	log.Println("=== ShopBootX Go版本启动 ===")

	// 加载JSON配置文件
	log.Println("正在加载配置文件...")
	cfg := config.LoadConfigWithDefault("config.json")
	log.Printf("配置文件加载完成，服务器模式: %s", cfg.Server.Mode)

	// 初始化日志配置
	logger, err := initLogger(cfg)
	if err != nil {
		log.Printf("警告: 日志初始化失败: %v", err)
	}

	// 设置Gin模式
	//gin.SetMode(cfg.Server.Mode)
	gin.SetMode(gin.DebugMode)
	// 建立数据库连接
	log.Println("正在连接数据库...")
	log.Printf("数据库连接信息: %s:%d/%s", cfg.Database.Host, cfg.Database.Port, cfg.Database.Database)

	db, err := gorm.Open(sqlserver.Open(cfg.GetDSN()), &gorm.Config{
		NamingStrategy: schema.NamingStrategy{
			TablePrefix:   "",    // 不加前缀
			SingularTable: true,  // 单数表名
			// 注意不要用引号转义
		},
		DisableAutomaticPing: false,
	})
	if err != nil {
		log.Printf("错误: 数据库连接失败: %v", err)
		log.Printf("数据库DSN: %s", cfg.GetDSN())
		log.Fatal("程序终止: 无法连接到数据库")
	}
	log.Println("数据库连接成功")

	// 配置数据库连接池
	sqlDB, err := db.DB()
	if err != nil {
		log.Printf("错误: 获取数据库实例失败: %v", err)
		log.Fatal("程序终止: 无法获取数据库实例")
	}

	// 设置连接池参数
	sqlDB.SetMaxIdleConns(cfg.Database.MaxIdleConns)
	sqlDB.SetMaxOpenConns(cfg.Database.MaxOpenConns)
	sqlDB.SetConnMaxLifetime(0) // 使用默认值，因为ConnMaxLifetime需要time.Duration类型

	log.Printf("数据库连接池配置完成 - 最大空闲连接: %d, 最大打开连接: %d",
		cfg.Database.MaxIdleConns, cfg.Database.MaxOpenConns)

	// 测试数据库连接
	if err := sqlDB.Ping(); err != nil {
		log.Printf("错误: 数据库连接测试失败: %v", err)
		log.Fatal("程序终止: 数据库连接测试失败")
	}
	checkColumn(sqlDB)
	log.Println("数据库连接测试成功")
	// 创建Gin路由器实例
	log.Println("正在初始化HTTP路由器...")
	r := gin.New() // 使用gin.New()而不是gin.Default()，以便自定义中间件

	// 添加自定义中间件
	r.Use(middleware.LoggingMiddleware())            // 日志中间件
	r.Use(middleware.ErrorLoggingMiddleware())       // 错误日志中间件
	r.Use(middleware.DatabaseLoggingMiddleware())    // 数据库操作日志中间件
	r.Use(middleware.PerformanceLoggingMiddleware()) // 性能监控中间件
	r.Use(gin.Recovery())                            // 恢复中间件

	// 设置静态文件服务
	// 提供图片文件的访问服务，路径为/img，对应配置的文件路径
	log.Printf("设置静态文件服务: /img -> %s", cfg.File.UploadPath)
	r.Static("/img", cfg.File.UploadPath)

	controllers.RegisterRoutes(r, db)
	log.Println("API路由注册完成")

	// 启动HTTP服务器
	serverAddr := cfg.GetServerAddr()
	log.Printf("正在启动HTTP服务器: %s", serverAddr)
	log.Printf("服务器配置: 主机=%s, 端口=%d, 模式=%s", cfg.Server.Host, cfg.Server.Port, cfg.Server.Mode)

	// 记录启动时间
	startTime := time.Now()
	log.Printf("服务器启动时间: %s", startTime.Format("2006-01-02 15:04:05"))

	// 记录启动详情
	if logger != nil {
		startupDetails := map[string]interface{}{
			"服务器地址":  serverAddr,
			"数据库主机":  cfg.Database.Host,
			"数据库端口":  cfg.Database.Port,
			"数据库名称":  cfg.Database.Database,
			"服务器模式":  cfg.Server.Mode,
			"文件上传路径": cfg.File.UploadPath,
		}
		logger.LogStartup("ShopBootX Go服务器", startupDetails)
	}

	if err := r.Run(serverAddr); err != nil {
		log.Printf("错误: 服务器启动失败: %v", err)
		log.Printf("服务器运行时间: %v", time.Since(startTime))

		if logger != nil {
			logger.LogShutdown("ShopBootX Go服务器", time.Since(startTime))
		}

		log.Fatal("程序终止: HTTP服务器启动失败")
	}
}

func checkColumn(db *sql.DB) {
	// 目前需要检查 Purchase 中是否有 Bloqueado  Purchase Details表中是否有 Entrada
	// 检查字段是否存在
	// 一条 SQL 判断并添加字段
	sql := `
		IF NOT EXISTS (
			SELECT 1
			FROM INFORMATION_SCHEMA.COLUMNS
			WHERE TABLE_NAME = 'Purchase'
			  AND COLUMN_NAME = 'Bloqueado'
		)
		BEGIN
			ALTER TABLE Purchase ADD Bloqueado INT;
		END
	`
	if _, err := db.Exec(sql); err != nil {
		log.Fatal(err)
	}
	sql = `
		IF NOT EXISTS (
			SELECT 1 
			FROM INFORMATION_SCHEMA.COLUMNS 
			WHERE TABLE_NAME = '[Purchase Details]' 
			AND COLUMN_NAME = 'Entrada'
		)
		BEGIN
    		ALTER TABLE [Purchase Details] ADD Entrada NUMERIC(18,3) NULL;
		END
	`
	if _, err := db.Exec(sql); err != nil {
		log.Println(err)
	}
	log.Println("字段检查完成，不存在则已添加")
}
