// Package middleware 提供HTTP中间件
package middleware

import (
	"log"
	"time"

	"github.com/gin-gonic/gin"
)

// LoggingMiddleware 日志中间件
// 记录所有HTTP请求的详细信息，包括请求方法、路径、状态码、响应时间等
func LoggingMiddleware() gin.HandlerFunc {
	return gin.LoggerWithFormatter(func(param gin.LogFormatterParams) string {
		// 记录请求开始时间
		log.Printf("=== HTTP请求开始 ===")
		log.Printf("请求时间: %s", param.TimeStamp.Format("2006-01-02 15:04:05"))
		log.Printf("请求方法: %s", param.Method)
		log.Printf("请求路径: %s", param.Path)
		log.Printf("客户端IP: %s", param.ClientIP)
		log.Printf("用户代理: %s", param.Request.UserAgent())
		log.Printf("请求大小: %d 字节", param.Request.ContentLength)
		
		// 记录响应信息
		log.Printf("响应状态码: %d", param.StatusCode)
		log.Printf("响应大小: %d 字节", param.BodySize)
		log.Printf("响应时间: %v", param.Latency)
		
		// 根据状态码记录不同级别的日志
		if param.StatusCode >= 400 {
			log.Printf("错误: HTTP请求失败 - 状态码: %d, 路径: %s", param.StatusCode, param.Path)
		} else {
			log.Printf("成功: HTTP请求完成 - 状态码: %d, 路径: %s", param.StatusCode, param.Path)
		}
		
		log.Printf("=== HTTP请求结束 ===")
		
		// 返回空字符串，因为我们使用自定义的日志格式
		return ""
	})
}

// ErrorLoggingMiddleware 错误日志中间件
// 专门记录错误和异常的中间件
func ErrorLoggingMiddleware() gin.HandlerFunc {
	return func(c *gin.Context) {
		// 开始时间
		start := time.Now()
		
		// 处理请求
		c.Next()
		
		// 记录错误信息
		if len(c.Errors) > 0 {
			log.Printf("=== HTTP请求错误 ===")
			log.Printf("请求时间: %s", time.Now().Format("2006-01-02 15:04:05"))
			log.Printf("请求方法: %s", c.Request.Method)
			log.Printf("请求路径: %s", c.Request.URL.Path)
			log.Printf("客户端IP: %s", c.ClientIP())
			log.Printf("状态码: %d", c.Writer.Status())
			log.Printf("响应时间: %v", time.Since(start))
			
			for i, err := range c.Errors {
				log.Printf("错误 %d: %v", i+1, err.Error())
			}
			log.Printf("=== HTTP请求错误结束 ===")
		}
	}
}

// DatabaseLoggingMiddleware 数据库操作日志中间件
// 记录数据库相关的操作和错误
func DatabaseLoggingMiddleware() gin.HandlerFunc {
	return func(c *gin.Context) {
		// 在请求处理前记录数据库操作开始
		log.Printf("=== 数据库操作开始 ===")
		log.Printf("请求路径: %s", c.Request.URL.Path)
		log.Printf("请求方法: %s", c.Request.Method)
		
		// 处理请求
		c.Next()
		
		// 在请求处理后记录数据库操作结束
		log.Printf("=== 数据库操作结束 ===")
	}
}

// PerformanceLoggingMiddleware 性能监控中间件
// 记录请求的性能指标
func PerformanceLoggingMiddleware() gin.HandlerFunc {
	return func(c *gin.Context) {
		start := time.Now()
		
		// 处理请求
		c.Next()
		
		duration := time.Since(start)
		
		// 记录性能信息
		log.Printf("=== 性能监控 ===")
		log.Printf("请求路径: %s", c.Request.URL.Path)
		log.Printf("请求方法: %s", c.Request.Method)
		log.Printf("响应时间: %v", duration)
		log.Printf("状态码: %d", c.Writer.Status())
		log.Printf("响应大小: %d 字节", c.Writer.Size())
		
		// 如果响应时间超过阈值，记录警告
		if duration > 5*time.Second {
			log.Printf("警告: 请求响应时间过长 - %v", duration)
		}
		
		log.Printf("=== 性能监控结束 ===")
	}
} 