// Package utils 提供工具函数
package utils

import (
	"fmt"
	"log"
	"os"
	"path/filepath"
	"time"
)

// Logger 日志管理器
type Logger struct {
	infoLogger    *log.Logger
	errorLogger   *log.Logger
	warningLogger *log.Logger
	debugLogger   *log.Logger
}

// NewLogger 创建新的日志管理器
func NewLogger(logDir, logFile string) (*Logger, error) {
	// 创建日志目录
	if err := os.MkdirAll(logDir, 0755); err != nil {
		return nil, fmt.Errorf("创建日志目录失败: %v", err)
	}

	// 创建日志文件
	logFilePath := filepath.Join(logDir, logFile)
	file, err := os.OpenFile(logFilePath, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0666)
	if err != nil {
		return nil, fmt.Errorf("创建日志文件失败: %v", err)
	}

	// 创建不同级别的日志记录器
	infoLogger := log.New(file, "[INFO] ", log.LstdFlags|log.Lshortfile)
	errorLogger := log.New(file, "[ERROR] ", log.LstdFlags|log.Lshortfile)
	warningLogger := log.New(file, "[WARNING] ", log.LstdFlags|log.Lshortfile)
	debugLogger := log.New(file, "[DEBUG] ", log.LstdFlags|log.Lshortfile)

	return &Logger{
		infoLogger:    infoLogger,
		errorLogger:   errorLogger,
		warningLogger: warningLogger,
		debugLogger:   debugLogger,
	}, nil
}

// Info 记录信息日志
func (l *Logger) Info(format string, v ...interface{}) {
	message := fmt.Sprintf(format, v...)
	l.infoLogger.Printf("%s", message)
	// 同时输出到控制台
	fmt.Printf("[INFO] %s\n", message)
}

// Error 记录错误日志
func (l *Logger) Error(format string, v ...interface{}) {
	message := fmt.Sprintf(format, v...)
	l.errorLogger.Printf("%s", message)
	// 同时输出到控制台
	fmt.Printf("[ERROR] %s\n", message)
}

// Warning 记录警告日志
func (l *Logger) Warning(format string, v ...interface{}) {
	message := fmt.Sprintf(format, v...)
	l.warningLogger.Printf("%s", message)
	// 同时输出到控制台
	fmt.Printf("[WARNING] %s\n", message)
}

// Debug 记录调试日志
func (l *Logger) Debug(format string, v ...interface{}) {
	message := fmt.Sprintf(format, v...)
	l.debugLogger.Printf("%s", message)
	// 同时输出到控制台
	fmt.Printf("[DEBUG] %s\n", message)
}

// LogRequest 记录HTTP请求日志
func (l *Logger) LogRequest(method, path, clientIP string, statusCode int, duration time.Duration) {
	l.Info("HTTP请求: %s %s - IP: %s - 状态码: %d - 耗时: %v", 
		method, path, clientIP, statusCode, duration)
}

// LogDatabase 记录数据库操作日志
func (l *Logger) LogDatabase(operation, table string, rowsAffected int64, err error) {
	if err != nil {
		l.Error("数据库操作失败: %s %s - 影响行数: %d - 错误: %v", 
			operation, table, rowsAffected, err)
	} else {
		l.Info("数据库操作成功: %s %s - 影响行数: %d", 
			operation, table, rowsAffected)
	}
}

// LogFileOperation 记录文件操作日志
func (l *Logger) LogFileOperation(operation, filePath string, err error) {
	if err != nil {
		l.Error("文件操作失败: %s %s - 错误: %v", operation, filePath, err)
	} else {
		l.Info("文件操作成功: %s %s", operation, filePath)
	}
}

// LogConfig 记录配置相关日志
func (l *Logger) LogConfig(operation, configPath string, err error) {
	if err != nil {
		l.Error("配置操作失败: %s %s - 错误: %v", operation, configPath, err)
	} else {
		l.Info("配置操作成功: %s %s", operation, configPath)
	}
}

// LogStartup 记录启动日志
func (l *Logger) LogStartup(component string, details map[string]interface{}) {
	l.Info("=== %s 启动 ===", component)
	for key, value := range details {
		l.Info("%s: %v", key, value)
	}
}

// LogShutdown 记录关闭日志
func (l *Logger) LogShutdown(component string, duration time.Duration) {
	l.Info("=== %s 关闭 - 运行时间: %v ===", component, duration)
} 