package controller

import (
	"encoding/base64"
	"fmt"
	"github.com/gin-gonic/gin"
	"io/ioutil"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"supermarket-go/internal/model"
	"supermarket-go/internal/service"
	"supermarket-go/internal/util"
)

type Proveedor struct {
	EmpresaID int64  `json:"empresaID"`
	NombreES  string `json:"nombreES"`
}

// ImageUploadRequest 图片上传请求结构
type ImageUploadRequest struct {
	ArticuloID string `json:"articuloID" binding:"required"`
	Dibujo     string `json:"dibujo" binding:"required"`
}

// 统一响应结构
func buildResponse(code, message string, data interface{}) map[string]interface{} {
	return map[string]interface{}{
		"code":    code,
		"message": message,
		"data":    data,
	}
}

// 假设有全局变量 articuloService 实现了相关方法
var articuloService model.ArticuloService

func RegisterArticuloRoutes(r *gin.Engine) {
	log.Println("[INFO] 注册 articulo 路由组: /articulo")
	articuloService = service.NewArticuloService(model.DB)
	g := r.Group("/articulo")
	{
		g.POST("/scan", ArticuloScanHandler)
		g.POST("/update", ArticuloUpdateHandler)
		g.POST("/edit", ArticuloEditHandler)
		g.POST("/add", ArticuloAddHandler)
		g.POST("/getClass", ArticuloGetClassHandler)
		g.POST("/proveedor", ArticuloProveedorHandler)
		g.POST("/updateByArticuloIDOrCodigoBarra", ArticuloUpdateByArticuloIDOrCodigoBarraHandler)
		g.POST("/getByArticuloIDOrCodigoBarra", ArticuloGetByArticuloIDOrCodigoBarraHandler)
		g.POST("/zbm", ArticuloZbmHandler)
		g.POST("/upload", ImageUploadHandler)
		g.GET("/image/:articuloID", ImageGetHandler)
	}
	log.Println("[INFO] articulo 路由注册完成")
}

func ArticuloScanHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品扫描请求 =====")

	var request model.ArticuloScanRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("[ERROR] 解析扫描请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 扫描请求参数: ArticuloID=%s, CodigoBarra=%s",
		request.ArticuloID, request.CodigoBarra)

	articulo := articuloService.Scan(request)
	if articulo == nil {
		log.Printf("[WARN] 未找到商品: ArticuloID=%s, CodigoBarra=%s",
			request.ArticuloID, request.CodigoBarra)
		c.JSON(http.StatusOK, buildResponse("404", "未找到商品", nil))
		return
	}

	log.Printf("[INFO] 成功找到商品: ArticuloID=%s, Descripcion=%s",
		articulo.Articulo.ArticuloID, articulo.Articulo.Descripcion)

	// 字段映射
	resp := model.ArticuloScanResponse{
		ArticuloID:               articulo.Articulo.ArticuloID,
		EmpresaID:                fmt.Sprintf("%d", articulo.Articulo.EmpresaID),
		StockDA:                  fmt.Sprintf("%.2f", articulo.Stockda),
		CodigoBarra:              articulo.Articulo.CodigoBarra,
		Descripcion:              articulo.Articulo.Descripcion,
		PrecioDetalle:            articulo.Articulo.PrecioDetalle,
		PrecioDomicilio:          articulo.Articulo.PrecioDomicilio,
		PrecioSocio:              articulo.Articulo.PrecioSocio,
		PrecioMayor:              articulo.Articulo.PrecioMayor,
		PrecioFactura:            articulo.Articulo.PrecioFactura,
		PrecioInternet:           articulo.Articulo.PrecioInternet,
		PrecioAmigo:              articulo.Articulo.PrecioAmigo,
		PrecioEspecial:           articulo.Articulo.PrecioEspecial,
		PrecioOferta:             articulo.Articulo.PrecioOferta,
		PrecioCoste:              articulo.Articulo.PrecioCoste,
		NombreES:                 articulo.Articulo.NombreES,
		NombreCN:                 articulo.Articulo.NombreCN,
		CantidadPorUnidad:        articulo.Articulo.CantidadPorUnidad,
		CantidadPorUnidad2:       articulo.Articulo.CantidadPorUnidad2,
		VolumenPeso:              articulo.Articulo.VolumenPeso,
		ClaseID:                  int(articulo.Articulo.ClaseID),
		UnidadNombre:             articulo.Articulo.UnidadNombre,
		Descuento:                articulo.Articulo.Descuento,
		FacturaPorcentaje:        articulo.Articulo.FacturaPorcentaje,
		Oferta:                   int(articulo.Articulo.Oferta),
		ProveedorDD:              articulo.Proveedordd,
		Privado:                  int(articulo.Articulo.Privado),
		UsarPrecioPorCantidad:    int(articulo.Articulo.UsarPrecioPorCantidad),
		UsarDescuentoPorCantidad: int(articulo.Articulo.UsarDescuentoPorCantidad),
		Precio1:                  articulo.Articulo.Precio1,
		Descuento1:               articulo.Articulo.Descuento1,
		Cantidad1:                articulo.Articulo.Cantidad1,
		Precio2:                  articulo.Articulo.Precio2,
		Descuento2:               articulo.Articulo.Descuento2,
		Cantidad2:                articulo.Articulo.Cantidad2,
		Precio3:                  articulo.Articulo.Precio3,
		Descuento3:               articulo.Articulo.Descuento3,
		Cantidad3:                articulo.Articulo.Cantidad3,
		Precio4:                  articulo.Articulo.Precio4,
		Descuento4:               articulo.Articulo.Descuento4,
		Cantidad4:                articulo.Articulo.Cantidad4,
		Precio5:                  articulo.Articulo.Precio5,
		Descuento5:               articulo.Articulo.Descuento5,
		Cantidad5:                articulo.Articulo.Cantidad5,
		Precio6:                  articulo.Articulo.Precio6,
		Descuento6:               articulo.Articulo.Descuento6,
		SitioEnAlmacen:           articulo.Articulo.SitioEnAlmacen,
		MiniStock:                articulo.Articulo.MiniStock,
		MaxiStock:                articulo.Articulo.MaxiStock,
		DescuentoCambioProhibido: int(articulo.Articulo.DescuentoCambioProhibido),
		UnidadUsarRegla:          int(articulo.Articulo.UnidadUsarRegla),
		MultiCodigo:              articulo.Articulo.MultiCodigo,
		Bloqueado:                int(articulo.Articulo.Bloqueado),
		//Observacion:              *articulo.Articulo.Observacion,
		DibujoID: int(articulo.Articulo.DibujoID),
		// 处理指针类型字段，暂时设为 nil
		PrimeraLetra:           nil,
		Pinyin:                 nil,
		FechaPrecioCoste:       nil,
		ValeDescuento:          nil,
		FechaCaducada:          nil,
		DibujoMD5:              nil,
		TiempoDibujoModificado: nil,
		ConsultaParcial:        nil,
		FechaEntrada:           nil,
		SignoEspecial:          nil,
		Dibujo:                 nil,
	}

	// 处理 Dibujo 字段的类型转换 - 压缩后返回缩略图
	if len(articulo.Dibujo) > 0 {
		log.Printf("[INFO] 商品包含图片数据，长度: %d bytes", len(articulo.Dibujo))
		
		// articulo.Dibujo 是 base64 编码的字节数组，先转换为字符串
		dibujoBase64Str := string(articulo.Dibujo)
		
		// 压缩图片（缩略图：最大800x800，质量75%）
		compressedBase64, err := util.CompressImageToBase64(dibujoBase64Str, 800, 800, 75)
		if err != nil {
			log.Printf("[WARN] 图片压缩失败，返回原始图片: %v", err)
			// 如果压缩失败，仍然返回原始图片的base64字符串
			resp.Dibujo = &dibujoBase64Str
		} else {
			resp.Dibujo = &compressedBase64
			log.Printf("[INFO] 图片已压缩为缩略图")
		}
	}

	// 处理时间字段的类型转换
	if articulo.Articulo.FechaPrecioCoste != nil {
		fechaPrecioCoste := articulo.Articulo.FechaPrecioCoste.Format("2006-01-02 15:04:05")
		resp.FechaPrecioCoste = &fechaPrecioCoste
	}

	if articulo.Articulo.FechaCaducada != nil {
		fechaCaducada := articulo.Articulo.FechaCaducada.Format("2006-01-02 15:04:05")
		resp.FechaCaducada = &fechaCaducada
	}

	log.Println("[INFO] ===== 商品扫描请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "扫描成功", resp))
}

func ArticuloUpdateHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品更新请求 =====")

	var request model.ArticuloInputRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("[ERROR] 解析更新请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 更新请求参数: ArticuloID=%s, CodigoBarra=%s, PrecioDetalle=%s, StockDA=%s",
		request.ArticuloID, request.CodigoBarra, request.PrecioDetalle, request.StockDA)

	result := articuloService.Update(request)

	if result != nil {
		log.Printf("[ERROR] 商品更新失败: %v", result)
	} else {
		log.Printf("[INFO] 商品更新成功: ArticuloID=%s", request.ArticuloID)
	}

	log.Println("[INFO] ===== 商品更新请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "修改价格成功", result))
}

func ArticuloEditHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品编辑请求 =====")

	var ac model.Articulo
	if err := c.ShouldBindJSON(&ac); err != nil {
		log.Printf("[ERROR] 解析编辑请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 编辑请求参数: ArticuloID=%s, Descripcion=%s",
		ac.ArticuloID, ac.Descripcion)

	result := articuloService.Edit(ac)

	if result != nil {
		log.Printf("[ERROR] 商品编辑失败: %v", result)
	} else {
		log.Printf("[INFO] 商品编辑成功: ArticuloID=%s", ac.ArticuloID)
	}

	log.Println("[INFO] ===== 商品编辑请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "修改价格成功", result))
}

func ArticuloAddHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品添加请求 =====")

	var request model.ArticuloInputRequest
	if err := c.ShouldBindJSON(&request); err != nil {
		log.Printf("[ERROR] 解析添加请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 添加请求参数: ArticuloID=%s, CodigoBarra=%s, Descripcion=%s",
		request.ArticuloID, request.CodigoBarra, request.NombreES)

	result := articuloService.Add(request)

	if result != nil {
		log.Printf("[ERROR] 商品添加失败: %v", result)
	} else {
		log.Printf("[INFO] 商品添加成功: ArticuloID=%s", request.ArticuloID)
	}

	log.Println("[INFO] ===== 商品添加请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "添加商品成功", result))
}

func ArticuloGetClassHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取商品类别请求 =====")

	articuloClases := articuloService.GetAcClass()
	log.Printf("[INFO] 查询到 %d 个商品类别", len(articuloClases))

	var claseBriefResponses []model.ClaseBriefResponse
	for _, articuloClase := range articuloClases {
		claseBriefResponses = append(claseBriefResponses, model.ClaseBriefResponse{
			ClaseID:  fmt.Sprintf("%d", articuloClase.ClaseID),
			NombreES: articuloClase.NombreES,
		})
	}

	log.Println("[INFO] ===== 获取商品类别请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "获取商品类别成功", claseBriefResponses))
}

func ArticuloProveedorHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取供应商请求 =====")

	proveedors := articuloService.GetProveedor()
	log.Printf("[INFO] 查询到 %d 个供应商", len(proveedors))

	var proveedorResponse []model.ProveedorResponse
	for _, proveedor := range proveedors {
		proveedorResponse = append(proveedorResponse, model.ProveedorResponse{
			EmpresaID: fmt.Sprintf("%d", proveedor.EmpresaID),
			NombreES:  proveedor.NombreES,
		})
	}

	log.Println("[INFO] ===== 获取供应商请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "获取供应商成功", proveedorResponse))
}

func ArticuloUpdateByArticuloIDOrCodigoBarraHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理按商品ID或条码更新请求 =====")

	var ac model.Articulo
	if err := c.ShouldBindJSON(&ac); err != nil {
		log.Printf("[ERROR] 解析更新请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 更新请求参数: ArticuloID=%s", ac.ArticuloID)

	num := articuloService.UpdateByArticuloIDOrCodigoBarra(ac)
	log.Printf("[INFO] 更新影响行数: %d", num)

	articulos := articuloService.GetByArticuloIDOrCodigoBarra(ac.ArticuloID)
	var articulo *model.Articulo
	if len(articulos) > 0 {
		articulo = &articulos[0]
		log.Printf("[INFO] 查询到更新后的商品: ArticuloID=%s", articulo.ArticuloID)
	} else {
		log.Printf("[WARN] 未查询到更新后的商品")
	}

	msg := "修改按量定价信息成功" + itoa(num)
	log.Println("[INFO] ===== 按商品ID或条码更新请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", msg, articulo))
}

func ArticuloGetByArticuloIDOrCodigoBarraHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理按商品ID或条码查询请求 =====")

	var ac model.Articulo
	if err := c.ShouldBindJSON(&ac); err != nil {
		log.Printf("[ERROR] 解析查询请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 查询请求参数: ArticuloID=%s", ac.ArticuloID)

	articulos := articuloService.GetByArticuloIDOrCodigoBarraLike(ac.ArticuloID)
	log.Printf("[INFO] 查询到 %d 个商品", len(articulos))

	if len(articulos) > 1 {
		log.Printf("[WARN] 查询结果过多，需要更精确的查询条件")
		c.JSON(http.StatusOK, buildResponse("1001", "请填写完整编号后再查询，目前编号查询出多个", nil))
		return
	} else if len(articulos) == 0 {
		log.Printf("[WARN] 未查询到商品: ArticuloID=%s", ac.ArticuloID)
		c.JSON(http.StatusOK, buildResponse("1002", "未查询到商品", nil))
		return
	}

	log.Printf("[INFO] 成功查询到商品: ArticuloID=%s, Descripcion=%s",
		articulos[0].ArticuloID, articulos[0].Descripcion)

	log.Println("[INFO] ===== 按商品ID或条码查询请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "查询产品信息成功", articulos[0]))
}

func ArticuloZbmHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理自编码生成请求 =====")

	var zbmParam model.ZbmParam
	if err := c.ShouldBindJSON(&zbmParam); err != nil {
		log.Printf("[ERROR] 解析自编码请求失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	//log.Printf("[INFO] 自编码请求参数: CodePrefix=%s, CodeLength=%d",
	//	zbmParam.CodePrefix, zbmParam.CodeLength)

	result := articuloService.Zbm(zbmParam)

	log.Printf("[INFO] 生成自编码: %v", result)
	log.Println("[INFO] ===== 自编码生成请求处理完成 =====")
	c.JSON(http.StatusOK, buildResponse("200", "获取新自编码成功", result))
}


func ImageGetHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理图片获取请求 =====")

	articuloID := c.Param("articuloID")
	if articuloID == "" {
		log.Printf("[ERROR] ArticuloID参数为空")
		c.JSON(http.StatusBadRequest, buildResponse("400", "商品ID不能为空", nil))
		return
	}

	log.Printf("[INFO] 图片获取请求参数: ArticuloID=%s", articuloID)

	// 构建图片文件路径
	filePath := filepath.Join("image", fmt.Sprintf("%s.jpg", articuloID))

	// 检查文件是否存在
	if _, err := os.Stat(filePath); os.IsNotExist(err) {
		log.Printf("[WARN] 图片文件不存在: ArticuloID=%s, FilePath=%s", articuloID, filePath)
		c.JSON(http.StatusNotFound, buildResponse("404", "图片不存在", nil))
		return
	}

	// 读取文件内容
	fileBytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		log.Printf("[ERROR] 图片读取失败: %v", err)
		c.JSON(http.StatusInternalServerError, buildResponse("500", "图片读取失败", nil))
		return
	}

	// 转换为 Base64
	base64Str := base64.StdEncoding.EncodeToString(fileBytes)

	// 返回 JSON 响应
	c.JSON(http.StatusOK, buildResponse("200", "图片获取成功", gin.H{
		"articulo_id": articuloID,
		"image_base64": base64Str,
	}))

	log.Printf("[INFO] 图片获取成功: ArticuloID=%s, FilePath=%s", articuloID, filePath)
	log.Println("[INFO] ===== 图片获取请求处理完成 =====")
}

func itoa(i int) string {
	return fmt.Sprintf("%d", i)
}

// parseInt64 工具函数
func parseInt64(s string) int64 {
	var i int64
	fmt.Sscanf(s, "%d", &i)
	return i
}
