package controller

import (
	"github.com/gin-gonic/gin"
	"log"
	"net/http"
	"supermarket-go/internal/service"
	"supermarket-go/internal/util"
)

type EmpleadoAuthController struct {
	AuthService     *service.EmpleadoAuthService
	EmpleadoService *service.EmpleadoService
}

func RegisterEmpleadoAuthRoutes(r *gin.Engine, authSvc *service.EmpleadoAuthService, empSvc *service.EmpleadoService) {
	log.Println("[INFO] 注册 empleadoAuth 路由组: /empleadoAuth")
	ctrl := &EmpleadoAuthController{AuthService: authSvc, EmpleadoService: empSvc}
	g := r.Group("/empleadoAuth")
	{
		g.POST("/login", ctrl.Login)
		g.GET("/info", ctrl.Info)
		g.GET("/listUsers", ctrl.ListUsers)
	}
	log.Println("[INFO] empleadoAuth 路由注册完成")
}

// Login 职员登录
func (ctrl *EmpleadoAuthController) Login(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理员工登录请求 =====")

	empleadoId := c.PostForm("id")
	contrasena := c.PostForm("contrasena")

	log.Printf("[INFO] 登录请求参数: empleadoId=%s", empleadoId)

	if empleadoId == "" || contrasena == "" {
		log.Printf("[ERROR] 登录参数不完整: empleadoId=%s, contrasena=%s",
			empleadoId, func() string {
				if contrasena == "" {
					return "空"
				}
				return "已提供"
			}())
		response := util.BuildResponse(400, "员工ID或密码不能为空", "")
		c.JSON(http.StatusOK, response)
		return
	}

	log.Printf("[INFO] 开始验证员工登录: empleadoId=%s", empleadoId)
	token := ctrl.AuthService.Login(empleadoId, contrasena)

	if token != "" {
		log.Printf("[INFO] 员工登录成功: empleadoId=%s", empleadoId)
		response := util.BuildSuccessResponse("登录成功", token)
		c.JSON(http.StatusOK, response)
	} else {
		log.Printf("[WARN] 员工登录失败: empleadoId=%s, 原因: ID或密码错误", empleadoId)
		response := util.BuildResponse(403, "员工ID或密码错误", "")
		c.JSON(http.StatusOK, response)
	}

	log.Println("[INFO] ===== 员工登录请求处理完成 =====")
}

// Info 获取职员信息
func (ctrl *EmpleadoAuthController) Info(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取员工信息请求 =====")

	token := c.GetHeader("Authorization")
	if token == "" {
		log.Printf("[ERROR] 缺少Authorization header")
		response := util.BuildResponse(401, "缺少认证信息", "")
		c.JSON(http.StatusOK, response)
		return
	}

	log.Printf("[INFO] 验证token: %s", token[:min(len(token), 20)]+"...")

	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		log.Printf("[ERROR] token验证失败: %v", err)
		response := util.BuildResponse(403, "验证信息无效", "")
		c.JSON(http.StatusOK, response)
		return
	}

	log.Printf("[INFO] token验证成功，员工ID: %s", empleadoId)

	empleado := ctrl.EmpleadoService.GetEmpleadoById(empleadoId)
	if empleado == nil {
		log.Printf("[ERROR] 未找到员工信息: empleadoId=%s", empleadoId)
		response := util.BuildInternalServerErrorResponse("找不到该员工", nil)
		c.JSON(http.StatusOK, response)
		return
	}

	log.Printf("[INFO] 成功获取员工信息: empleadoId=%s, nombre=%s",
		empleadoId, empleado.Nombre)

	data := map[string]interface{}{
		"name": empleado.Nombre,
	}
	response := util.BuildSuccessResponse("", data)
	c.JSON(http.StatusOK, response)

	log.Println("[INFO] ===== 获取员工信息请求处理完成 =====")
}

// ListUsers 获取用户列表
func (ctrl *EmpleadoAuthController) ListUsers(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取用户列表请求 =====")

	names := ctrl.EmpleadoService.GetAllEmpleadoNames()
	log.Printf("[INFO] 查询到 %d 个员工", len(names))

	response := util.BuildSuccessResponseData(names)
	c.JSON(http.StatusOK, response)

	log.Println("[INFO] ===== 获取用户列表请求处理完成 =====")
}
