package controller

import (
	"github.com/gin-gonic/gin"
	"log"
	"net/http"
	"strconv"
	"supermarket-go/internal/model"
	"supermarket-go/internal/service"
	"supermarket-go/internal/util"
	"time"
)

// 全局变量
var orderService model.OrderService

// RegisterOrderRoutes 注册订单相关路由
func RegisterOrderRoutes(r *gin.Engine) {
	log.Println("[INFO] 注册 order 路由组: /order")
	orderService = service.NewOrderService(model.DB)
	g := r.Group("/order")
	{
		g.GET("/getTicket", GetTicketHandler)
		g.GET("/getAllSuspendedTickets", GetAllSuspendedTicketsHandler)
		g.POST("/submitOrder", SubmitOrderHandler)
	}
	log.Println("[INFO] order 路由注册完成")
}

// GetTicketHandler 获取订单详情
func GetTicketHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取订单请求 =====")

	ticketIdStr := c.Query("ticketId")
	if ticketIdStr == "" {
		log.Printf("[ERROR] 缺少必要参数: ticketId")
		c.JSON(http.StatusBadRequest, buildResponse("400", "缺少必要参数: ticketId", nil))
		return
	}

	ticketId, err := strconv.ParseInt(ticketIdStr, 10, 64)
	if err != nil {
		log.Printf("[ERROR] 解析ticketId失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "ticketId格式错误", nil))
		return
	}

	log.Printf("[INFO] 获取订单: TicketID=%d", ticketId)

	nestedTicket := orderService.GetTicketById(ticketId)
	if nestedTicket == nil {
		log.Printf("[WARN] 未找到订单: TicketID=%d", ticketId)
		c.JSON(http.StatusOK, buildResponse("404", "未找到订单", nil))
		return
	}

	// 转换Ticket为前端需要的结构体
	ticketResp := convertTicketToResponse(nestedTicket.Ticket)
	detailResp := make([]TicketArticuloResponse, 0, len(nestedTicket.Details))
	for _, d := range nestedTicket.Details {
		detailResp = append(detailResp, convertTicketArticuloToResponse(d))
	}

	resp := map[string]interface{}{
		"ticket":  ticketResp,
		"details": detailResp,
	}

	log.Printf("[INFO] 成功获取订单: TicketID=%d", ticketId)
	c.JSON(http.StatusOK, map[string]interface{}{
		"code":    "200",
		"data":    resp,
		"message": "",
	})
}

// TicketResponse 用于返回给前端的订单结构体
// 字段与前端JSON严格一致
// 这里只写部分字段，全部字段请根据JSON补全

type TicketResponse struct {
	DocumentoNo             int64   `json:"documentoNo"`
	DocumentoNoEnServidor   int64   `json:"documentoNoEnServidor"`
	DocumentoNoOriginal     int64   `json:"documentoNoOriginal"`
	Fecha                   *int64  `json:"fecha"`
	FechaEntrada            *int64  `json:"fechaEntrada"`
	FechaOriginal           *int64  `json:"fechaOriginal"`
	TiempoUltimoEditar      *int64  `json:"tiempoUltimoEditar"`
	TiempoUltimoActualizar  *int64  `json:"tiempoUltimoActualizar"`
	DocumentoOrigenTipo     string  `json:"documentoOrigenTipo"`
	DocumentoOrigenNo       uint64  `json:"documentoOrigenNo"`
	DocumentoDestinoTipo    string  `json:"documentoDestinoTipo"`
	DocumentoDestinoNo      uint64  `json:"documentoDestinoNo"`
	CobrarTipo              uint8   `json:"cobrarTipo"`
	FormaDePago             string  `json:"formaDePago"`
	Efectivo                float64 `json:"efectivo"`
	Descuento               float64 `json:"descuento"`
	Descuento2              float64 `json:"descuento2"`
	DescuentoDirecto        float64 `json:"descuentoDirecto"`
	Total                   float64 `json:"total"`
	DescuentoTotal          float64 `json:"descuentoTotal"`
	DescuentoExtra          float64 `json:"descuentoExtra"`
	DescuentoUsarRegla      uint16  `json:"descuentoUsarRegla"`
	Ganancias               float64 `json:"ganancias"`
	CobroTotal              float64 `json:"cobroTotal"`
	CantidadTotal           float64 `json:"cantidadTotal"`
	PaqueteTotal            float64 `json:"paqueteTotal"`
	CajaTotal               float64 `json:"cajaTotal"`
	VolumenPesoTotal        float64 `json:"volumenPesoTotal"`
	LineaTotal              int     `json:"lineaTotal"`
	ReglaIVA                uint8   `json:"reglaIVA"`
	PrecioUsado             uint8   `json:"precioUsado"`
	CajonID                 uint16  `json:"cajonID"`
	EmpresaID               int64   `json:"empresaID"`
	SucursalID              uint16  `json:"sucursalID"`
	ClienteIVAClase         uint8   `json:"clienteIVAClase"`
	ClienteReglaIVA         uint8   `json:"clienteReglaIVA"`
	UsarClienteReglaIVA     uint8   `json:"usarClienteReglaIVA"`
	AgenteID                uint16  `json:"agenteID"`
	TransportistaID         uint16  `json:"transportistaID"`
	TransporteModoID        uint16  `json:"transporteModoID"`
	EnvioGasto              float64 `json:"envioGasto"`
	CocheNo                 string  `json:"cocheNo"`
	OperadorID              uint16  `json:"operadorID"`
	VerificadorID           uint16  `json:"verificadorID"`
	AlmacenID               uint16  `json:"almacenID"`
	DocumentoAdjuntoSimbolo string  `json:"documentoAdjuntoSimbolo"`
	PreFormaDePagoNo        uint    `json:"preFormaDePagoNo"`
	OrdenadorNombre         string  `json:"ordenadorNombre"`
	DeviceUUID              string  `json:"deviceUUID"`
	DocumentoHash           string  `json:"documentoHash"`
	SubTiendaID             string  `json:"subTiendaID"`
	DocumentoIntercambioNo  int64   `json:"documentoIntercambioNo"`
	AbonoTotal              float64 `json:"abonoTotal"`
	Cobrado                 int8    `json:"cobrado"`
	Bloqueado               uint8   `json:"bloqueado"`
	Anulado                 uint8   `json:"anulado"`
	Modificando             uint8   `json:"modificando"`
	EmailMandado            uint8   `json:"emailMandado"`
	ProgressID              uint8   `json:"progressID"`
	ProgressStatus          uint8   `json:"progressStatus"`
	LogID                   uint64  `json:"logID"`
	HayAdjunto              uint8   `json:"hayAdjunto"`
	Imprimido               uint8   `json:"imprimido"`
	Observacion             *string `json:"observacion"`
	IVASerie                uint8   `json:"ivaserie"`
}

// TicketArticuloResponse 用于返回给前端的订单明细结构体
// 字段与前端JSON严格一致
// 这里只写部分字段，全部字段请根据JSON补全

type TicketArticuloResponse struct {
	DocumentoNo              uint64  `json:"documentoNo"`
	ArticuloID               string  `json:"articuloID"`
	PropCount                uint16  `json:"propCount"`
	PropID1                  string  `json:"propID1"`
	PropID2                  string  `json:"propID2"`
	Propiedad                string  `json:"propiedad"`
	SerialNo                 string  `json:"serialNo"`
	CodigoBarra              string  `json:"codigoBarra"`
	CodigoBarraBalanza       uint8   `json:"codigoBarraBalanza"`
	NombreES                 string  `json:"nombreES"`
	NombreCN                 string  `json:"nombreCN"`
	Precio                   float64 `json:"precio"`
	Cantidad                 float64 `json:"cantidad"`
	Descuento                float64 `json:"descuento"`
	DescuentoCambioProhibido uint8   `json:"descuentoCambioProhibido"`
	PrecioCoste              float64 `json:"precioCoste"`
	Comentario               *string `json:"comentario"`
	NecesitaSerialNo         uint8   `json:"necesitaSerialNo"`
	Temporal                 uint8   `json:"temporal"`
	OrdenNo                  int     `json:"ordenNo"`
	IVA                      float64 `json:"iva"`
	REQ                      float64 `json:"req"`
}

// 转换函数，将Ticket结构体转换为TicketResponse
func convertTicketToResponse(t model.Ticket) TicketResponse {
	return TicketResponse{
		DocumentoNo:             t.DocumentoNo,
		DocumentoNoEnServidor:   t.DocumentoNoEnServidor,
		DocumentoNoOriginal:     t.DocumentoNoOriginal,
		Fecha:                   timePtrToMillis(&t.Fecha),
		FechaEntrada:            timeToMillisPtr(t.FechaEntrada),
		FechaOriginal:           timePtrToMillis(t.FechaOriginal),
		TiempoUltimoEditar:      timePtrToMillis(t.TiempoUltimoEditar),
		TiempoUltimoActualizar:  timePtrToMillis(&t.TiempoUltimoActualizar),
		DocumentoOrigenTipo:     t.DocumentoOrigenTipo,
		DocumentoOrigenNo:       t.DocumentoOrigenNo,
		DocumentoDestinoTipo:    t.DocumentoDestinoTipo,
		DocumentoDestinoNo:      t.DocumentoDestinoNo,
		CobrarTipo:              t.CobrarTipo,
		FormaDePago:             t.FormaDePago,
		Efectivo:                t.Efectivo,
		Descuento:               t.Descuento,
		Descuento2:              t.Descuento2,
		DescuentoDirecto:        t.DescuentoDirecto,
		Total:                   t.Total,
		DescuentoTotal:          t.DescuentoTotal,
		DescuentoExtra:          t.DescuentoExtra,
		DescuentoUsarRegla:      t.DescuentoUsarRegla,
		Ganancias:               t.Ganancias,
		CobroTotal:              t.CobroTotal,
		CantidadTotal:           t.CantidadTotal,
		PaqueteTotal:            t.PaqueteTotal,
		CajaTotal:               t.CajaTotal,
		VolumenPesoTotal:        t.VolumenPesoTotal,
		LineaTotal:              t.LineaTotal,
		ReglaIVA:                t.ReglaIVA,
		PrecioUsado:             t.PrecioUsado,
		CajonID:                 t.CajonID,
		EmpresaID:               t.EmpresaID,
		SucursalID:              t.SucursalID,
		ClienteIVAClase:         t.ClienteIVAClase,
		ClienteReglaIVA:         t.ClienteReglaIVA,
		UsarClienteReglaIVA:     boolToUint8(t.UsarClienteReglaIVA),
		AgenteID:                t.AgenteID,
		TransportistaID:         t.TransportistaID,
		TransporteModoID:        t.TransporteModoID,
		EnvioGasto:              t.EnvioGasto,
		CocheNo:                 t.CocheNo,
		OperadorID:              t.OperadorID,
		VerificadorID:           t.VerificadorID,
		AlmacenID:               t.AlmacenID,
		DocumentoAdjuntoSimbolo: t.DocumentoAdjuntoSimbolo,
		PreFormaDePagoNo:        t.PreFormaDePagoNo,
		OrdenadorNombre:         t.OrdenadorNombre,
		DeviceUUID:              t.DeviceUUID,
		DocumentoHash:           t.DocumentoHash,
		SubTiendaID:             t.SubTiendaID,
		DocumentoIntercambioNo:  t.DocumentoIntercambioNo,
		AbonoTotal:              t.AbonoTotal,
		Cobrado:                 t.Cobrado,
		Bloqueado:               boolToUint8(t.Bloqueado),
		Anulado:                 boolToUint8(t.Anulado),
		Modificando:             boolToUint8(t.Modificando),
		EmailMandado:            boolToUint8(t.EmailMandado),
		ProgressID:              t.ProgressID,
		ProgressStatus:          t.ProgressStatus,
		LogID:                   t.LogID,
		HayAdjunto:              boolToUint8(t.HayAdjunto),
		Imprimido:               boolToUint8(t.Imprimido),
		Observacion:             t.Observacion,
		IVASerie:                t.IVASerie,
	}
}

// 转换函数，将TicketArticulo结构体转换为TicketArticuloResponse
func convertTicketArticuloToResponse(d model.TicketArticulo) TicketArticuloResponse {
	return TicketArticuloResponse{
		DocumentoNo:              d.DocumentoNo,
		ArticuloID:               d.ArticuloID,
		PropCount:                d.PropCount,
		PropID1:                  d.PropID1,
		PropID2:                  d.PropID2,
		Propiedad:                d.Propiedad,
		SerialNo:                 d.SerialNo,
		CodigoBarra:              d.CodigoBarra,
		CodigoBarraBalanza:       boolToUint8(d.CodigoBarraBalanza),
		NombreES:                 d.NombreES,
		NombreCN:                 d.NombreCN,
		Precio:                   d.Precio,
		Cantidad:                 d.Cantidad,
		Descuento:                d.Descuento,
		DescuentoCambioProhibido: boolToUint8(d.DescuentoCambioProhibido),
		PrecioCoste:              d.PrecioCoste,
		Comentario:               d.Comentario,
		NecesitaSerialNo:         boolToUint8(d.NecesitaSerialNo),
		Temporal:                 boolToUint8(d.Temporal),
		OrdenNo:                  d.OrdenNo,
		IVA:                      d.IVA,
		REQ:                      d.REQ,
	}
}

// 工具函数：将*time.Time转为毫秒时间戳指针
func timePtrToMillis(t *time.Time) *int64 {
	if t == nil {
		return nil
	}
	ms := t.UnixNano() / 1e6
	return &ms
}

// 工具函数：将time.Time转为毫秒时间戳指针
func timeToMillisPtr(t time.Time) *int64 {
	ms := t.UnixNano() / 1e6
	return &ms
}

// 工具函数：bool转uint8
func boolToUint8(b bool) uint8 {
	if b {
		return 1
	}
	return 0
}

// GetAllSuspendedTicketsHandler 获取所有挂单
func GetAllSuspendedTicketsHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取所有挂单请求 =====")

	detailedStr := c.DefaultQuery("detailed", "1")
	detailed := detailedStr == "1" || detailedStr == "true"

	log.Printf("[INFO] 获取所有挂单: detailed=%v", detailed)

	tickets := orderService.GetAllSuspendedTickets(detailed)

	log.Printf("[INFO] 成功获取 %d 个挂单", len(tickets))
	c.JSON(http.StatusOK, buildResponse("200", "获取成功", tickets))
}

// SubmitOrderHandler 提交订单
func SubmitOrderHandler(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理提交订单请求 =====")

	// 获取Authorization头
	token := c.GetHeader("Authorization")
	if token == "" {
		log.Printf("[ERROR] 缺少Authorization头")
		c.JSON(http.StatusUnauthorized, buildResponse("401", "缺少Authorization头", nil))
		return
	}

	// 从token中获取员工ID
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		log.Printf("[ERROR] 验证信息无效: %v", err)
		c.JSON(http.StatusForbidden, buildResponse("403", "验证信息无效", nil))
		return
	}

	// 解析请求体
	var data model.OrderDTO
	if err := c.ShouldBindJSON(&data); err != nil {
		log.Printf("[ERROR] 解析请求体失败: %v", err)
		c.JSON(http.StatusBadRequest, buildResponse("400", "请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 提交订单: EmpleadoID=%d, TicketNo=%d, CheckOut=%v",
		empleadoId, data.TicketNo, data.CheckOut)

	var result *int64

	if data.CheckOut {
		// 结账
		hasValidTicketNo := data.TicketNo > 0

		if hasValidTicketNo {
			// 结算已有的挂单
			result = orderService.CheckOutSuspendedTicket(empleadoId, data.TicketNo, data.Paid)
		} else {
			// 直接结算 - 与Java版本保持一致
			originalTotal := data.CalculateOriginalTotal()
			discountedTotal := data.CalculateDiscountedTotal(data.AdditionalDiscount)
			total := originalTotal - discountedTotal

			// 创建挂单数据
			colgando := model.NestedTicketColgando{
				TicketColgando: model.TicketColgando{
					Total:      total,
					LineaTotal: len(data.Items),
					Descuento:  data.AdditionalDiscount * 100, // 转换为百分比
					EmpresaID:  data.EmpresaID,
				},
			}

			// 转换商品详情
			for i, item := range data.Items {
				articulo := item.MapToColgandoArticulo(i)
				colgando.Details = append(colgando.Details, articulo)
			}

			result = orderService.CheckOutSuspendedTicketByData(empleadoId, colgando, data.Paid)
		}
	} else {
		// 挂单 - 与Java版本保持一致
		if num := orderService.SubmitSuspendedTicket(empleadoId, data); num != nil {
			result = num
		} else {
			result = nil
		}
	}

	if result != nil && *result > 0 {
		log.Printf("[INFO] 提交成功: Result=%d", *result)
		c.JSON(http.StatusOK, buildResponse("200", "提交成功", *result))
	} else {
		log.Printf("[ERROR] 提交失败")
		c.JSON(http.StatusBadRequest, buildResponse("400", "提交失败", nil))
	}
}
