package controller

import (
	"github.com/gin-gonic/gin"
	"net/http"
	"strconv"
	"supermarket-go/internal/model"
	"supermarket-go/internal/service"
	"supermarket-go/internal/util"
)

type ReturnGoodsController struct {
	Service *service.ReturnGoodsService
}

func RegisterReturnGoodsRoutes(r *gin.Engine, svc *service.ReturnGoodsService) {
	ctrl := &ReturnGoodsController{Service: svc}
	g := r.Group("/returnGoods")
	{
		g.GET("/listReturnedAlbarans", ctrl.ListReturnedAlbarans)
		g.GET("/listReturnedArticulos", ctrl.ListReturnedArticulos)
		g.POST("/submitReturnAlbaran", ctrl.SubmitReturnAlbaran)
		g.POST("/deleteReturnedAlbaran", ctrl.DeleteReturnedAlbaran)
		g.POST("/deleteReturnedArticulo", ctrl.DeleteReturnedArticulo)
		g.POST("/updateReturnedArticulo", ctrl.UpdateReturnedArticulo)
		g.POST("/bloqueado", ctrl.UpdateBloqueado)
		g.POST("/addReturnedArticulo", ctrl.AddReturnedArticulo)
	}
}

// ListReturnedAlbarans 列出所有退货的单据
func (ctrl *ReturnGoodsController) ListReturnedAlbarans(c *gin.Context) {
	code := c.Query("code")
	documentoNoStr := c.Query("documentoNo")
	bloqueado := c.Query("Bloqueado")

	var documentoNo *int64
	if documentoNoStr != "" {
		if val, err := strconv.ParseInt(documentoNoStr, 10, 64); err == nil {
			documentoNo = &val
		}
	}

	var bloqueadoPtr *string
	if bloqueado != "" && bloqueado != "-1" {
		bloqueadoPtr = &bloqueado
	}

	data := ctrl.Service.ListAllReturnedOrder(code, bloqueadoPtr, documentoNo)
	c.JSON(http.StatusOK, util.BuildSuccessResponseData(data))
}

// ListReturnedArticulos 列出所有退货单中的商品
func (ctrl *ReturnGoodsController) ListReturnedArticulos(c *gin.Context) {
	documentoNo := c.Query("documentoNo")
	if documentoNo == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数不能为空", nil))
		return
	}

	data := ctrl.Service.ListAllArticulosByDocumentoNo(documentoNo)
	c.JSON(http.StatusOK, util.BuildSuccessResponseData(data))
}

// SubmitReturnAlbaran 提交退货单数据
func (ctrl *ReturnGoodsController) SubmitReturnAlbaran(c *gin.Context) {
	token := c.GetHeader("Authorization")
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	var data model.ReturnGoodsListDTO
	if err := c.ShouldBindJSON(&data); err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("请求参数错误", err.Error()))
		return
	}

	if ctrl.Service.SubmitReturnGoods(empleadoId, data, false) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("提交成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("提交失败", ""))
	}
}

// DeleteReturnedAlbaran 删除退货单
func (ctrl *ReturnGoodsController) DeleteReturnedAlbaran(c *gin.Context) {
	token := c.GetHeader("Authorization")
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	documentoNoStr := c.PostForm("documentoNo")
	if documentoNoStr == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数不能为空", nil))
		return
	}

	documentoNo, err := strconv.ParseInt(documentoNoStr, 10, 64)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数格式错误", nil))
		return
	}

	if ctrl.Service.DeleteReturnedGoods(empleadoId, documentoNo) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("删除成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("删除失败", ""))
	}
}

// DeleteReturnedArticulo 删除退货单中的商品
func (ctrl *ReturnGoodsController) DeleteReturnedArticulo(c *gin.Context) {
	token := c.GetHeader("Authorization")
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	documentoNoStr := c.PostForm("documentoNo")
	code := c.PostForm("code")
	name := c.PostForm("nombre")
	if documentoNoStr == "" || code == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo与code不能为空", nil))
		return
	}

	documentoNo, err := strconv.ParseInt(documentoNoStr, 10, 64)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数格式错误", nil))
		return
	}

	if ctrl.Service.DeleteReturnedGood(empleadoId, documentoNo, code, name) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("删除成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("删除失败", ""))
	}
}

// UpdateReturnedArticulo 更新退货单中的商品
func (ctrl *ReturnGoodsController) UpdateReturnedArticulo(c *gin.Context) {
	token := c.GetHeader("Authorization")
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	documentoNoStr := c.PostForm("documentoNo")
	code := c.PostForm("code")
	amountStr := c.PostForm("amount")
	appendStr := c.DefaultPostForm("append", "0")
	precioStr := c.PostForm("precio")
	name := c.PostForm("nombre")

	if documentoNoStr == "" || amountStr == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo、amount不能为空", nil))
		return
	}

	documentoNo, err := strconv.ParseInt(documentoNoStr, 10, 64)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数格式错误", nil))
		return
	}
	amount, err := strconv.Atoi(amountStr)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("amount格式错误", nil))
		return
	}
	appendFlag := appendStr == "1" || appendStr == "true"

	var precioPtr *float64
	if precioStr != "" {
		if v, e := strconv.ParseFloat(precioStr, 64); e == nil {
			precioPtr = &v
		}
	}

	if ctrl.Service.UpdateReturnedGood(empleadoId, documentoNo, code, amount, appendFlag, precioPtr, name) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("修改成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("修改失败", ""))
	}
}

// UpdateBloqueado 更新退货单的锁定状态
func (ctrl *ReturnGoodsController) UpdateBloqueado(c *gin.Context) {
	token := c.GetHeader("Authorization")
	if _, err := util.GetEmpleadoIdFromToken(token); err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	documentoNoStr := c.PostForm("documentoNo")
	bloqueado := c.PostForm("bloqueado")
	if documentoNoStr == "" || bloqueado == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo与bloqueado不能为空", nil))
		return
	}

	documentoNo, err := strconv.ParseInt(documentoNoStr, 10, 64)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数格式错误", nil))
		return
	}

	if ctrl.Service.UpdateBloqueado(documentoNo, bloqueado) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("审核成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("审核失败", ""))
	}
}

// AddReturnedArticulo 向退货单添加商品
func (ctrl *ReturnGoodsController) AddReturnedArticulo(c *gin.Context) {
	token := c.GetHeader("Authorization")
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	documentoNoStr := c.Query("documentoNo")
	if documentoNoStr == "" {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数不能为空", nil))
		return
	}

	documentoNo, err := strconv.ParseInt(documentoNoStr, 10, 64)
	if err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("documentoNo参数格式错误", nil))
		return
	}

	var data model.ReturnGoodsListDTO
	if err := c.ShouldBindJSON(&data); err != nil {
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("请求参数错误", err.Error()))
		return
	}

	if ctrl.Service.AddReturnGoods(empleadoId, documentoNo, data) {
		c.JSON(http.StatusOK, util.BuildSuccessResponse("添加成功", nil))
	} else {
		c.JSON(http.StatusInternalServerError, util.BuildInternalServerErrorResponse("添加失败", ""))
	}
}
