package model

import (
	"fmt"
	"log"
	"os"
	"time"

	"gopkg.in/yaml.v2"
	"gorm.io/driver/mysql"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
)

type DBConfig struct {
	Host      string `yaml:"host"`
	Port      int    `yaml:"port"`
	User      string `yaml:"user"`
	Password  string `yaml:"password"`
	DBName    string `yaml:"dbname"`
	Charset   string `yaml:"charset"`
	ParseTime bool   `yaml:"parseTime"`
	Loc       string `yaml:"loc"`
}

type Config struct {
	Server struct {
		Port int `yaml:"port"`
	} `yaml:"server"`
	// DatabaseA 为现有数据库连接A（保持向后兼容，沿用原来的键名 database）
	Database DBConfig `yaml:"database"`
	// DatabaseB 为新增的目标库B
	DatabaseB DBConfig `yaml:"database_b"`
}

var DB *gorm.DB
var DBB *gorm.DB
var AppConfig Config

func InitDB() {
	log.Println("[INFO] 开始初始化数据库连接...")

	f, err := os.Open("config.yaml")
	if err != nil {
		log.Fatalf("[ERROR] 无法打开配置文件: %v", err)
	}
	defer f.Close()

	decoder := yaml.NewDecoder(f)
	if err := decoder.Decode(&AppConfig); err != nil {
		log.Fatalf("[ERROR] 解析配置文件失败: %v", err)
	}

	cfgA := AppConfig.Database
	log.Printf("[INFO] 数据库A配置: Host=%s, Port=%d, DBName=%s, User=%s",
		cfgA.Host, cfgA.Port, cfgA.DBName, cfgA.User)

	dsnA := fmt.Sprintf("%s:%s@tcp(%s:%d)/%s?charset=%s&parseTime=%t&loc=%s",
		cfgA.User, cfgA.Password, cfgA.Host, cfgA.Port, cfgA.DBName, cfgA.Charset, cfgA.ParseTime, cfgA.Loc)

	// 配置GORM日志
	gormLogger := logger.New(
		log.New(os.Stdout, "\r\n", log.LstdFlags), // io writer
		logger.Config{
			SlowThreshold:             time.Second, // 慢SQL阈值
			LogLevel:                  logger.Info, // 日志级别
			IgnoreRecordNotFoundError: true,        // 忽略记录未找到错误
			Colorful:                  true,        // 彩色打印
		},
	)

	DB, err = gorm.Open(mysql.Open(dsnA), &gorm.Config{
		Logger: gormLogger,
	})
	if err != nil {
		log.Fatalf("[ERROR] 数据库连接失败: %v", err)
	}

	// 获取底层sql.DB对象并设置连接池参数
	sqlDB, err := DB.DB()
	if err != nil {
		log.Fatalf("[ERROR] 获取数据库实例失败: %v", err)
	}

	// 设置连接池参数
	sqlDB.SetMaxIdleConns(10)
	sqlDB.SetMaxOpenConns(100)
	sqlDB.SetConnMaxLifetime(time.Hour)

	log.Println("[INFO] 数据库A连接成功，连接池已配置")
	log.Printf("[INFO] 数据库A连接池配置: MaxIdleConns=%d, MaxOpenConns=%d, ConnMaxLifetime=%v",
		10, 100, time.Hour)

	// 初始化数据库B
	cfgB := AppConfig.DatabaseB
	if (cfgB != DBConfig{} && cfgB.Host != "") {
		log.Printf("[INFO] 数据库B配置: Host=%s, Port=%d, DBName=%s, User=%s",
			cfgB.Host, cfgB.Port, cfgB.DBName, cfgB.User)

		dsnB := fmt.Sprintf("%s:%s@tcp(%s:%d)/%s?charset=%s&parseTime=%t&loc=%s",
			cfgB.User, cfgB.Password, cfgB.Host, cfgB.Port, cfgB.DBName, cfgB.Charset, cfgB.ParseTime, cfgB.Loc)

		DBB, err = gorm.Open(mysql.Open(dsnB), &gorm.Config{
			Logger: gormLogger,
		})
		if err != nil {
			log.Fatalf("[ERROR] 数据库B连接失败: %v", err)
		}

		sqlDBB, err := DBB.DB()
		if err != nil {
			log.Fatalf("[ERROR] 获取数据库B实例失败: %v", err)
		}
		sqlDBB.SetMaxIdleConns(10)
		sqlDBB.SetMaxOpenConns(100)
		sqlDBB.SetConnMaxLifetime(time.Hour)

		log.Println("[INFO] 数据库B连接成功，连接池已配置")
	} else {
		log.Println("[WARN] 未配置数据库B（database_b），将跳过数据库B的初始化")
	}
}
