package service

import (
	"fmt"
	"gorm.io/gorm"
	"log"
	"strconv"
	"supermarket-go/internal/model"
)

type AlbaranproveedorarticuloService struct {
	db              *gorm.DB
	articuloService model.ArticuloService
}

func NewAlbaranproveedorarticuloService(db *gorm.DB) *AlbaranproveedorarticuloService {
	return &AlbaranproveedorarticuloService{
		db:              db,
		articuloService: NewArticuloService(db),
	}
}

func (s *AlbaranproveedorarticuloService) QueryOne(code, articuloID string) *model.DyDzOrderInfo2Response {
	var art model.AlbaranProveedorArticulo

	// 构建查询条件，与Java代码保持一致
	query := s.db.Where("DocumentoNo = ?", code)

	// 支持通过ArticuloID或CodigoBarra查询
	query = query.Where("(ArticuloID = ? OR CodigoBarra = ?)", articuloID, articuloID)

	if err := query.First(&art).Error; err != nil {
		return nil
	}

	// 关联查询articulo信息，与Java代码保持一致
	articulo := s.articuloService.GetByArticuloID(art.ArticuloID)

	// 转换为响应结构
	response := &model.DyDzOrderInfo2Response{
		DocumentoNo:              art.DocumentoNo,
		ArticuloID:               art.ArticuloID,
		PropCount:                art.PropCount,
		PropID1:                  art.PropID1,
		PropID2:                  art.PropID2,
		Propiedad:                art.Propiedad,
		SerialNo:                 art.SerialNo,
		CodigoBarra:              art.CodigoBarra,
		CodigoBarraBalanza:       art.CodigoBarraBalanza,
		NombreES:                 art.NombreES,
		NombreCN:                 art.NombreCN,
		Precio:                   art.Precio,
		Cantidad:                 art.Cantidad,
		Descuento:                art.Descuento,
		DescuentoCambioProhibido: art.DescuentoCambioProhibido,
		PrecioCoste:              art.PrecioCoste,
		Comentario:               art.Comentario,
		NecesitaSerialNo:         art.NecesitaSerialNo,
		Temporal:                 art.Temporal,
		OrdenNo:                  art.OrdenNo,
		IVA:                      art.IVA,
		REQ:                      art.REQ,
	}

	// 转换articulo信息
	if articulo != nil {
		response.Articulo = s.convertArticuloToResponse(articulo)
	}

	return response
}

// convertArticuloToResponse 将Articulo转换为ArticuloResponse
func (s *AlbaranproveedorarticuloService) convertArticuloToResponse(articulo *model.Articulo) *model.ArticuloResponse {
	var fechaPrecioCoste *int64
	if articulo.FechaPrecioCoste != nil {
		timestamp := articulo.FechaPrecioCoste.UnixMilli()
		fechaPrecioCoste = &timestamp
	}

	var fechaCaducada *string
	if articulo.FechaCaducada != nil {
		dateStr := articulo.FechaCaducada.Format("2006-01-02")
		fechaCaducada = &dateStr
	}

	// 查询库存信息
	var stockDA *string
	var stock model.Stock
	if err := s.db.Where("ArticuloID = ?", articulo.ArticuloID).First(&stock).Error; err == nil {
		stockStr := fmt.Sprintf("%.2f", stock.Stock)
		stockDA = &stockStr
	}

	// 查询供应商信息
	var proveedorDD *string
	var proveedor model.Proveedor
	if err := s.db.Where("EmpresaID = ?", articulo.EmpresaID).First(&proveedor).Error; err == nil {
		proveedorDD = &proveedor.NombreES
	}

	return &model.ArticuloResponse{
		ArticuloID:               articulo.ArticuloID,
		EmpresaID:                strconv.FormatInt(articulo.EmpresaID, 10),
		StockDA:                  stockDA,
		CodigoBarra:              articulo.CodigoBarra,
		Descripcion:              articulo.Descripcion,
		PrecioDetalle:            articulo.PrecioDetalle,
		PrecioDomicilio:          articulo.PrecioDomicilio,
		PrecioSocio:              articulo.PrecioSocio,
		PrecioMayor:              articulo.PrecioMayor,
		PrecioFactura:            articulo.PrecioFactura,
		PrecioInternet:           articulo.PrecioInternet,
		PrecioAmigo:              articulo.PrecioAmigo,
		PrecioEspecial:           articulo.PrecioEspecial,
		PrecioOferta:             articulo.PrecioOferta,
		PrecioCoste:              articulo.PrecioCoste,
		NombreES:                 articulo.NombreES,
		PrimeraLetra:             nil, // 需要计算
		NombreCN:                 articulo.NombreCN,
		Pinyin:                   nil, // 需要计算
		CantidadPorUnidad:        articulo.CantidadPorUnidad,
		CantidadPorUnidad2:       articulo.CantidadPorUnidad2,
		VolumenPeso:              articulo.VolumenPeso,
		ClaseID:                  int(articulo.ClaseID),
		UnidadNombre:             articulo.UnidadNombre,
		FechaPrecioCoste:         fechaPrecioCoste,
		Descuento:                articulo.Descuento,
		ValeDescuento:            nil, // 需要计算
		FacturaPorcentaje:        articulo.FacturaPorcentaje,
		FechaCaducada:            fechaCaducada,
		Oferta:                   int(articulo.Oferta),
		ProveedorDD:              proveedorDD,
		Privado:                  int(articulo.Privado),
		UsarPrecioPorCantidad:    int(articulo.UsarPrecioPorCantidad),
		UsarDescuentoPorCantidad: int(articulo.UsarDescuentoPorCantidad),
		Precio1:                  articulo.Precio1,
		Descuento1:               articulo.Descuento1,
		Cantidad1:                articulo.Cantidad1,
		Precio2:                  articulo.Precio2,
		Descuento2:               articulo.Descuento2,
		Cantidad2:                articulo.Cantidad2,
		Precio3:                  articulo.Precio3,
		Descuento3:               articulo.Descuento3,
		Cantidad3:                articulo.Cantidad3,
		Precio4:                  articulo.Precio4,
		Descuento4:               articulo.Descuento4,
		Cantidad4:                articulo.Cantidad4,
		Precio5:                  articulo.Precio5,
		Descuento5:               articulo.Descuento5,
		Cantidad5:                articulo.Cantidad5,
		Precio6:                  articulo.Precio6,
		Descuento6:               articulo.Descuento6,
		SitioEnAlmacen:           articulo.SitioEnAlmacen,
		MiniStock:                articulo.MiniStock,
		MaxiStock:                articulo.MaxiStock,
		DibujoMD5:                nil, // 需要计算
		TiempoDibujoModificado:   nil, // 数据库表中没有这个字段
		DescuentoCambioProhibido: int(articulo.DescuentoCambioProhibido),
		UnidadUsarRegla:          int(articulo.UnidadUsarRegla),
		MultiCodigo:              articulo.MultiCodigo,
		ConsultaParcial:          nil, // 需要从articuloextra表查询
		FechaEntrada: func() *string {
			//if articulo.FechaEntrada != nil {
			//	dateStr := articulo.FechaEntrada.Format("2006-01-02")
			//	return &dateStr
			//}
			return nil
		}(),
		Bloqueado: int(articulo.Bloqueado),
		//SignoEspecial: articulo.SignoEspecial, // 直接使用，因为类型匹配
		Observacion: func() string {
			if articulo.Observacion != nil {
				return *articulo.Observacion
			}
			return ""
		}(),
		Dibujo:   nil, // Dibujo字段在单独的dibujo表中
		DibujoID: int(articulo.DibujoID),
	}
}

func (s *AlbaranproveedorarticuloService) QueryList(code, articuloID, bloqueado string) []model.DyDzOrderInfoResponse {
	var arts []model.AlbaranProveedorArticulo

	query := s.db.Model(&model.AlbaranProveedorArticulo{})

	if code != "" {
		query = query.Where("DocumentoNo = ?", code)
	}

	if articuloID != "" {
		query = query.Where("ArticuloID = ? OR CodigoBarra = ?", articuloID, articuloID)
	}

	if bloqueado != "" && bloqueado == "-1" {
		query = query.Where("(COALESCE(Comentario, 0) - Cantidad) < 0")
	}

	if err := query.Find(&arts).Error; err != nil {
		log.Printf("QueryList err:%v", err.Error())
		return nil
	}

	// 转换为响应格式
	var responses []model.DyDzOrderInfoResponse
	for _, art := range arts {
		// 关联查询articulo信息
		articulo := s.articuloService.GetByArticuloID(art.ArticuloID)

		response := model.DyDzOrderInfoResponse{
			DocumentoNo:              art.DocumentoNo,
			ArticuloID:               art.ArticuloID,
			PropCount:                art.PropCount,
			PropID1:                  art.PropID1,
			PropID2:                  art.PropID2,
			Propiedad:                art.Propiedad,
			SerialNo:                 art.SerialNo,
			CodigoBarra:              art.CodigoBarra,
			CodigoBarraBalanza:       art.CodigoBarraBalanza,
			NombreES:                 art.NombreES,
			NombreCN:                 art.NombreCN,
			Precio:                   art.Precio,
			Cantidad:                 art.Cantidad,
			Descuento:                art.Descuento,
			DescuentoCambioProhibido: art.DescuentoCambioProhibido,
			PrecioCoste:              art.PrecioCoste,
			Comentario:               art.Comentario,
			NecesitaSerialNo:         art.NecesitaSerialNo,
			Temporal:                 art.Temporal,
			OrdenNo:                  art.OrdenNo,
			IVA:                      art.IVA,
			REQ:                      art.REQ,
		}

		// 转换articulo信息
		if articulo != nil {
			response.Articulo = s.convertArticuloToResponse(articulo)
		}

		responses = append(responses, response)
	}

	return responses
}

func (s *AlbaranproveedorarticuloService) UpdateOne(articuloID, precio, precioDetalle, comentario, comentarioMax string) int {
	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	counter := 0

	// 更新albaranproveedorarticulo表
	if precio != "" || comentario != "" || comentarioMax != "" {
		updates := make(map[string]interface{})
		if precio != "" {
			updates["Precio"] = precio
		}
		if comentario != "" {
			// 先根据 ArticuloID 查原始 Comentario得值，然后将查询后得值加comentario后赋值给 updates["Comentario"] 即可
			var art model.AlbaranProveedorArticulo
			if err := tx.Where("ArticuloID = ?", articuloID).First(&art).Error; err != nil {
				tx.Rollback()
				return 0
			}

			// 解析原始 Comentario 值
			originalComentario := 0.0
			if art.Comentario != "" {
				var err error
				originalComentario, err = strconv.ParseFloat(art.Comentario, 64)
				if err != nil {
					originalComentario = 0.0
				}
			}

			// 解析新的 comentario 值
			newComentario, err := strconv.ParseFloat(comentario, 64)
			if err != nil {
				tx.Rollback()
				return 0
			}

			// 累加并赋值
			sumComentario := originalComentario + newComentario
			updates["Comentario"] = fmt.Sprintf("%.2f", sumComentario)
		}
		if comentarioMax != "" {
			updates["Comentario"] = comentarioMax
		}
		result := tx.Model(&model.AlbaranProveedorArticulo{}).
			Where("ArticuloID = ?", articuloID).
			Updates(updates)
		if result.Error != nil {
			tx.Rollback()
			return 0
		}
		counter += int(result.RowsAffected)
	}

	// 更新articulo表
	if precioDetalle != "" {
		result := tx.Model(&model.Articulo{}).
			Where("ArticuloID = ?", articuloID).Or("CodigoBarra = ?", articuloID).
			Update("PrecioDetalle", precioDetalle)
		if result.Error != nil {
			tx.Rollback()
			return 0
		}
		counter += int(result.RowsAffected)
	}

	if err := tx.Commit().Error; err != nil {
		return 0
	}

	return counter
}

func (s *AlbaranproveedorarticuloService) DyDzOrderStatus(code, bloqueado string) int {
	// 查询所有相关的albaranproveedorarticulo记录
	var arts []model.AlbaranProveedorArticulo
	s.db.Where("DocumentoNo = ?", code).Find(&arts)

	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	counter := 0

	// 根据bloqueado参数决定操作
	if bloqueado == "1" {
		bloqueado = "0"
	} else {
		bloqueado = "1"
		// 更新articulo表的PrecioCoste
		for _, art := range arts {
			if err := tx.Model(&model.Articulo{}).
				Where("ArticuloID = ?", art.ArticuloID).
				Update("PrecioCoste", art.Precio).Error; err != nil {
				tx.Rollback()
				return 0
			}
			counter++
		}
	}

	// 更新albaranproveedor表的Bloqueado状态
	if err := tx.Model(&model.AlbaranProveedor{}).
		Where("DocumentoNo = ?", code).
		Update("Bloqueado", bloqueado).Error; err != nil {
		tx.Rollback()
		return 0
	}
	counter++

	// 更新VerificadorID
	if err := tx.Model(&model.AlbaranProveedor{}).
		Where("DocumentoNo = ?", code).
		Update("VerificadorID", 8).Error; err != nil {
		tx.Rollback()
		return 0
	}
	counter++

	if err := tx.Commit().Error; err != nil {
		return 0
	}

	return counter
}

func (s *AlbaranproveedorarticuloService) DyDzOrderJiaodui(code string) int {
	// 查询所有相关的albaranproveedorarticulo记录
	var arts []model.AlbaranProveedorArticulo
	s.db.Where("DocumentoNo = ?", code).Find(&arts)

	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	var totalDifference float64 = 0

	for _, art := range arts {
		// 计算差异
		comentario, _ := strconv.ParseFloat(art.Comentario, 64)
		cantidad, _ := strconv.ParseFloat(fmt.Sprintf("%v", art.Cantidad), 64)
		precio, _ := strconv.ParseFloat(fmt.Sprintf("%v", art.Precio), 64)

		difference := comentario - cantidad
		priceDifference := difference * precio
		totalDifference += priceDifference

		// 更新albaranproveedorarticulo表的Cantidad
		if err := tx.Model(&model.AlbaranProveedorArticulo{}).
			Where("DocumentoNo = ?", code).
			Update("Cantidad", art.Comentario).Error; err != nil {
			tx.Rollback()
			return 0
		}

		// 更新stock表的Stock
		if err := tx.Model(&model.Stock{}).
			Where("ArticuloID = ?", art.ArticuloID).
			Update("Stock", gorm.Expr("stock + ?", difference)).Error; err != nil {
			tx.Rollback()
			return 0
		}
	}

	// 更新albaranproveedor表的Total
	if err := tx.Model(&model.AlbaranProveedor{}).
		Where("DocumentoNo = ?", code).
		Update("total", gorm.Expr("total + ?", totalDifference)).Error; err != nil {
		tx.Rollback()
		return 0
	}

	if err := tx.Commit().Error; err != nil {
		return 0
	}

	return 0 // 按照Java实现返回0
}
