package service

import (
	"encoding/base64"
	"fmt"
	"gorm.io/gorm"
	"log"
	"os"
	"path/filepath"
	"strconv"
	"supermarket-go/internal/model"
)

type ArticuloServiceImpl struct {
	db *gorm.DB
}

// 辅助函数：解析字符串到 float64
func parseFloat64(s string) float64 {
	if s == "" {
		return 0
	}
	val, err := strconv.ParseFloat(s, 64)
	if err != nil {
		return 0
	}
	return val
}

// 辅助函数：解析字符串到 uint16
func parseUint16(s string) uint16 {
	if s == "" {
		return 0
	}
	val, err := strconv.ParseUint(s, 10, 16)
	if err != nil {
		return 0
	}
	return uint16(val)
}

// 辅助函数：解析字符串到 int64
func parseInt64(s string) int64 {
	if s == "" {
		return 0
	}
	val, err := strconv.ParseInt(s, 10, 64)
	if err != nil {
		return 0
	}
	return val
}

func NewArticuloService(db *gorm.DB) model.ArticuloService {
	return &ArticuloServiceImpl{db: db}
}

func (s *ArticuloServiceImpl) Scan(request model.ArticuloScanRequest) *model.ArticuloScanResult {
	log.Printf("[INFO] 开始扫描商品: ArticuloID=%s, CodigoBarra=%s",
		request.ArticuloID, request.CodigoBarra)

	var result model.ArticuloScanResult

	// 首先尝试通过ArticuloID查询
	if request.ArticuloID != "" {
		log.Printf("[INFO] 尝试通过ArticuloID查询: %s", request.ArticuloID)
		sql := `
			SELECT a.*, b.nombreES AS proveedordd, b.empresaid AS proveedorid, c.stock AS stockda
			FROM articulo a
			LEFT JOIN proveedor b ON a.EmpresaID = b.EmpresaID
			LEFT JOIN stock c ON a.articuloid = c.articuloid
			WHERE a.articuloid = ? OR a.codigobarra = ?
			LIMIT 1
		`
		if err := s.db.Raw(sql, request.ArticuloID, request.ArticuloID).Scan(&result).Error; err == nil && result.ArticuloID != "" {
			log.Printf("[INFO] 通过ArticuloID成功找到商品: %s", result.ArticuloID)
			filePath := fmt.Sprintf("image/%s.jpg", result.ArticuloID)
			result.Dibujo = ImageToBase64Bytes(filePath)
			return &result
		} else if err != nil {
			log.Printf("[ERROR] 通过ArticuloID查询失败: %v", err)
		} else {
			log.Printf("[WARN] 通过ArticuloID未找到商品: %s", request.ArticuloID)
		}
	}

	// 如果ArticuloID查询失败，尝试通过CodigoBarra查询
	if request.CodigoBarra != "" {
		log.Printf("[INFO] 尝试通过CodigoBarra查询: %s", request.CodigoBarra)
		sql2 := `
			SELECT a.*, b.nombreES AS proveedordd, b.empresaid AS proveedorid, c.stock AS stockda
			FROM articulo a
			LEFT JOIN proveedor b ON a.EmpresaID = b.EmpresaID
			LEFT JOIN stock c ON a.articuloid = c.articuloid
			WHERE a.codigobarra = ?
			LIMIT 1
		`
		if err := s.db.Raw(sql2, request.CodigoBarra).Scan(&result).Error; err == nil && result.ArticuloID != "" {
			log.Printf("[INFO] 通过CodigoBarra成功找到商品: %s", result.ArticuloID)
			filePath := fmt.Sprintf("image/%s.jpg", result.ArticuloID)
			result.Dibujo = ImageToBase64Bytes(filePath)
			return &result
		} else if err != nil {
			log.Printf("[ERROR] 通过CodigoBarra查询失败: %v", err)
		} else {
			log.Printf("[WARN] 通过CodigoBarra未找到商品: %s", request.CodigoBarra)
		}
	}

	log.Printf("[WARN] 未找到商品: ArticuloID=%s, CodigoBarra=%s",
		request.ArticuloID, request.CodigoBarra)
	return nil
}

func (s *ArticuloServiceImpl) Update(request model.ArticuloInputRequest) interface{} {
	log.Printf("[INFO] 开始更新商品: ArticuloID=%s", request.ArticuloID)

	// 更新价格
	if len(request.ArticuloID) > 0 && len(request.PrecioDetalle) > 0 {
		log.Printf("[INFO] 更新商品价格: CodigoBarra=%s, PrecioDetalle=%s",
			request.ArticuloID, request.PrecioDetalle)
		err := s.db.Exec("UPDATE articulo SET preciodetalle = ? WHERE ArticuloID = ?",
			request.PrecioDetalle, request.ArticuloID).Error
		if err != nil {
			log.Printf("[ERROR] 更新商品价格失败: %v", err)
			return err.Error()
		}
		log.Printf("[INFO] 商品价格更新成功")
	}

	// 更新库存
	if request.StockDA > 0 {
		log.Printf("[INFO] 更新商品库存: ArticuloID=%s, StockDA=%d",
			request.ArticuloID, request.StockDA)

		var count int64
		err := s.db.Model(&model.Stock{}).
			Where("ArticuloId = ?", request.ArticuloID).
			Count(&count).Error
		if err != nil {
			log.Printf("[ERROR] 查询库存记录失败: %v", err)
			return err.Error()
		}

		if count > 0 {
			log.Printf("[INFO] 更新现有库存记录")
			err = s.db.Model(&model.Stock{}).
				Where("ArticuloId = ?", request.ArticuloID).
				Update("stock", request.StockDA).Error
			if err != nil {
				log.Printf("[ERROR] 更新库存失败: %v", err)
				return err.Error()
			}
		} else {
			log.Printf("[INFO] 创建新的库存记录")
			err = s.db.Table("stock").Create(map[string]interface{}{
				"ArticuloID": request.ArticuloID,
				"stock":      request.StockDA,
			}).Error
			if err != nil {
				log.Printf("[ERROR] 创建库存记录失败: %v", err)
				return err.Error()
			}
		}
		log.Printf("[INFO] 库存更新成功")
	}

	// 更新图片
	//if len(request.Dibujo) > 0 {
	//	log.Printf("[INFO] 更新商品图片: ArticuloID=%s, 图片长度=%d",
	//		request.ArticuloID, len(request.Dibujo))
	//
	//	var dibujos []model.ArticuloDibujo
	//	err := s.db.Table("articulodibujo").
	//		Where("ArticuloId = ?", request.ArticuloID).
	//		Order("OrdenNo DESC").
	//		Find(&dibujos).Error
	//	if err != nil {
	//		log.Printf("[ERROR] 查询商品图片记录失败: %v", err)
	//		return err.Error()
	//	}
	//
	//	if len(dibujos) > 0 {
	//		dibujo := dibujos[0]
	//		log.Printf("[INFO] 更新现有图片: DibujoID=%d", dibujo.DibujoID)
	//		err = s.db.Table("dibujo").
	//			Where("DibujoID = ?", dibujo.DibujoID).
	//			Update("Dibujo", []byte(request.Dibujo)).Error
	//		if err != nil {
	//			log.Printf("[ERROR] 更新图片失败: %v", err)
	//			return err.Error()
	//		}
	//	} else {
	//		log.Printf("[INFO] 创建新的图片记录")
	//		var curTotal int64
	//		err = s.db.Table("dibujo").Count(&curTotal).Error
	//		if err != nil {
	//			log.Printf("[ERROR] 查询图片总数失败: %v", err)
	//			return err.Error()
	//		}
	//		curTotal = curTotal + 1
	//		dibujo := model.ArticuloDibujo{
	//			ArticuloID: request.ArticuloID,
	//			DibujoMD5:  uuid.NewString(),
	//			DibujoID:   curTotal,
	//			OrdenNo:    1,
	//		}
	//		err = s.db.Create(&dibujo).Error
	//		if err != nil {
	//			log.Printf("[ERROR] 创建图片关联记录失败: %v", err)
	//			return err.Error()
	//		}
	//		err = s.db.Table("dibujo").Create(map[string]interface{}{
	//			"DibujoID": dibujo.DibujoID,
	//			"Dibujo":   []byte(request.Dibujo),
	//		}).Error
	//		if err != nil {
	//			log.Printf("[ERROR] 创建图片记录失败: %v", err)
	//			return err.Error()
	//		}
	//		request.DibujoID = fmt.Sprintf("%d", curTotal)
	//	}
	//	log.Printf("[INFO] 图片更新成功")
	//}
	// 判断图片是否为空 不为空下载
	if len(request.Dibujo) > 0 {
		filePath := fmt.Sprintf("image/%s.jpg", request.ArticuloID)
		saveBase64Image([]byte(request.Dibujo), filePath)
	}
	// 更新商品基本信息
	log.Printf("[INFO] 更新商品基本信息: ArticuloID=%s", request.ArticuloID)
	dibujoID, _ := strconv.Atoi(request.DibujoID) // 字符串转int
	updates := map[string]interface{}{}

	if request.CodigoBarra != "" {
		updates["CodigoBarra"] = request.CodigoBarra
	}
	if request.NombreES != "" {
		updates["NombreES"] = request.NombreES
	}
	if request.PrecioCoste != "" {
		updates["PrecioCoste"] = parseFloat64(request.PrecioCoste)
	}
	if request.PrecioDetalle != "" {
		updates["PrecioDetalle"] = parseFloat64(request.PrecioDetalle)
	}
	if request.PrecioMayor != "" {
		updates["PrecioMayor"] = parseFloat64(request.PrecioMayor)
	}
	if request.ClaseID != "" {
		updates["ClaseID"] = parseUint16(request.ClaseID)
	}
	if request.EmpresaID != "" {
		updates["EmpresaID"] = parseInt64(request.EmpresaID)
	}
	if request.Observacion != "" {
		updates["Observacion"] = request.Observacion
	}
	// if request.Descuento != 0 {
	// 	updates["Descuento"] = request.Descuento
	// }
	updates["Descuento"] = request.Descuento
	if len(request.DibujoID) != 0 {
		updates["DibujoID"] = dibujoID
	}

	// 必须指定更新哪条记录，可以用主键
	err := s.db.Model(&model.Articulo{}).Where("ArticuloID = ?", request.ArticuloID).Updates(updates).Error
	if err != nil {
		log.Printf("[ERROR] 更新商品信息失败: %v", err)
		return err.Error()
	}

	log.Printf("[INFO] 商品更新完成: ArticuloID=%s", request.ArticuloID)
	return nil
}

func saveBase64Image(data []byte, filename string) error {
	// 解码 []byte base64 数据
	decoded, err := base64.StdEncoding.DecodeString(string(data))
	if err != nil {
		return fmt.Errorf("解码失败: %w", err)
	}

	// 保存文件
	err = os.WriteFile(filename, decoded, 0644)
	if err != nil {
		return fmt.Errorf("写入文件失败: %w", err)
	}
	return nil
}

func (s *ArticuloServiceImpl) Edit(ac model.Articulo) interface{} {
	log.Printf("[INFO] 编辑商品: ArticuloID=%s", ac.ArticuloID)
	//return s.Update(ac)
	return nil
}

func (s *ArticuloServiceImpl) Add(request model.ArticuloInputRequest) interface{} {
	log.Printf("[INFO] 开始添加商品: ArticuloID=%s", request.ArticuloID)

	// 处理库存
	if request.StockDA > 0 {
		log.Printf("[INFO] 处理商品库存: ArticuloID=%s, StockDA=%d",
			request.ArticuloID, request.StockDA)

		var count int64
		err := s.db.Model(&model.Stock{}).
			Where("ArticuloId = ?", request.ArticuloID).
			Count(&count).Error
		if err != nil {
			log.Printf("[ERROR] 查询库存记录失败: %v", err)
			return err.Error()
		}

		if count > 0 {
			log.Printf("[INFO] 更新现有库存记录")
			err = s.db.Model(&model.Stock{}).
				Where("ArticuloId = ?", request.ArticuloID).
				Update("stock", request.StockDA).Error
			if err != nil {
				log.Printf("[ERROR] 更新库存失败: %v", err)
				return err.Error()
			}
		} else {
			log.Printf("[INFO] 创建新的库存记录")
			err = s.db.Table("stock").Create(map[string]interface{}{
				"ArticuloID": request.ArticuloID,
				"stock":      request.StockDA,
			}).Error
			if err != nil {
				log.Printf("[ERROR] 创建库存记录失败: %v", err)
				return err.Error()
			}
		}
	}
	//查询商品表articulo是否有数据 有数据则更新  没有则插入
	var existing model.Articulo
	s.db.Where("ArticuloID = ? OR CodigoBarra = ?", request.ArticuloID, request.CodigoBarra).First(&existing)
	if existing.ArticuloID == "" {
		// 插入
		log.Printf("[INFO] 未找到商品，执行插入: ArticuloID=%s", request.ArticuloID)
		var observacionPtr *string
		if request.Observacion != "" {
			observacionPtr = &request.Observacion
		}
		dibujoIDInt, _ := strconv.Atoi(request.DibujoID)
		claseIDUint := parseUint16(request.ClaseID)
		articulo := model.Articulo{
			ArticuloID:    request.ArticuloID,
			EmpresaID:     parseInt64(request.EmpresaID),
			CodigoBarra:   request.CodigoBarra,
			NombreES:      request.NombreES,
			NombreCN:      request.NombreCN,
			PrecioCoste:   parseFloat64(request.PrecioCoste),
			PrecioDetalle: parseFloat64(request.PrecioDetalle),
			PrecioMayor:   parseFloat64(request.PrecioMayor),
			ClaseID:       int(claseIDUint),
			Descuento:     float64(request.Descuento),
			Observacion:   observacionPtr,
			DibujoID:      int64(dibujoIDInt),
		}
		if err := s.db.Create(&articulo).Error; err != nil {
			log.Printf("[ERROR] 插入商品失败: %v", err)
			return err.Error()
		}
		log.Printf("[INFO] 插入商品成功: ArticuloID=%s", request.ArticuloID)
	} else {
		// 更新
		log.Printf("[INFO] 已存在商品，执行更新: ArticuloID=%s", existing.ArticuloID)
		dibujoID, _ := strconv.Atoi(request.DibujoID)
		updates := map[string]interface{}{}
		if request.CodigoBarra != "" {
			updates["CodigoBarra"] = request.CodigoBarra
		}
		if request.NombreES != "" {
			updates["NombreES"] = request.NombreES
		}
		if request.NombreCN != "" {
			updates["NombreCN"] = request.NombreCN
		}
		if request.PrecioCoste != "" {
			updates["PrecioCoste"] = parseFloat64(request.PrecioCoste)
		}
		if request.PrecioDetalle != "" {
			updates["PrecioDetalle"] = parseFloat64(request.PrecioDetalle)
		}
		if request.PrecioMayor != "" {
			updates["PrecioMayor"] = parseFloat64(request.PrecioMayor)
		}
		if request.ClaseID != "" {
			updates["ClaseID"] = int(parseUint16(request.ClaseID))
		}
		if request.EmpresaID != "" {
			updates["EmpresaID"] = parseInt64(request.EmpresaID)
		}
		if request.Observacion != "" {
			// 字段为可空，保持与模型类型一致
			obs := request.Observacion
			updates["Observacion"] = &obs
		}
		//if request.Descuento != 0 {
		//	updates["Descuento"] = float64(request.Descuento)
		//}
		updates["Descuento"] = float64(request.Descuento)
		if request.DibujoID != "" {
			updates["DibujoID"] = dibujoID
		}
		if len(updates) > 0 {
			if err := s.db.Model(&model.Articulo{}).Where("ArticuloID = ?", existing.ArticuloID).Updates(updates).Error; err != nil {
				log.Printf("[ERROR] 更新商品失败: %v", err)
				return err.Error()
			}
		}
		log.Printf("[INFO] 商品添加成功: ArticuloID=%s", existing.ArticuloID)
	}
	if len(request.Dibujo) > 0 {
		filePath := fmt.Sprintf("image/%s.jpg", request.ArticuloID)
		saveBase64Image([]byte(request.Dibujo), filePath)
	}
	log.Printf("[INFO] 商品添加完成: ArticuloID=%s", request.ArticuloID)
	return nil
}

func (s *ArticuloServiceImpl) GetAcClass() []model.ArticuloClase {
	log.Println("[INFO] 查询所有商品类别")
	var classes []model.ArticuloClase
	err := s.db.Find(&classes).Error
	if err != nil {
		log.Printf("[ERROR] 查询商品类别失败: %v", err)
		return []model.ArticuloClase{}
	}
	log.Printf("[INFO] 查询到 %d 个商品类别", len(classes))
	return classes
}

func (s *ArticuloServiceImpl) GetProveedor() []model.Proveedor {
	log.Println("[INFO] 查询所有供应商")
	var proveedores []model.Proveedor
	err := s.db.Find(&proveedores).Error
	if err != nil {
		log.Printf("[ERROR] 查询供应商失败: %v", err)
		return []model.Proveedor{}
	}
	log.Printf("[INFO] 查询到 %d 个供应商", len(proveedores))
	return proveedores
}

func (s *ArticuloServiceImpl) UpdateByArticuloIDOrCodigoBarra(articulo model.Articulo) int {
	log.Printf("[INFO] 按商品ID或条码更新商品: ArticuloID=%s", articulo.ArticuloID)

	updates := map[string]interface{}{
		//"usar_precio_por_cantidad": usarPrecioPorCantidad,
		//"precio_detalle":           preciodetalle,
		//"precio1":                  precio1, "descuento1": descuento1, "cantidad1": cantidad1,
		//"precio2": precio2, "descuento2": descuento2, "cantidad2": cantidad2,
		//"precio3": precio3, "descuento3": descuento3, "cantidad3": cantidad3,
		//"precio4": precio4, "descuento4": descuento4, "cantidad4": cantidad4,
		//"precio5": precio5, "descuento5": descuento5, "cantidad5": cantidad5,
		//"precio6": precio6,
	}

	res := s.db.Model(&model.Articulo{}).
		//Where("articulo_id LIKE ? OR codigo_barra LIKE ?", "%"+articuloID+"%", "%"+articuloID+"%").
		Updates(updates)

	log.Printf("[INFO] 更新完成，影响行数: %d", res.RowsAffected)
	return int(res.RowsAffected)
}

func (s *ArticuloServiceImpl) GetByArticuloIDOrCodigoBarraLike(articuloID string) []model.Articulo {
	log.Printf("[INFO] 模糊查询商品: ArticuloID=%s", articuloID)
	var articulos []model.Articulo
	err := s.db.Where("ArticuloID LIKE ? OR CodigoBarra LIKE ?",
		"%"+articuloID+"%", "%"+articuloID+"%").Find(&articulos).Error
	if err != nil {
		log.Printf("[ERROR] 模糊查询商品失败: %v", err)
		return []model.Articulo{}
	}
	log.Printf("[INFO] 模糊查询到 %d 个商品", len(articulos))
	return articulos
}

func (s *ArticuloServiceImpl) GetByArticuloIDOrCodigoBarra(articuloID string) []model.Articulo {
	log.Printf("[INFO] 查询商品: ArticuloID=%s", articuloID)
	var articulos []model.Articulo
	err := s.db.Where("ArticuloID = ? OR CodigoBarra = ?",
		articuloID, articuloID).Find(&articulos).Error
	if err != nil {
		log.Printf("[ERROR] 查询商品失败: %v", err)
		return []model.Articulo{}
	}
	log.Printf("[INFO] 查询到 %d 个商品", len(articulos))
	return articulos
}

func (s *ArticuloServiceImpl) GetByArticuloIDOrCodigoBarraActually(articuloID string) []model.Articulo {
	log.Printf("[INFO] 精确查询商品: ArticuloID=%s", articuloID)
	var articulos []model.Articulo
	err := s.db.Where("ArticuloID = ? OR CodigoBarra = ?", articuloID, articuloID).Find(&articulos).Error
	if err != nil {
		log.Printf("[ERROR] 精确查询商品失败: %v", err)
		return []model.Articulo{}
	}
	log.Printf("[INFO] 精确查询到 %d 个商品", len(articulos))
	return articulos
}

func (s *ArticuloServiceImpl) GetByArticuloID(articuloID string) *model.Articulo {
	log.Printf("[INFO] 根据ArticuloID查询商品: ArticuloID=%s", articuloID)
	var articulo model.Articulo
	err := s.db.Where("ArticuloID = ?", articuloID).First(&articulo).Error
	if err != nil {
		log.Printf("[ERROR] 根据ArticuloID查询商品失败: %v", err)
		return nil
	}
	log.Printf("[INFO] 查询到商品: ArticuloID=%s, NombreES=%s", articulo.ArticuloID, articulo.NombreES)
	return &articulo
}

func (s *ArticuloServiceImpl) GetByProductName(productName string) []model.Articulo {
	log.Printf("[INFO] 根据商品名称查询商品: ProductName=%s", productName)
	var articulos []model.Articulo
	err := s.db.Where("NombreES LIKE ? OR NombreCN LIKE ?", "%"+productName+"%", "%"+productName+"%").Find(&articulos).Error
	if err != nil {
		log.Printf("[ERROR] 根据商品名称查询商品失败: %v", err)
		return []model.Articulo{}
	}
	log.Printf("[INFO] 根据商品名称查询到 %d 个商品", len(articulos))
	return articulos
}

func (s *ArticuloServiceImpl) Zbm(param model.ZbmParam) interface{} {
	log.Printf("[INFO] 生成自编码: CodePrefix=%s, CodeLength=%d",
		param.CodePrefix, param.CodeLength)

	var count int64
	err := s.db.Model(&model.Articulo{}).
		Where("articulo_id LIKE ? AND LENGTH(articulo_id) = ?",
			param.CodePrefix+"%", param.CodeLength).
		Count(&count).Error

	if err != nil {
		log.Printf("[ERROR] 查询自编码计数失败: %v", err)
		return ""
	}

	nextCode := fmt.Sprintf("%s%0*d", param.CodePrefix,
		param.CodeLength-len(param.CodePrefix), count+1)

	log.Printf("[INFO] 生成自编码成功: %s (当前计数: %d)", nextCode, count+1)
	return nextCode
}

// UpdateDibujoID 更新商品的图片ID
func (s *ArticuloServiceImpl) UpdateDibujoID(articuloID string, dibujoID int64) error {
	return s.db.Model(&model.Articulo{}).Where("ArticuloID = ?", articuloID).Update("DibujoID", dibujoID).Error
}

func ImageToBase64Bytes(path string) []byte {
	// 读取文件
	data, _ := os.ReadFile(path)

	// 编码成 base64 并返回 []byte
	encoded := make([]byte, base64.StdEncoding.EncodedLen(len(data)))
	base64.StdEncoding.Encode(encoded, data)

	return encoded
}

// UploadImage 处理图片上传
func (s *ArticuloServiceImpl) UploadImage(articuloID, dibujo string) error {
	log.Printf("[INFO] 开始处理图片上传: ArticuloID=%s", articuloID)

	// 检查图片数据是否为空
	if len(dibujo) == 0 {
		return fmt.Errorf("图片数据不能为空")
	}

	// 生成文件路径
	filePath := fmt.Sprintf("image/%s.jpg", articuloID)

	// 确保目录存在
	dir := filepath.Dir(filePath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		log.Printf("[ERROR] 创建目录失败: %v", err)
		return fmt.Errorf("创建目录失败: %w", err)
	}

	// 保存base64图片
	if err := s.saveBase64Image([]byte(dibujo), filePath); err != nil {
		log.Printf("[ERROR] 保存图片失败: %v", err)
		return fmt.Errorf("保存图片失败: %w", err)
	}

	log.Printf("[INFO] 图片上传完成: ArticuloID=%s, FilePath=%s", articuloID, filePath)
	return nil
}

// saveBase64Image 保存base64编码的图片到文件
func (s *ArticuloServiceImpl) saveBase64Image(data []byte, filename string) error {
	// 解码 []byte base64 数据
	decoded, err := base64.StdEncoding.DecodeString(string(data))
	if err != nil {
		return fmt.Errorf("解码失败: %w", err)
	}

	// 保存文件
	err = os.WriteFile(filename, decoded, 0644)
	if err != nil {
		return fmt.Errorf("写入文件失败: %w", err)
	}

	log.Printf("[INFO] 图片保存成功: %s", filename)
	return nil
}
