package service

import (
	"fmt"
	"gorm.io/gorm"
	"io"
	"log"
	"os"
	"regexp"
	"supermarket-go/internal/model"
	"supermarket-go/internal/util"
	"sync"
)

// ArticuloTableImportServiceImpl 商品表格导入服务实现
type ArticuloTableImportServiceImpl struct {
	db                    *gorm.DB
	returnGoodsService    *ReturnGoodsService
	articulodibujoService ArticulodibujoService
}

// NewArticuloTableImportService 创建商品表格导入服务
func NewArticuloTableImportService(db *gorm.DB) ArticuloTableImportService {
	// 创建依赖的服务
	articuloService := NewArticuloService(db)
	procedureService := NewProcedureService(db)
	articuloClaseService := NewArticuloClaseService(db)
	ivaTipoService := NewIvaTipoService(db)
	returnGoodsService := NewReturnGoodsService(db, articuloService, procedureService, articuloClaseService, ivaTipoService)
	articulodibujoService := NewArticulodibujoService(db)

	return &ArticuloTableImportServiceImpl{
		db:                    db,
		returnGoodsService:    returnGoodsService,
		articulodibujoService: articulodibujoService,
	}
}

// Resolve 解析URL，返回解析结果
func (s *ArticuloTableImportServiceImpl) Resolve(url string) (*model.ResolveResult, error) {
	log.Printf("[INFO] 开始解析URL: %s", url)

	// 提取域名
	domain := s.extractDomain(url)
	if domain == "" {
		log.Printf("[ERROR] 无法提取域名: %s", url)
		return nil, nil
	}

	log.Printf("[INFO] 提取到域名: %s", domain)

	// 根据域名选择合适的解析器
	resolver := s.createResolver(domain, url)
	if resolver == nil {
		log.Printf("[ERROR] 无法为域名创建解析器: %s", domain)
		return nil, nil
	}

	// 初始化解析器
	if err := resolver.Init(url); err != nil {
		log.Printf("[ERROR] 初始化解析器失败: %v", err)
		return nil, err
	}

	result := &model.ResolveResult{
		Binding:  domain, // 暂时使用域名作为binding
		Resolver: resolver,
	}

	log.Printf("[INFO] 成功创建解析器: %T", resolver)
	return result, nil
}

// Import 导入商品数据
func (s *ArticuloTableImportServiceImpl) Import(empresaID int64, empleadoID int, importType int, data []model.TableRow, updateSalePrice bool, updateOnly bool, isSaveImage bool) (bool, error) {
	log.Printf("[INFO] ====== Articulo Import Start ======")
	log.Printf("[INFO] Type: %d, empresaID: %d, empleadoID: %d, data: %d", importType, empresaID, empleadoID, len(data))
	//log.Printf("[INFO] updateSalePrice: %v, updateOnly: %v", updateSalePrice, updateOnly)

	// 开始数据库事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 创建商品映射表，类似于Java版本的mapping
	mapping := make(map[model.TableRow]*model.Articulo)
	wg := sync.WaitGroup{}
	var mutex sync.Mutex
	for _, tableRow := range data {
		wg.Add(1)
		go func(tableRow model.TableRow) {
			defer wg.Done()
			var foundArticulo *model.Articulo

			switch importType {
			case 0: // 按ID匹配
				// 先按ID查找，再按条码查找，最后按条码作为ID查找
				if articulo := s.findArticuloByID(tableRow.ID); articulo != nil {
					foundArticulo = articulo
				} else if articulos := s.findArticulosByBarCode(tableRow.BarCode); len(articulos) > 0 {
					foundArticulo = &articulos[0]
				}
				//} else if articulo := s.findArticuloByID(tx, tableRow.BarCode); articulo != nil {
				//	foundArticulo = articulo
				//}
			case 1: // 按条码匹配
				// 先按条码查找，再按ID查找，最后按条码作为ID查找
				if articulos := s.findArticulosByBarCode(tableRow.BarCode); len(articulos) > 0 {
					foundArticulo = &articulos[0]
				} else if articulo := s.findArticuloByID(tableRow.ID); articulo != nil {
					foundArticulo = articulo
				} else if articulo := s.findArticuloByID(tableRow.BarCode); articulo != nil {
					foundArticulo = articulo
				}
			default: // ID和条码都匹配
				// 先按ID查找，再按条码查找
				if articulo := s.findArticuloByID(tableRow.ID); articulo != nil {
					foundArticulo = articulo
				} else if articulos := s.findArticulosByBarCode(tableRow.BarCode); len(articulos) > 0 {
					foundArticulo = &articulos[0]
				}
			}
			mutex.Lock()
			mapping[tableRow] = foundArticulo
			mutex.Unlock()
		}(tableRow)
	}
	wg.Wait()
	// 打印现有商品信息
	log.Printf("[INFO] Existing in Database:")
	existing := make(map[model.TableRow]*model.Articulo)
	for tableRow, articulo := range mapping {
		if articulo != nil {
			existing[tableRow] = articulo
			log.Printf("[INFO] id: %s, code: %s, name: %s, price: %f, salePrice: %f, quantity: %d ==> id: %s, code: %s, name: %s, priceCost: %f, sellPrice: %f",
				tableRow.ID, tableRow.BarCode, tableRow.Name, tableRow.Price, tableRow.SalePrice, tableRow.Quantity,
				articulo.ArticuloID, articulo.CodigoBarra, articulo.NombreES, articulo.PrecioCoste, articulo.PrecioDetalle)
		}
	}

	// 打印新商品信息
	log.Printf("[INFO] Incoming:")
	incoming := make(map[model.TableRow]bool)
	for tableRow, articulo := range mapping {
		if articulo == nil {
			incoming[tableRow] = true
			log.Printf("[INFO] id: %s, code: %s, name: %s, price: %f, quantity: %d",
				tableRow.ID, tableRow.BarCode, tableRow.Name, tableRow.Price, tableRow.Quantity)
		}
	}

	// 获取现有商品的库存信息
	existingArticuloStockMap := make(map[string]*model.Stock)
	for _, articulo := range existing {
		if stock := s.getStockByArticuloID(articulo.ArticuloID); stock != nil {
			existingArticuloStockMap[articulo.ArticuloID] = stock
		}
	}

	// 获取现有商品的图片信息
	//existingArticuloDibujoMap := make(map[string]struct {
	//	ArticuloDibujo *model.ArticuloDibujo
	//	Dibujo         *model.Dibujo
	//})
	//for _, articulo := range existing {
	//	if articuloDibujo := s.getArticuloDibujoByArticuloID(tx, articulo.ArticuloID); articuloDibujo != nil {
	//		if dibujo := s.getDibujoByID(tx, articuloDibujo.DibujoID); dibujo != nil {
	//			existingArticuloDibujoMap[articulo.ArticuloID] = struct {
	//				ArticuloDibujo *model.ArticuloDibujo
	//				Dibujo         *model.Dibujo
	//			}{
	//				ArticuloDibujo: articuloDibujo,
	//				Dibujo:         dibujo,
	//			}
	//		}
	//	}
	//}

	// 对于不存在的商品直接插入
	var newArticulos []model.Articulo
	for tableRow := range incoming {
		newArticulo := model.Articulo{}

		switch importType {
		case 0:
			// 根据 ID 更新 条码, 也就是 条码 保持 和 ID 一致
			newArticulo.ArticuloID = tableRow.ID
			newArticulo.CodigoBarra = tableRow.ID
		case 1:
			// 根据 条码 更新 ID, 也就是 ID 保持 和 条码 一致
			newArticulo.ArticuloID = tableRow.BarCode
			newArticulo.CodigoBarra = tableRow.BarCode
		default:
			newArticulo.ArticuloID = tableRow.ID
			newArticulo.CodigoBarra = tableRow.BarCode
		}

		newArticulo.ClaseID = tableRow.ClassID
		newArticulo.NombreES = tableRow.Name
		if tableRow.NombreES != "" {
			newArticulo.NombreES = tableRow.NombreES
		}
		newArticulo.NombreCN = tableRow.NombreCN
		newArticulo.PrecioCoste = tableRow.Price
		newArticulo.PrecioDetalle = tableRow.SalePrice
		newArticulo.EmpresaID = empresaID

		log.Printf("[INFO] 保存新商品: id: %s, code: %s, name: %s, price: %f, salePrice: %f",
			newArticulo.ArticuloID, newArticulo.CodigoBarra, newArticulo.NombreES, newArticulo.PrecioCoste, newArticulo.PrecioDetalle)

		newArticulos = append(newArticulos, newArticulo)
	}

	if len(newArticulos) > 0 {
		if err := tx.Create(&newArticulos).Error; err != nil {
			log.Printf("[ERROR] 批量保存新商品失败: %v", err)
			tx.Rollback()
			return false, err
		}
	}

	// 对存在的商品信息进行更新
	for tableRow, original := range existing {
		log.Printf("[INFO] 修改商品 => id: %s, code: %s", original.ArticuloID, original.CodigoBarra)

		// 保留原有商品的销售价
		originalPrecioDetalle := original.PrecioDetalle

		// 删除原有的商品
		//tx.Exec("Delete from articulo where articulo = ? OR CodigoBarra = ?", original.ArticuloID, original.CodigoBarra)
		//if err := tx.Delete(original).Error; err != nil {
		//	log.Printf("[ERROR] 删除原有商品失败: %v", err)
		//	tx.Rollback()
		//	return false, err
		//}
		// 将供货单的商品重新添加进去
		newArticulo := model.Articulo{}

		// 商品 ID
		switch importType {
		case 0:
			// 根据 ID 更新 条码, 也就是 条码 保持 和 ID 一致
			newArticulo.ArticuloID = tableRow.ID
			newArticulo.CodigoBarra = tableRow.ID
		case 1:
			// 根据 条码 更新 ID, 也就是 ID 保持 和 条码 一致
			newArticulo.ArticuloID = tableRow.BarCode
			newArticulo.CodigoBarra = tableRow.BarCode
		default:
			newArticulo.ArticuloID = tableRow.ID
			newArticulo.CodigoBarra = tableRow.BarCode
		}
		tx.Exec("Delete from articulo where ArticuloID = ? OR CodigoBarra = ?", newArticulo.ArticuloID, newArticulo.CodigoBarra)
		// 商品类别
		newArticulo.ClaseID = tableRow.ClassID
		// 商品名称
		newArticulo.NombreES = tableRow.NombreES
		newArticulo.NombreCN = tableRow.NombreCN
		// 商品进价
		newArticulo.PrecioCoste = tableRow.Price

		if updateSalePrice {
			// 更新销售价
			log.Printf("[INFO] 将销售价 %f 更新为 %f", originalPrecioDetalle, tableRow.SalePrice)
			newArticulo.PrecioDetalle = tableRow.SalePrice
		} else {
			// 保留原有的销售价
			log.Printf("[INFO] 保留原有的销售价 %f", originalPrecioDetalle)
			newArticulo.PrecioDetalle = originalPrecioDetalle
		}

		newArticulo.EmpresaID = empresaID

		log.Printf("[INFO] 删除原有商品并保存新商品: id: %s, code: %s, name: %s, price: %f, salePrice: %f",
			newArticulo.ArticuloID, newArticulo.CodigoBarra, newArticulo.NombreES, newArticulo.PrecioCoste, newArticulo.PrecioDetalle)

		if err := tx.Create(&newArticulo).Error; err != nil {
			log.Printf("[ERROR] 保存新商品失败: %v", err)
			tx.Rollback()
			return false, err
		}
	}

	// 更新 ArticuloExtra 表
	//log.Printf("[INFO] 更新 ArticuloExtra 表")
	//for tableRow := range incoming {
	//	articuloID := tableRow.ID
	//	if importType == 1 {
	//		articuloID = tableRow.BarCode
	//	}
	//	s.saveOrUpdateArticuloExtra(tx, "", articuloID, tableRow.BarCode, tableRow.Name)
	//}
	//
	//for tableRow, _ := range existing {
	//	originalArticuloID := tableRow.ID
	//	articuloID := tableRow.ID
	//	if importType == 1 {
	//		originalArticuloID = tableRow.ID
	//		articuloID = tableRow.BarCode
	//	}
	//	s.saveOrUpdateArticuloExtra(tx, originalArticuloID, articuloID, tableRow.BarCode, tableRow.Name)
	//}

	if !updateOnly {
		// 更新商品库存数量
		for tableRow := range incoming {
			articuloID := tableRow.ID
			if importType == 1 {
				articuloID = tableRow.BarCode
			}
			s.saveOrUpdateStock(tx, articuloID, tableRow.Quantity)
		}

		for tableRow, articulo := range existing {
			realArticuloID := tableRow.ID
			if importType == 1 {
				realArticuloID = tableRow.BarCode
			}
			originalArticuloID := articulo.ArticuloID
			originalStock := existingArticuloStockMap[originalArticuloID]

			if originalStock != nil {
				log.Printf("[INFO] 商品 %s (new: %s) 原有库存: %d", originalArticuloID, realArticuloID, originalStock.Stock)
				// 删除原有库存
				if err := tx.Delete(originalStock).Error; err != nil {
					log.Printf("[ERROR] 删除原有库存失败: %v", err)
					tx.Rollback()
					return false, err
				}
				// 保存新库存
				s.saveOrUpdateStock(tx, realArticuloID, int(originalStock.Stock)+tableRow.Quantity)
			} else {
				s.saveOrUpdateStock(tx, realArticuloID, tableRow.Quantity)
			}
		}

		//// 插入图片
		//log.Printf("[INFO] 保存商品图片")
		//for tableRow := range mapping {
		//	s.downloadAndSaveDibujo(tx, tableRow, importType)
		//}
	} else {
		log.Printf("[INFO] 由于仅更新销售价, 跳过库存、ArticuloExtra 以及供货单的更新, 但库存仍要被重新写入")

		// 保存商品库存
		for tableRow := range incoming {
			articuloID := tableRow.ID
			if importType == 1 {
				articuloID = tableRow.BarCode
			}
			s.saveOrUpdateStock(tx, articuloID, tableRow.Quantity)
		}

		// 重新写入库存
		for tableRow, articulo := range existing {
			realArticuloID := tableRow.ID
			if importType == 1 {
				realArticuloID = tableRow.BarCode
			}
			originalArticuloID := articulo.ArticuloID
			if originalStock := existingArticuloStockMap[originalArticuloID]; originalStock != nil {
				s.saveOrUpdateStock(tx, realArticuloID, int(originalStock.Stock))
			}
		}

		// 重新写入图片
		//for tableRow, articulo := range existing {
		//	realArticuloID := tableRow.ID
		//	if importType == 1 {
		//		realArticuloID = tableRow.BarCode
		//	}
		//	originalArticuloID := articulo.ArticuloID
		//
		//	//if dibujoData := existingArticuloDibujoMap[originalArticuloID]; dibujoData.ArticuloDibujo != nil {
		//	//	log.Printf("[INFO] 保存商品 %s 图片 ID: %d", realArticuloID, dibujoData.ArticuloDibujo.DibujoID)
		//
		//	// 保存商品图片关联
		//	//newArticuloDibujo := *dibujoData.ArticuloDibujo
		//	//newArticuloDibujo.ArticuloID = realArticuloID
		//	//if err := tx.Create(&newArticuloDibujo).Error; err != nil {
		//	//	log.Printf("[ERROR] 保存商品图片关联失败: %v", err)
		//	//	tx.Rollback()
		//	//	return false, err
		//	//}
		//
		//	//// 更新商品表中的图片 ID
		//	//if err := tx.Model(&model.Articulo{}).Where("articuloid = ?", realArticuloID).Update("dibujoid", dibujoData.ArticuloDibujo.DibujoID).Error; err != nil {
		//	//	log.Printf("[ERROR] 更新商品图片ID失败: %v", err)
		//	//	tx.Rollback()
		//	//	return false, err
		//	//}
		//	//}
		//}
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("[ERROR] 提交事务失败: %v", err)
		return false, err
	}
	// 插入进货单
	log.Printf("[INFO] 正在保存进货单")
	var articuloList []model.ReturnGoodsListArticulo
	for tableRow := range mapping {
		articuloList = append(articuloList, model.ReturnGoodsListArticulo{
			Type: importType,
			Value: func() string {
				if importType == 1 {
					return tableRow.BarCode
				}
				return tableRow.ID
			}(),
			Amount:   tableRow.Quantity,
			Precio:   tableRow.Price,
			Nombre:   tableRow.Name,
			NombreCN: tableRow.NombreCN,
			NombreES: tableRow.NombreES,
			ClassID:  tableRow.ClassID,
		})
	}

	returnGoodsList := model.ReturnGoodsListDTO{
		EmpresaId:    fmt.Sprintf("%d", empresaID),
		ArticuloList: articuloList,
	}

	if !s.returnGoodsService.SubmitReturnGoods(empleadoID, returnGoodsList, true) {
		log.Printf("[ERROR] 保存进货单失败")
		tx.Rollback()
		return false, fmt.Errorf("保存进货单失败")
	}
	// 保存商品图片
	if isSaveImage {
		wg = sync.WaitGroup{}
		for tableRow := range incoming {
			wg.Add(1)
			go func(tableRow model.TableRow) {
				s.downloadAndSaveDibujo(tableRow, importType)
				wg.Done()
			}(tableRow)
		}
		for tableRow := range existing {
			wg.Add(1)
			go func(tableRow model.TableRow) {
				s.downloadAndSaveDibujo(tableRow, importType)
				wg.Done()
			}(tableRow)
		}
		wg.Wait()
	}
	log.Printf("[INFO] ====== Articulo Import End======")
	return true, nil
}

// extractDomain 提取域名
func (s *ArticuloTableImportServiceImpl) extractDomain(url string) string {
	re := regexp.MustCompile(`^(?:https?://)?([^/?:]+)`)
	matches := re.FindStringSubmatch(url)
	if len(matches) > 1 {
		return matches[1]
	}
	return ""
}

// createResolver 根据域名创建解析器
func (s *ArticuloTableImportServiceImpl) createResolver(domain, url string) model.ArticleImportPageResolver {
	// 这里应该根据不同的域名返回不同的解析器
	// 暂时返回一个默认解析器
	log.Printf("[INFO] 为域名 %s 创建默认解析器", domain)
	return &DefaultArticleImportPageResolver{
		url: url,
	}
}

// DefaultArticleImportPageResolver 默认商品导入页面解析器
type DefaultArticleImportPageResolver struct {
	url string
}

func (r *DefaultArticleImportPageResolver) GetExcelDownloadUrl() string {
	// 默认实现，返回原URL
	return r.url
}

func (r *DefaultArticleImportPageResolver) GetTableData() []model.TableRow {
	// 默认实现，返回空数据
	// 实际应该解析网页内容
	return []model.TableRow{}
}

func (r *DefaultArticleImportPageResolver) GetDomain() string {
	re := regexp.MustCompile(`^(?:https?://)?([^/?:]+)`)
	matches := re.FindStringSubmatch(r.url)
	if len(matches) > 1 {
		return matches[1]
	}
	return ""
}

func (r *DefaultArticleImportPageResolver) Init(url string) error {
	r.url = url
	return nil
}

// findArticuloByID 根据ID查找商品（返回指针）
func (s *ArticuloTableImportServiceImpl) findArticuloByID(id string) *model.Articulo {
	var articulo model.Articulo
	if err := s.db.Where("ArticuloID = ?", id).First(&articulo).Error; err != nil {
		return nil
	}
	return &articulo
}

// findArticulosByBarCode 根据条码查找商品（返回切片）
func (s *ArticuloTableImportServiceImpl) findArticulosByBarCode(barCode string) []model.Articulo {
	var articulos []model.Articulo
	if err := s.db.Where("CodigoBarra = ?", barCode).Find(&articulos).Error; err != nil {
		return nil
	}
	return articulos
}

// getStockByArticuloID 根据商品ID获取库存
func (s *ArticuloTableImportServiceImpl) getStockByArticuloID(articuloID string) *model.Stock {
	var stock model.Stock
	if err := s.db.Where("ArticuloID = ?", articuloID).First(&stock).Error; err != nil {
		return nil
	}
	return &stock
}

// getArticuloDibujoByArticuloID 根据商品ID获取商品图片
func (s *ArticuloTableImportServiceImpl) getArticuloDibujoByArticuloID(tx *gorm.DB, articuloID string) *model.ArticuloDibujo {
	var articuloDibujo model.ArticuloDibujo
	if err := tx.Where("ArticuloID = ?", articuloID).First(&articuloDibujo).Error; err != nil {
		return nil
	}
	return &articuloDibujo
}

// getDibujoByID 根据ID获取图片
func (s *ArticuloTableImportServiceImpl) getDibujoByID(tx *gorm.DB, dibujoID int64) *model.Dibujo {
	var dibujo model.Dibujo
	if err := tx.Where("DibujoID = ?", dibujoID).First(&dibujo).Error; err != nil {
		return nil
	}
	return &dibujo
}

// saveOrUpdateArticuloExtra 保存或更新ArticuloExtra
func (s *ArticuloTableImportServiceImpl) saveOrUpdateArticuloExtra(tx *gorm.DB, originalArticuloID, articuloID, code, name string) {
	// 这里需要实现ArticuloExtra的保存逻辑
	// 由于Go版本可能没有ArticuloExtra模型，暂时跳过
	log.Printf("[INFO] 保存ArticuloExtra: originalArticuloID=%s, articuloID=%s, code=%s, name=%s",
		originalArticuloID, articuloID, code, name)
}

// saveOrUpdateStock 保存或更新库存
func (s *ArticuloTableImportServiceImpl) saveOrUpdateStock(tx *gorm.DB, articuloID string, quantity int) {
	var stock model.Stock
	if err := tx.Where("articuloid = ?", articuloID).First(&stock).Error; err != nil {
		// 不存在则创建
		stock = model.Stock{
			AlmacenID:  1, // 默认仓库ID
			ArticuloID: articuloID,
			PropCount:  0,
			PropID1:    "",
			PropID2:    "",
			Stock:      float64(quantity),
		}
		if err := tx.Create(&stock).Error; err != nil {
			log.Printf("[ERROR] 创建库存失败: %v", err)
		}
	} else {
		// 存在则更新
		stock.Stock = float64(quantity)
		if err := tx.Save(&stock).Error; err != nil {
			log.Printf("[ERROR] 更新库存失败: %v", err)
		}
	}
}

// downloadAndSaveDibujo 下载并保存图片
func (s *ArticuloTableImportServiceImpl) downloadAndSaveDibujo(tableRow model.TableRow, importType int) bool {
	url := tableRow.Icon
	articuloID := tableRow.ID
	if importType == 1 {
		articuloID = tableRow.BarCode
	}

	if url == "" {
		log.Printf("[INFO] 图片URL为空，跳过下载")
		return false
	}

	log.Printf("[INFO] 下载图片: %s", url)

	// 下载图片
	networkUtils := &util.NetworkUtils{}
	inputStream, err := networkUtils.DownloadFileIS(url)
	if err != nil {
		log.Printf("[ERROR] 下载图片失败: %v", err)
		return false
	}
	defer inputStream.Close()

	// 确保 image 文件夹存在
	if err := os.MkdirAll("image", os.ModePerm); err != nil {
		log.Printf("[ERROR] 创建目录失败: %v", err)
		return false
	}

	// 拼接文件路径（用 articuloID 作为文件名）
	filePath := fmt.Sprintf("image/%s.jpg", articuloID)

	// 创建文件
	outFile, err := os.Create(filePath)
	if err != nil {
		log.Printf("[ERROR] 创建文件失败: %v", err)
		return false
	}
	defer outFile.Close()

	// 拷贝数据到文件
	_, err = io.Copy(outFile, inputStream)
	if err != nil {
		log.Printf("[ERROR] 保存图片失败: %v", err)
		return false
	}

	log.Printf("[INFO] 图片保存成功, 路径: %s", filePath)
	return true
}
