package service

import (
	"gorm.io/gorm"
	"log"
	"strconv"
	"supermarket-go/internal/model"
	"supermarket-go/internal/util"
)

type EmpleadoAuthService struct {
	db *gorm.DB
}

func NewEmpleadoAuthService(db *gorm.DB) *EmpleadoAuthService {
	log.Println("[INFO] 创建新的 EmpleadoAuthService 实例")
	return &EmpleadoAuthService{db: db}
}

// Login 员工登录
func (s *EmpleadoAuthService) Login(empleadoLoginName, contrasena string) string {
	log.Printf("[INFO] 开始员工登录验证: empleadoLoginName=%s", empleadoLoginName)

	var empleado model.Empleado

	// 先尝试通过姓名查找
	log.Printf("[INFO] 尝试通过姓名查找员工: %s", empleadoLoginName)
	err := s.db.Where("Nombre = ?", empleadoLoginName).First(&empleado).Error
	if err != nil {
		log.Printf("[WARN] 通过姓名未找到员工: %s", empleadoLoginName)

		// 如果通过姓名找不到，尝试通过ID查找
		if id, err := strconv.Atoi(empleadoLoginName); err == nil {
			log.Printf("[INFO] 尝试通过ID查找员工: %d", id)
			err = s.db.Where("EmpleadoID = ?", id).First(&empleado).Error
			if err != nil {
				log.Printf("[ERROR] 通过ID也未找到员工: %d", id)
				// 员工ID不存在
				return ""
			}
			log.Printf("[INFO] 通过ID找到员工: EmpleadoID=%d, Nombre=%s",
				empleado.EmpleadoID, empleado.Nombre)
		} else {
			log.Printf("[ERROR] 员工登录名格式无效: %s", empleadoLoginName)
			// 员工ID不存在
			return ""
		}
	} else {
		log.Printf("[INFO] 通过姓名找到员工: EmpleadoID=%d, Nombre=%s",
			empleado.EmpleadoID, empleado.Nombre)
	}

	// 验证密码
	log.Printf("[INFO] 开始验证密码: EmpleadoID=%d", empleado.EmpleadoID)
	if empleado.Contrasena == util.Md5PassWd(contrasena) {
		log.Printf("[INFO] 密码验证成功: EmpleadoID=%d", empleado.EmpleadoID)

		// 登录成功，生成JWT token
		payloads := map[string]interface{}{
			"id":   empleado.EmpleadoID,
			"name": empleado.Nombre,
		}

		// 设置token过期时间为24000天
		expiration := int64(24000 * 24 * 60 * 60 * 1000)
		log.Printf("[INFO] 生成JWT token: EmpleadoID=%d, 过期时间=%v",
			empleado.EmpleadoID, expiration)
		token, err := util.BuildJwtToken(payloads, expiration)
		if err != nil {
			log.Printf("[ERROR] 生成JWT token失败: EmpleadoID=%d, Error=%v",
				empleado.EmpleadoID, err)
			return ""
		}

		log.Printf("[INFO] JWT token生成成功: EmpleadoID=%d", empleado.EmpleadoID)
		return token
	} else {
		log.Printf("[WARN] 密码验证失败: EmpleadoID=%d", empleado.EmpleadoID)
		// 密码错误
		return ""
	}
}

type EmpleadoService struct {
	db *gorm.DB
}

func NewEmpleadoService(db *gorm.DB) *EmpleadoService {
	log.Println("[INFO] 创建新的 EmpleadoService 实例")
	return &EmpleadoService{db: db}
}

// GetEmpleadoById 根据ID获取员工信息
func (s *EmpleadoService) GetEmpleadoById(id int) *model.Empleado {
	log.Printf("[INFO] 根据ID查询员工信息: EmpleadoID=%d", id)

	var empleado model.Empleado
	if err := s.db.Where("EmpleadoID = ?", id).First(&empleado).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			log.Printf("[WARN] 未找到员工: EmpleadoID=%d", id)
		} else {
			log.Printf("[ERROR] 查询员工失败: EmpleadoID=%d, Error=%v", id, err)
		}
		return nil
	}

	log.Printf("[INFO] 成功查询到员工: EmpleadoID=%d, Nombre=%s",
		id, empleado.Nombre)
	return &empleado
}

// GetEmpleadoByName 根据姓名获取员工信息
func (s *EmpleadoService) GetEmpleadoByName(name string) *model.Empleado {
	log.Printf("[INFO] 根据姓名查询员工信息: Nombre=%s", name)

	var empleado model.Empleado
	if err := s.db.Where("Nombre = ?", name).First(&empleado).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			log.Printf("[WARN] 未找到员工: Nombre=%s", name)
		} else {
			log.Printf("[ERROR] 查询员工失败: Nombre=%s, Error=%v", name, err)
		}
		return nil
	}

	log.Printf("[INFO] 成功查询到员工: EmpleadoID=%d, Nombre=%s",
		empleado.EmpleadoID, empleado.Nombre)
	return &empleado
}

// GetAllEmpleadoNames 获取所有员工姓名
func (s *EmpleadoService) GetAllEmpleadoNames() map[int]string {
	log.Println("[INFO] 查询所有员工姓名")

	var empleados []model.Empleado
	err := s.db.Find(&empleados).Error
	if err != nil {
		log.Printf("[ERROR] 查询所有员工失败: %v", err)
		return make(map[int]string)
	}

	names := make(map[int]string)
	for _, e := range empleados {
		names[e.EmpleadoID] = e.Nombre
	}

	log.Printf("[INFO] 成功查询到 %d 个员工", len(empleados))
	return names
}
