package service

import (
	"gorm.io/gorm"
	"strconv"
	"time"
)

type ProcedureService struct {
	db *gorm.DB
}

func NewProcedureService(db *gorm.DB) *ProcedureService {
	return &ProcedureService{db: db}
}

// GenerateDocumentoNo 生成单据编号，对应Java的generateDocumentoNo方法
func (s *ProcedureService) GenerateDocumentoNo(tableName, dateStr string) int64 {
	// 调用存储过程生成单据编号
	//var documentoNo int64
	//
	//// 这里需要根据实际的数据库存储过程来实现
	//// 暂时使用简单的实现方式
	//query := `CALL new_document_no(?, ?, ?)`
	//
	//// 执行存储过程
	//if err := s.db.Raw(query, tableName, dateStr, &documentoNo).Scan(&documentoNo).Error; err != nil {
	//	// 如果存储过程调用失败，使用备用方案
	//	documentoNo = s.generateDocumentoNoFallback(tableName, dateStr)
	//}

	// 执行存储过程，通过会话变量接收OUT参数
	if err := s.db.Exec("CALL new_document_no(?, ?, @docNo)", tableName, dateStr).Error; err != nil {
		return 0
	}

	// 查询OUT参数值
	var result struct{ DocNo int64 }
	if err := s.db.Raw("SELECT @docNo AS doc_no").Scan(&result).Error; err != nil {
		return 0
	}

	return result.DocNo

	//return documentoNo
}

// generateDocumentoNoFallback 备用方案：手动生成单据编号
func (s *ProcedureService) generateDocumentoNoFallback(tableName, dateStr string) int64 {
	// 获取当前日期作为前缀
	datePrefix := time.Now().Format("20060102")

	// 查询当前最大编号
	var maxNo int64
	query := `SELECT COALESCE(MAX(DocumentoNo), 0) FROM ` + tableName + ` WHERE DocumentoNo LIKE ?`
	prefix := datePrefix + "%"

	s.db.Raw(query, prefix).Scan(&maxNo)

	// 生成新的编号
	if maxNo == 0 {
		// 如果没有记录，从当前日期开始
		newNo, _ := strconv.ParseInt(datePrefix+"0001", 10, 64)
		return newNo
	} else {
		// 递增编号
		return maxNo + 1
	}
}

// GenerateTicketNo 生成票据编号
func (s *ProcedureService) GenerateTicketNo(dateStr string) int64 {
	var ticketNo int64

	query := `SELECT new_ticket_no(?)`

	if err := s.db.Raw(query, dateStr).Scan(&ticketNo).Error; err != nil {
		// 备用方案
		ticketNo = s.generateTicketNoFallback(dateStr)
	}

	return ticketNo
}

// generateTicketNoFallback 备用方案：手动生成票据编号
func (s *ProcedureService) generateTicketNoFallback(dateStr string) int64 {
	datePrefix := time.Now().Format("20060102")

	var maxNo int64
	query := `SELECT COALESCE(MAX(DocumentoNo), 0) FROM ticket WHERE DocumentoNo LIKE ?`
	prefix := datePrefix + "%"

	s.db.Raw(query, prefix).Scan(&maxNo)

	if maxNo == 0 {
		newNo, _ := strconv.ParseInt(datePrefix+"0001", 10, 64)
		return newNo
	} else {
		return maxNo + 1
	}
}
