package service

import (
	"context"
	"log"
	"time"

	"gorm.io/gorm"
	"gorm.io/gorm/clause"
	"supermarket-go/internal/model"
)

// DataSyncService 负责跨库同步
type DataSyncService struct {
	dbA *gorm.DB
	dbB *gorm.DB
}

func NewDataSyncService(dbA *gorm.DB, dbB *gorm.DB) *DataSyncService {
	return &DataSyncService{dbA: dbA, dbB: dbB}
}

// SyncArticuloMissing 将数据库A中的 articulo 在B中不存在的记录插入到B
// 通过 INSERT IGNORE/ON CONFLICT DO NOTHING 的方式实现“存在则跳过，不存在则新增”
func (s *DataSyncService) SyncArticuloMissing(ctx context.Context) error {
	if s.dbA == nil || s.dbB == nil {
		log.Printf("[WARN] 同步跳过：数据库连接为空 dbA=%v dbB=%v", s.dbA != nil, s.dbB != nil)
		return nil
	}

	const pageSize = 500
	log.Printf("[INFO] 开始同步 articulo（缺失即插入），批大小=%d（游标式批处理）", pageSize)

	// 使用 GORM 的批处理API，接近游标的方式逐批拉取，避免一次性加载太多数据
	var batch []model.Articulo
	err := s.dbA.WithContext(ctx).
		Model(&model.Articulo{}).
		Order("ArticuloID").
		FindInBatches(&batch, pageSize, func(tx *gorm.DB, _ int) error {
			if len(batch) == 0 {
				return nil
			}
			// 目标库B，存在则跳过，不存在则插入
			if err := s.dbB.WithContext(ctx).Clauses(clause.OnConflict{DoNothing: true}).Create(&batch).Error; err != nil {
				log.Printf("[ERROR] 插入到B库 articulo 失败: %v", err)
				return err
			}
			return nil
		}).Error
	if err != nil {
		return err
	}

	log.Printf("[INFO] 同步 articulo 完成")
	return nil
}

// SyncProveedorMissing 将数据库A中的 proveedor 在B中不存在的记录插入到B
// 仅同步 EmpresaID 位数为 9 的记录；存在则跳过，不存在则新增
func (s *DataSyncService) SyncProveedorMissing(ctx context.Context) error {
	if s.dbA == nil || s.dbB == nil {
		log.Printf("[WARN] 同步跳过：数据库连接为空 dbA=%v dbB=%v", s.dbA != nil, s.dbB != nil)
		return nil
	}

	const pageSize = 500
	log.Printf("[INFO] 开始同步 proveedor（仅 EmpresaID 长度=9，缺失即插入），批大小=%d（游标式批处理）", pageSize)

	// 使用 GORM 的批处理API，逐批拉取符合条件的数据，避免一次性加载过多
	var batch []model.Proveedor
	err := s.dbA.WithContext(ctx).
		Model(&model.Proveedor{}).
		Where("LENGTH(EmpresaID) = ?", 9).
		Order("EmpresaID").
		FindInBatches(&batch, pageSize, func(tx *gorm.DB, _ int) error {
			if len(batch) == 0 {
				return nil
			}
			// 目标库B，存在则跳过，不存在则插入
			if err := s.dbB.WithContext(ctx).Clauses(clause.OnConflict{DoNothing: true}).Create(&batch).Error; err != nil {
				log.Printf("[ERROR] 插入到B库 proveedor 失败: %v", err)
				return err
			}
			return nil
		}).Error
	if err != nil {
		return err
	}

	log.Printf("[INFO] 同步 proveedor 完成")
	return nil
}

// SyncClienteMissing 将数据库A中的 cliente 在B中不存在的记录插入到B
// 仅同步 EmpresaID 位数为 9 的记录；存在则跳过，不存在则新增
func (s *DataSyncService) SyncClienteMissing(ctx context.Context) error {
	if s.dbA == nil || s.dbB == nil {
		log.Printf("[WARN] 同步跳过：数据库连接为空 dbA=%v dbB=%v", s.dbA != nil, s.dbB != nil)
		return nil
	}

	const pageSize = 500
	log.Printf("[INFO] 开始同步 cliente（仅 EmpresaID 长度=9，缺失即插入），批大小=%d（游标式批处理）", pageSize)

	// 使用 GORM 的批处理API，逐批拉取符合条件的数据，避免一次性加载过多
	var batch []model.Cliente
	err := s.dbA.WithContext(ctx).
		Model(&model.Cliente{}).
		Where("LENGTH(EmpresaID) = ?", 9).
		Order("EmpresaID").
		FindInBatches(&batch, pageSize, func(tx *gorm.DB, _ int) error {
			if len(batch) == 0 {
				return nil
			}
			// 目标库B，存在则跳过，不存在则插入
			if err := s.dbB.WithContext(ctx).Clauses(clause.OnConflict{DoNothing: true}).Create(&batch).Error; err != nil {
				log.Printf("[ERROR] 插入到B库 cliente 失败: %v", err)
				return err
			}
			return nil
		}).Error
	if err != nil {
		return err
	}

	log.Printf("[INFO] 同步 cliente 完成")
	return nil
}

// StartClienteCron 启动定时器：每 interval 运行一次，同步 cliente；启动时立即执行一次
func (s *DataSyncService) StartClienteCron(ctx context.Context, interval time.Duration) {
	if s.dbB == nil {
		log.Printf("[WARN] 未配置数据库B，跳过 cliente 定时同步任务")
		return
	}

	// 首次立即执行
	if err := s.SyncClienteMissing(ctx); err != nil {
		log.Printf("[ERROR] 首次同步 cliente 失败: %v", err)
	}

	ticker := time.NewTicker(interval)
	go func() {
		for {
			select {
			case <-ctx.Done():
				ticker.Stop()
				log.Printf("[INFO] cliente 定时同步终止")
				return
			case <-ticker.C:
				if err := s.SyncClienteMissing(ctx); err != nil {
					log.Printf("[ERROR] 定时同步 cliente 失败: %v", err)
				}
			}
		}
	}()
}

// StartProveedorCron 启动定时器：每 interval 运行一次，同步 proveedor；启动时立即执行一次
func (s *DataSyncService) StartProveedorCron(ctx context.Context, interval time.Duration) {
	if s.dbB == nil {
		log.Printf("[WARN] 未配置数据库B，跳过 proveedor 定时同步任务")
		return
	}

	// 首次立即执行
	if err := s.SyncProveedorMissing(ctx); err != nil {
		log.Printf("[ERROR] 首次同步 proveedor 失败: %v", err)
	}

	ticker := time.NewTicker(interval)
	go func() {
		for {
			select {
			case <-ctx.Done():
				ticker.Stop()
				log.Printf("[INFO] proveedor 定时同步终止")
				return
			case <-ticker.C:
				if err := s.SyncProveedorMissing(ctx); err != nil {
					log.Printf("[ERROR] 定时同步 proveedor 失败: %v", err)
				}
			}
		}
	}()
}

// StartArticuloCron 启动定时器：每 interval 运行一次，同步 articulo；启动时立即执行一次
func (s *DataSyncService) StartArticuloCron(ctx context.Context, interval time.Duration) {
	if s.dbB == nil {
		log.Printf("[WARN] 未配置数据库B，跳过定时同步任务")
		return
	}

	// 首次立即执行
	if err := s.SyncArticuloMissing(ctx); err != nil {
		log.Printf("[ERROR] 首次同步 articulo 失败: %v", err)
	}

	ticker := time.NewTicker(interval)
	go func() {
		for {
			select {
			case <-ctx.Done():
				ticker.Stop()
				log.Printf("[INFO] articulo 定时同步终止")
				return
			case <-ticker.C:
				if err := s.SyncArticuloMissing(ctx); err != nil {
					log.Printf("[ERROR] 定时同步 articulo 失败: %v", err)
				}
			}
		}
	}()
}
