package util

import (
	"bytes"
	"encoding/base64"
	"fmt"
	"image"
	"image/jpeg"
	"log"

	"golang.org/x/image/draw"
)

// CompressImageToBase64 压缩图片并转换为base64字符串
// originalBase64: 原始图片的base64字符串
// maxWidth: 最大宽度（像素），默认800
// maxHeight: 最大高度（像素），默认800
// quality: JPEG质量（1-100），默认75
// 返回压缩后的base64字符串和错误
func CompressImageToBase64(originalBase64 string, maxWidth, maxHeight, quality int) (string, error) {
	if originalBase64 == "" {
		return "", fmt.Errorf("原始图片数据为空")
	}

	// 设置默认值
	if maxWidth <= 0 {
		maxWidth = 800
	}
	if maxHeight <= 0 {
		maxHeight = 800
	}
	if quality <= 0 || quality > 100 {
		quality = 75
	}

	// 解码base64
	imageData, err := base64.StdEncoding.DecodeString(originalBase64)
	if err != nil {
		log.Printf("[ERROR] base64解码失败: %v", err)
		return "", fmt.Errorf("base64解码失败: %w", err)
	}

	// 解码图片
	img, _, err := image.Decode(bytes.NewReader(imageData))
	if err != nil {
		log.Printf("[ERROR] 图片解码失败: %v", err)
		return "", fmt.Errorf("图片解码失败: %w", err)
	}

	// 获取原始尺寸
	bounds := img.Bounds()
	originalWidth := bounds.Dx()
	originalHeight := bounds.Dy()

	log.Printf("[INFO] 原始图片尺寸: %dx%d", originalWidth, originalHeight)

	// 计算缩放后的尺寸（保持宽高比）
	var newWidth, newHeight int
	if originalWidth <= maxWidth && originalHeight <= maxHeight {
		// 图片已经小于最大尺寸，不需要缩放，但仍可进行质量压缩
		newWidth = originalWidth
		newHeight = originalHeight
	} else {
		// 计算缩放比例
		widthRatio := float64(maxWidth) / float64(originalWidth)
		heightRatio := float64(maxHeight) / float64(originalHeight)
		ratio := widthRatio
		if heightRatio < widthRatio {
			ratio = heightRatio
		}

		newWidth = int(float64(originalWidth) * ratio)
		newHeight = int(float64(originalHeight) * ratio)
		log.Printf("[INFO] 缩放后尺寸: %dx%d (比例: %.2f)", newWidth, newHeight, ratio)
	}

	// 创建新图片
	dst := image.NewRGBA(image.Rect(0, 0, newWidth, newHeight))

	// 使用高质量的缩放算法
	draw.CatmullRom.Scale(dst, dst.Bounds(), img, bounds, draw.Over, nil)

	// 编码为JPEG
	var buf bytes.Buffer
	err = jpeg.Encode(&buf, dst, &jpeg.Options{Quality: quality})
	if err != nil {
		log.Printf("[ERROR] JPEG编码失败: %v", err)
		return "", fmt.Errorf("JPEG编码失败: %w", err)
	}

	// 转换为base64
	compressedBase64 := base64.StdEncoding.EncodeToString(buf.Bytes())

	originalSize := len(imageData)
	compressedSize := buf.Len()
	compressionRatio := float64(compressedSize) / float64(originalSize) * 100

	log.Printf("[INFO] 图片压缩完成: 原始大小=%d bytes, 压缩后大小=%d bytes, 压缩率=%.2f%%",
		originalSize, compressedSize, compressionRatio)

	return compressedBase64, nil
}

// CompressImageBytesToBase64 压缩图片字节数组并转换为base64字符串
func CompressImageBytesToBase64(imageBytes []byte, maxWidth, maxHeight, quality int) (string, error) {
	if len(imageBytes) == 0 {
		return "", fmt.Errorf("原始图片数据为空")
	}

	// 先转换为base64
	originalBase64 := base64.StdEncoding.EncodeToString(imageBytes)
	return CompressImageToBase64(originalBase64, maxWidth, maxHeight, quality)
}

