package util

import (
	"errors"
	"strings"
	"time"

	"github.com/golang-jwt/jwt/v5"
)

const (
	SIGN_KEY_EMPLEADO_AUTH = "empleado@auth"
)

type Claims struct {
	ID   int    `json:"id"`
	Name string `json:"name"`
	jwt.RegisteredClaims
}

// GetEmpleadoIdFromToken 从 JWT 中提取 "id" 字段（int 类型）
func GetEmpleadoIdFromToken(tokenString string) (int, error) {
	// 去掉 Bearer 前缀
	tokenString = strings.TrimSpace(strings.Replace(tokenString, "Bearer", "", 1))

	// 解析 token
	token, err := jwt.Parse(tokenString, func(token *jwt.Token) (interface{}, error) {
		// 检查签名算法
		if _, ok := token.Method.(*jwt.SigningMethodHMAC); !ok {
			return 0, errors.New("unexpected signing method")
		}
		return []byte(SIGN_KEY_EMPLEADO_AUTH), nil
	})

	if err != nil {
		return 0, err
	}

	if !token.Valid {
		return 0, errors.New("invalid token")
	}

	// 提取 claims
	claims, ok := token.Claims.(jwt.MapClaims)
	if !ok {
		return 0, errors.New("invalid claims format")
	}

	// 获取 id 字段
	idVal, ok := claims["id"]
	if !ok {
		return 0, errors.New("id field not found in token")
	}

	idFloat, ok := idVal.(float64)
	if !ok {
		return 0, errors.New("id field is not a number")
	}

	id := int(idFloat)
	return id, nil
}

func BuildJwtToken(payloads map[string]interface{}, expirationMillis int64) (string, error) {
	claims := jwt.MapClaims{}

	// 加入 claims
	for k, v := range payloads {
		claims[k] = v
	}

	// 固定字段
	claims["sub"] = "Backend Server"
	claims["exp"] = time.Now().Add(time.Duration(expirationMillis) * time.Millisecond).Unix()

	// 创建 token
	token := jwt.NewWithClaims(jwt.SigningMethodHS512, claims)

	// 签名
	signedToken, err := token.SignedString([]byte(SIGN_KEY_EMPLEADO_AUTH))
	if err != nil {
		return "", err
	}

	return signedToken, nil
}
