package controller

import (
	"github.com/gin-gonic/gin"
	"log"
	"net/http"
	"supermarket-go/internal/model"
	"supermarket-go/internal/util"
)

type StockController struct {
	StockService    model.StockService
	ArticuloService model.ArticuloService
}

func RegisterStockRoutes(r *gin.Engine, stockSvc model.StockService, articuloSvc model.ArticuloService) {
	log.Println("[INFO] 注册 stock 路由组: /stock")
	ctrl := &StockController{StockService: stockSvc, ArticuloService: articuloSvc}
	g := r.Group("/stock")
	{
		g.GET("/getArticuloStock", ctrl.GetArticuloStock)
	}
	log.Println("[INFO] stock 路由注册完成")
}

// GetArticuloStock 获取商品库存信息
func (ctrl *StockController) GetArticuloStock(c *gin.Context) {
	log.Println("[INFO] ===== 开始处理获取商品库存请求 =====")

	// 获取Authorization header
	token := c.GetHeader("Authorization")
	if token == "" {
		log.Printf("[ERROR] 缺少Authorization header")
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	log.Printf("[INFO] 验证token: %s", token[:min(len(token), 20)]+"...")

	// 验证token并获取员工ID
	empleadoId, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		log.Printf("[ERROR] token验证失败: %v", err)
		c.JSON(http.StatusForbidden, util.BuildResponse(403, "验证信息无效", ""))
		return
	}

	log.Printf("[INFO] token验证成功，员工ID: %s", empleadoId)

	// 获取查询参数
	articuloId := c.Query("articuloId")
	likeStr := c.Query("like")

	log.Printf("[INFO] 查询参数: articuloId=%s, like=%s", articuloId, likeStr)

	// 解析like参数
	like := false
	if likeStr == "true" {
		like = true
	}

	var articulo *model.Articulo
	if like {
		log.Printf("[INFO] 使用模糊查询模式")
		articulos := ctrl.ArticuloService.GetByArticuloIDOrCodigoBarra(articuloId)
		if len(articulos) > 0 {
			articulo = &articulos[0]
			log.Printf("[INFO] 模糊查询找到商品: ArticuloID=%s", articulo.ArticuloID)
		} else {
			log.Printf("[WARN] 模糊查询未找到商品: articuloId=%s", articuloId)
		}
	} else {
		log.Printf("[INFO] 使用精确查询模式")
		articulos := ctrl.ArticuloService.GetByArticuloIDOrCodigoBarraActually(articuloId)
		if len(articulos) > 0 {
			articulo = &articulos[0]
			log.Printf("[INFO] 精确查询找到商品: ArticuloID=%s", articulo.ArticuloID)
		} else {
			log.Printf("[WARN] 精确查询未找到商品: articuloId=%s", articuloId)
		}
	}

	if articulo == nil {
		log.Printf("[ERROR] 商品不存在: articuloId=%s", articuloId)
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("articulo: ["+articuloId+"] not found", nil))
		return
	}

	log.Printf("[INFO] 查询商品库存: ArticuloID=%s", articulo.ArticuloID)
	stock := ctrl.StockService.GetByArticuloId(articulo.ArticuloID)
	if stock == nil {
		log.Printf("[ERROR] 库存记录不存在: ArticuloID=%s", articulo.ArticuloID)
		c.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("stock not found", nil))
		return
	}

	log.Printf("[INFO] 成功获取库存信息: ArticuloID=%s, Stock=%.2f",
		articulo.ArticuloID, stock.Stock)
	log.Println("[INFO] ===== 获取商品库存请求处理完成 =====")
	c.JSON(http.StatusOK, util.BuildSuccessResponseData(stock))
}

// min 辅助函数
func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}
