package controller

import (
	"encoding/json"
	"fmt"
	"github.com/gin-gonic/gin"
	"gorm.io/gorm"
	"log"
	"net/http"
	"supermarket-go/internal/model"
	"time"
)

type StreamController struct {
	db *gorm.DB
}

func RegisterStreamRoutes(r *gin.Engine, db *gorm.DB) {
	log.Println("[INFO] 注册 stock 路由组: /stock")
	ctrl := &StreamController{db: db}
	g := r.Group("/")
	{
		g.GET("/stream_articulos_all_jsonl", ctrl.StreamAllArticulosAsJSONL)
		g.GET("/stream_articuloextra_since_jsonl", ctrl.StreamArticulosExtraSinceAsJSONL)
	}
	log.Println("[INFO] stock 路由注册完成")
}

// StreamAllArticulosAsJSONL 将所有 articulos 作为 JSON Lines 流式传输（使用游标遍历）
func (ctrl *StreamController) StreamAllArticulosAsJSONL(c *gin.Context) {
	log.Println("收到将所有 articulos 作为 JSON Lines 流式传输的请求")

	// 为 JSON Lines 文件下载设置头部
	c.Header("Content-Description", "File Transfer")
	c.Header("Content-Disposition", "attachment; filename=articulos.jsonl")
	c.Header("Content-Type", "application/x-ndjson; charset=utf-8")

	// 游标遍历参数
	const batchSize = 1000 // 每批处理的记录数
	offset := 0
	rowCount := 0

	for {
		// 使用 GORM 分页查询
		var articulos []model.Articulo
		result := ctrl.db.Limit(batchSize).Offset(offset).Find(&articulos)
		if result.Error != nil {
			log.Printf("查询 articulos 时出错: %v", result.Error)
			c.String(http.StatusInternalServerError, fmt.Sprintf("查询 articulos 时出错: %v", result.Error))
			return
		}

		// 如果没有更多记录，退出循环
		if len(articulos) == 0 {
			break
		}

		// 处理当前批次的记录
		for _, articulo := range articulos {
			// 将 articulo 结构体序列化为 JSON
			jsonBytes, err := json.Marshal(articulo)
			if err != nil {
				log.Printf("序列化记录为 JSON 时出错: %v", err)
				continue
			}

			// 写入 JSON 数据
			if _, err := c.Writer.Write(jsonBytes); err != nil {
				log.Printf("写入 JSON 数据时出错: %v", err)
				return // 写入错误时中止流
			}
			// 写入换行符
			if _, err := c.Writer.WriteString("\n"); err != nil {
				log.Printf("写入换行符时出错: %v", err)
				return // 写入错误时中止流
			}

			rowCount++
		}

		// 每批次后 flush 一次，以确保数据逐步发送
		if flusher, ok := c.Writer.(http.Flusher); ok {
			flusher.Flush()
		}

		log.Printf("已处理 %d 条记录，当前批次: %d 条", rowCount, len(articulos))

		// 如果返回的记录数小于批次大小，说明已经是最后一批
		if len(articulos) < batchSize {
			break
		}

		// 更新偏移量
		offset += batchSize
	}

	// 确保所有剩余数据都被写入
	if flusher, ok := c.Writer.(http.Flusher); ok {
		flusher.Flush()
	}

	log.Printf("成功将 %d 条 articulos 记录作为 JSON Lines 流式传输", rowCount)
}

// StreamArticulosExtraSinceAsJSONL 将 'articuloextra' 表中在指定时间之后的 ArticuloID 对应的 articulo 表记录作为 JSON Lines 文件流式传输。
// 时间参数 'since' 应为 RFC3339 格式 (例如: 2023-01-01T15:04:05Z)
func (ctrl *StreamController) StreamArticulosExtraSinceAsJSONL(c *gin.Context) {
	sinceTimeStr := c.Query("since")
	if sinceTimeStr == "" {
		log.Println("请求缺少 'since' 时间参数")
		c.String(http.StatusBadRequest, "请求缺少 'since' 时间参数 (应为 RFC3339 格式)")
		return
	}

	// 解析时间参数
	sinceTime, err := time.Parse(time.RFC3339, sinceTimeStr)
	if err != nil {
		log.Printf("解析 'since' 时间参数时出错: %v", err)
		c.String(http.StatusBadRequest, fmt.Sprintf("无效的 'since' 时间参数格式 (应为 RFC3339): %v", err))
		return
	}

	// 加载西班牙时区
	spainLocation, err := time.LoadLocation("Europe/Madrid")
	if err != nil {
		log.Printf("加载西班牙时区失败: %v", err)
		c.String(http.StatusInternalServerError, fmt.Sprintf("时区配置错误: %v", err))
		return
	}

	// 将输入时间转换为西班牙时区
	sinceTimeSpain := sinceTime.In(spainLocation)

	log.Printf("收到将 articuloextra 从 %s (西班牙时区: %s) 开始的 ArticuloID 对应的 articulo 数据作为 JSON Lines 流式传输的请求",
		sinceTime.Format(time.RFC3339), sinceTimeSpain.Format(time.RFC3339))

	// 第一步：从 articuloextra 表获取指定时间后的 ArticuloID 列表
	// 使用 GORM 查询
	var articuloExtras []model.ArticuloExtra
	result := ctrl.db.Select("DISTINCT ArticuloID").
		Where("UpdateTimeStamp > ?", sinceTimeSpain).
		Find(&articuloExtras)
	if result.Error != nil {
		log.Printf("查询 articuloextra 时出错: %v", result.Error)
		c.String(http.StatusInternalServerError, fmt.Sprintf("查询 articuloextra 时出错: %v", result.Error))
		return
	}

	// 收集所有 ArticuloID
	var articuloIDs []string
	for _, extra := range articuloExtras {
		articuloIDs = append(articuloIDs, extra.ArticuloID)
	}

	if len(articuloIDs) == 0 {
		log.Println("未找到指定时间后更新的 ArticuloID")
		c.String(http.StatusNotFound, fmt.Sprintf("未找到 %s (西班牙时区) 之后更新的 ArticuloID", sinceTimeSpain.Format(time.RFC3339)))
		return
	}

	log.Printf("从 articuloextra 表中找到 %d 个在 %s (西班牙时区) 之后更新的不同 ArticuloID", len(articuloIDs), sinceTimeSpain.Format(time.RFC3339))

	// 设置响应头 - 使用西班牙时区的时间格式化文件名
	c.Header("Content-Description", "File Transfer")
	c.Header("Content-Disposition", "attachment; filename=articulos_by_extra_since_"+sinceTimeSpain.Format("20060102150405")+"_CET.jsonl")
	c.Header("Content-Type", "application/x-ndjson; charset=utf-8")

	// 第二步：查询 articulo 表中对应的数据
	// 使用 GORM 查询
	var articulos []model.Articulo
	result = ctrl.db.Where("ArticuloID IN ?", articuloIDs).Find(&articulos)
	if result.Error != nil {
		log.Printf("查询 articulo 表时出错: %v", result.Error)
		c.String(http.StatusInternalServerError, fmt.Sprintf("查询 articulo 表时出错: %v", result.Error))
		return
	}

	rowCount := 0
	for _, articulo := range articulos {
		// 将 articulo 结构体序列化为 JSON
		jsonBytes, err := json.Marshal(articulo)
		if err != nil {
			log.Printf("序列化记录为 JSON 时出错: %v", err)
			continue
		}

		// 写入 JSON 数据
		if _, err := c.Writer.Write(jsonBytes); err != nil {
			log.Printf("写入 JSON 数据时出错: %v", err)
			return
		}
		// 写入换行符
		if _, err := c.Writer.WriteString("\n"); err != nil {
			log.Printf("写入换行符时出错: %v", err)
			return
		}

		rowCount++
		// 每 100 行 flush 一次，以确保数据逐步发送
		if rowCount%100 == 0 {
			if flusher, ok := c.Writer.(http.Flusher); ok {
				flusher.Flush()
			}
		}
	}

	// 确保所有剩余数据都被写入
	if flusher, ok := c.Writer.(http.Flusher); ok {
		flusher.Flush()
	}

	log.Printf("成功将 %d 条 articulo 记录作为 JSON Lines 流式传输 (基于 %s (西班牙时区) 之后更新的 articuloextra 的 ArticuloID)", rowCount, sinceTimeSpain.Format(time.RFC3339))
}
