package service

import (
	"errors"
	"gorm.io/gorm"
	"log"
	"supermarket-go/internal/model"
	"time"
)

type OrderServiceImpl struct {
	db                       *gorm.DB
	articuloService          model.ArticuloService
	articuloClaseService     model.ArticuloClaseService
	ivaTipoService           model.IvaTipoService
	stockService             model.StockService
	ticketDiarioTotalService model.TicketDiarioTotalService
	procedureService         model.ProcedureService
}

func NewOrderService(db *gorm.DB) model.OrderService {
	return &OrderServiceImpl{
		db:                       db,
		articuloService:          NewArticuloService(db),
		articuloClaseService:     NewArticuloClaseService(db),
		ivaTipoService:           NewIvaTipoService(db),
		stockService:             NewStockService(db),
		ticketDiarioTotalService: NewTicketDiarioTotalService(db),
		procedureService:         NewProcedureService(db),
	}
}

// GetTicketById 通过订单号获取订单
func (s *OrderServiceImpl) GetTicketById(ticketId int64) *model.NestedTicket {
	log.Printf("[INFO] 开始获取订单: TicketID=%d", ticketId)

	var ticket model.Ticket
	if err := s.db.Where("DocumentoNo = ?", ticketId).First(&ticket).Error; err != nil {
		log.Printf("[ERROR] 获取订单失败: %v", err)
		return nil
	}

	var details []model.TicketArticulo
	if err := s.db.Where("DocumentoNo = ?", ticketId).Find(&details).Error; err != nil {
		log.Printf("[ERROR] 获取订单详情失败: %v", err)
		return nil
	}

	return &model.NestedTicket{
		Ticket:  ticket,
		Details: details,
	}
}

// GetAllSuspendedTickets 获取所有挂单
func (s *OrderServiceImpl) GetAllSuspendedTickets(detailed bool) []model.NestedTicketColgando {
	log.Printf("[INFO] 开始获取所有挂单, detailed=%v", detailed)

	var colgandos []model.TicketColgando
	if err := s.db.Order("fecha DESC").Find(&colgandos).Error; err != nil {
		log.Printf("[ERROR] 获取挂单列表失败: %v", err)
		return nil
	}

	var result []model.NestedTicketColgando
	for _, colgando := range colgandos {
		nested := model.NestedTicketColgando{
			TicketColgando: colgando,
		}

		if detailed {
			var details []model.TicketColgandoArticulo
			if err := s.db.Where("DocumentoNo = ?", colgando.DocumentoNo).Find(&details).Error; err != nil {
				log.Printf("[ERROR] 获取挂单详情失败: %v", err)
			} else {
				nested.Details = details
			}
		}

		result = append(result, nested)
	}

	log.Printf("[INFO] 成功获取 %d 个挂单", len(result))
	return result
}

// GetSuspendedTicket 通过挂单号获取挂单
func (s *OrderServiceImpl) GetSuspendedTicket(documentoNo int64, detailed bool) *model.NestedTicketColgando {
	log.Printf("[INFO] 开始获取挂单: DocumentoNo=%d, detailed=%v", documentoNo, detailed)

	var colgando model.TicketColgando
	if err := s.db.Where("documentoNo = ?", documentoNo).First(&colgando).Error; err != nil {
		log.Printf("[ERROR] 获取挂单失败: %v", err)
		return nil
	}

	result := &model.NestedTicketColgando{
		TicketColgando: colgando,
	}

	if detailed {
		var details []model.TicketColgandoArticulo
		if err := s.db.Where("documentoNo = ?", documentoNo).Order("ordenNo").Find(&details).Error; err != nil {
			log.Printf("[ERROR] 获取挂单详情失败: %v", err)
		} else {
			result.Details = details
		}
	}

	return result
}

// SubmitSuspendedTicket 提交挂单
func (s *OrderServiceImpl) SubmitSuspendedTicket(empleadoId int, data model.OrderDTO) *int64 {
	log.Printf("[INFO] 开始提交挂单: EmpleadoID=%d, TicketNo=%d", empleadoId, data.TicketNo)

	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()
	var existingColgando bool
	if data.TicketNo > 0 {
		var colgando model.TicketColgando
		err := s.db.Where("DocumentoNo = ?", data.TicketNo).First(&colgando).Error
		if errors.Is(err, gorm.ErrRecordNotFound) {
			// 没有找到记录，执行某些逻辑
			existingColgando = false
		} else if err != nil {
			// 查询出错
			log.Println("查询出错:", err)
		} else {
			existingColgando = true
		}
	}

	// 计算总价和折扣 - 与Java版本保持一致
	originalTotal := data.CalculateOriginalTotal()
	discountedTotal := data.CalculateDiscountedTotal(data.AdditionalDiscount)
	total := originalTotal - discountedTotal

	// 计算数量总数和折扣总额
	var cantidadTotal float64
	var discountTotal float64
	for _, item := range data.Items {
		cantidadTotal += float64(item.Amount)
		originalItemTotal := item.Price * float64(item.Amount)
		discountTotal += originalItemTotal * (item.Discount + data.AdditionalDiscount*100)
	}

	// 创建或更新挂单
	var colgando model.TicketColgando
	if existingColgando {
		// 删除现有商品详情
		if err := tx.Where("documentoNo = ?", data.TicketNo).Delete(&model.TicketColgandoArticulo{}).Error; err != nil {
			log.Printf("[ERROR] 删除挂单商品详情失败: %v", err)
			tx.Rollback()
			return nil
		}
		// 更新挂单信息 - 与Java版本保持一致
		colgando.Total = total
		colgando.LineaTotal = len(data.Items)
		colgando.Descuento = data.AdditionalDiscount * 100
		colgando.DescuentoTotal = discountTotal
		colgando.CantidadTotal = cantidadTotal
		colgando.Fecha = time.Now()
		colgando.FechaEntrada = time.Now()

		if err := tx.Save(&colgando).Error; err != nil {
			log.Printf("[ERROR] 更新挂单失败: %v", err)
			tx.Rollback()
			return nil
		}
	} else {
		// 创建新挂单 - 与Java版本保持一致
		colgando = model.TicketColgando{
			Total:          total,
			LineaTotal:     len(data.Items),
			Descuento:      data.AdditionalDiscount * 100, // 转换为百分比
			DescuentoTotal: discountTotal,
			CantidadTotal:  cantidadTotal,
			EmpresaID:      data.EmpresaID,
			OperadorID:     uint16(empleadoId),
			Fecha:          time.Now(),
			FechaEntrada:   time.Now(),
		}

		if err := tx.Create(&colgando).Error; err != nil {
			log.Printf("[ERROR] 创建挂单失败: %v", err)
			tx.Rollback()
			return nil
		}
	}

	// 创建商品详情
	for i, item := range data.Items {
		articulo := item.MapToColgandoArticulo(i)
		articulo.DocumentoNo = uint64(colgando.DocumentoNo)

		if err := tx.Create(&articulo).Error; err != nil {
			log.Printf("[ERROR] 创建挂单商品详情失败: %v", err)
			tx.Rollback()
			return nil
		}
	}
	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("[ERROR] 提交事务失败: %v", err)
		return nil
	}

	log.Printf("[INFO] 成功提交挂单: DocumentoNo=%d", colgando.DocumentoNo)
	return &colgando.DocumentoNo
}

// DeleteSuspendedTicket 删除挂单
func (s *OrderServiceImpl) DeleteSuspendedTicket(documentoNo int64) bool {
	log.Printf("[INFO] 开始删除挂单: DocumentoNo=%d", documentoNo)

	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 删除商品详情
	if err := tx.Where("documentoNo = ?", documentoNo).Delete(&model.TicketColgandoArticulo{}).Error; err != nil {
		log.Printf("[ERROR] 删除挂单商品详情失败: %v", err)
		tx.Rollback()
		return false
	}

	// 删除挂单
	if err := tx.Where("documentoNo = ?", documentoNo).Delete(&model.TicketColgando{}).Error; err != nil {
		log.Printf("[ERROR] 删除挂单失败: %v", err)
		tx.Rollback()
		return false
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("[ERROR] 提交事务失败: %v", err)
		return false
	}

	log.Printf("[INFO] 成功删除挂单: DocumentoNo=%d", documentoNo)
	return true
}

// CheckOutSuspendedTicket 按挂单号结算
func (s *OrderServiceImpl) CheckOutSuspendedTicket(empleadoId int, documentoNo int64, paid []float64) *int64 {
	log.Printf("[INFO] 开始结算挂单: EmpleadoID=%d, DocumentoNo=%d", empleadoId, documentoNo)

	// 获取挂单信息
	colgando := s.GetSuspendedTicket(documentoNo, true)
	if colgando == nil {
		log.Printf("[ERROR] 挂单不存在: DocumentoNo=%d", documentoNo)
		return nil
	}

	return s.CheckOutSuspendedTicketByData(empleadoId, *colgando, paid)
}

// CheckOutSuspendedTicketByData 结算挂单数据
func (s *OrderServiceImpl) CheckOutSuspendedTicketByData(empleadoId int, colgando model.NestedTicketColgando, paid []float64) *int64 {
	log.Printf("[INFO] 开始结算挂单数据: EmpleadoID=%d, DocumentoNo=%d", empleadoId, colgando.TicketColgando.DocumentoNo)

	// 开始事务
	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	// 确保paid数组有3个元素
	if len(paid) < 3 {
		paid = append(paid, make([]float64, 3-len(paid))...)
	}

	// 生成新的订单号 - 与Java版本保持一致
	dateStr := time.Now().Format("2006-01-02")
	generatedTicketNo := s.procedureService.GenerateTicketNo(dateStr)

	// 确定付款方式 - 与Java版本保持一致
	var cobrarTipo int
	var formaDePago string
	if paid[0] > 0 && paid[1] == 0.0 && paid[2] == 0.0 {
		// 现金
		cobrarTipo = 0
		formaDePago = "Efectivo"
	} else if paid[0] == 0.0 && paid[1] > 0 && paid[2] == 0.0 {
		// 刷卡
		cobrarTipo = 1
		formaDePago = "Tarjeta"
	} else if paid[0] == 0.0 && paid[1] == 0.0 && paid[2] > 0 {
		// 代金券
		cobrarTipo = 5
		formaDePago = "Vale"
	} else {
		// 组合付款
		cobrarTipo = 11
		formaDePago = "Multi"
	}

	// 创建订单 - 与Java版本保持一致
	ticket := model.Ticket{
		DocumentoNo:            generatedTicketNo,
		Total:                  colgando.TicketColgando.Total,
		LineaTotal:             colgando.TicketColgando.LineaTotal,
		Descuento:              colgando.TicketColgando.Descuento,
		DescuentoTotal:         colgando.TicketColgando.DescuentoTotal,
		EmpresaID:              colgando.TicketColgando.EmpresaID,
		Fecha:                  time.Now(),
		FechaEntrada:           time.Now(),
		TiempoUltimoActualizar: time.Now(),
		Efectivo:               paid[0] + paid[1] + paid[2], // 总付款金额
		CobrarTipo:             uint8(cobrarTipo),
		FormaDePago:            formaDePago,
		OperadorID:             uint16(empleadoId),
	}

	if err := tx.Create(&ticket).Error; err != nil {
		log.Printf("[ERROR] 创建订单失败: %v", err)
		tx.Rollback()
		return nil
	}

	// 创建付款记录 - 与Java版本保持一致
	for i, amount := range paid {
		if amount > 0 {
			var formaDePagoStr string
			switch i {
			case 0:
				formaDePagoStr = "Efectivo"
			case 1:
				formaDePagoStr = "Tarjeta"
			case 2:
				formaDePagoStr = "Vale"
			default:
				formaDePagoStr = "Multi"
			}

			ticketCobro := model.TicketCobro{
				DocumentoNo: uint64(generatedTicketNo),
				Fecha:       time.Now(),
				Importe:     amount,
				FormaDePago: formaDePagoStr,
				OperadorID:  uint16(empleadoId),
			}

			if err := tx.Create(&ticketCobro).Error; err != nil {
				log.Printf("[ERROR] 创建付款记录失败: %v", err)
				tx.Rollback()
				return nil
			}
		}
	}

	// 创建订单商品详情 - 与Java版本保持一致
	for _, detail := range colgando.Details {
		// 获取税率详情 - 与Java版本保持一致
		actualArticulos := s.articuloService.GetByArticuloIDOrCodigoBarraActually(detail.ArticuloID)
		if len(actualArticulos) == 0 {
			log.Printf("[ERROR] 无法获取商品详情: %s, 结算失败", detail.ArticuloID)
			tx.Rollback()
			return nil
		}
		actualArticulo := actualArticulos[0]

		category := s.articuloClaseService.GetCategoryById(int(actualArticulo.ClaseID))
		if category == nil {
			log.Printf("[ERROR] 无法获取商品 %s 的分类: %d, 结算失败", detail.ArticuloID, actualArticulo.ClaseID)
			tx.Rollback()
			return nil
		}

		iva := s.ivaTipoService.GetIvaById(int(category.IVAID))
		if iva == nil {
			log.Printf("[ERROR] 无法获取商品分类 %d 的 IVA 详情: %d, 结算失败", category.ClaseID, category.IVAID)
			tx.Rollback()
			return nil
		}
		comentario := "0"
		ticketArticulo := model.TicketArticulo{
			DocumentoNo: uint64(generatedTicketNo),
			ArticuloID:  detail.ArticuloID,
			NombreES:    detail.NombreES,
			NombreCN:    detail.NombreCN,
			Precio:      detail.Precio,
			Cantidad:    detail.Cantidad,
			OrdenNo:     int(detail.OrdenNo),
			Descuento:   detail.Descuento,
			Comentario:  &comentario,
			IVA:         iva.IVA,
			REQ:         iva.REQ,
		}

		if err := tx.Create(&ticketArticulo).Error; err != nil {
			log.Printf("[ERROR] 创建订单商品详情失败: %v", err)
			tx.Rollback()
			return nil
		}

		// 更新库存 - 与Java版本保持一致
		if detail.ArticuloID != "" && detail.Cantidad > 0 {
			// 注意：这里传入的是负数，因为销售会减少库存
			// Java版本传入正数可能是错误的，这里我们传入负数来减少库存
			deltaStock := -int(detail.Cantidad)
			if !s.stockService.SaveOrUpdateStock(detail.ArticuloID, deltaStock) {
				log.Printf("[ERROR] 无法更新商品 %s 的库存, 结算数量: %.2f, 库存更新失败", detail.ArticuloID, detail.Cantidad)
				tx.Rollback()
				return nil
			}
		}
	}

	// 删除挂单和挂单商品详情
	if colgando.TicketColgando.DocumentoNo > 0 {
		if err := tx.Where("documentoNo = ?", colgando.TicketColgando.DocumentoNo).Delete(&model.TicketColgandoArticulo{}).Error; err != nil {
			log.Printf("[ERROR] 删除挂单商品详情失败: %v", err)
			tx.Rollback()
			return nil
		}

		if err := tx.Where("documentoNo = ?", colgando.TicketColgando.DocumentoNo).Delete(&model.TicketColgando{}).Error; err != nil {
			log.Printf("[ERROR] 删除挂单失败: %v", err)
			tx.Rollback()
			return nil
		}
	}

	// 更新日总额统计 - 与Java版本保持一致
	if !s.ticketDiarioTotalService.RecordTicket(paid) {
		log.Printf("[ERROR] 无法更新日总额统计: %d", generatedTicketNo)
		tx.Rollback()
		return nil
	}

	// 提交事务
	if err := tx.Commit().Error; err != nil {
		log.Printf("[ERROR] 提交事务失败: %v", err)
		return nil
	}

	log.Printf("[INFO] 成功结算挂单: TicketID=%d", generatedTicketNo)
	return &generatedTicketNo
}
