package service

import (
	"crypto/md5"
	"fmt"
	"gorm.io/gorm"
	"log"
	"math"
	"strconv"
	"supermarket-go/internal/model"
	"sync"
	"time"
)

type ReturnGoodsService struct {
	db                   *gorm.DB
	articuloService      model.ArticuloService
	procedureService     *ProcedureService
	articuloClaseService model.ArticuloClaseService
	ivaTipoService       model.IvaTipoService
}

// 用于防止重复提交的全局列表
var returnGoodsProcessList = make(map[string]bool)
var processListMutex sync.RWMutex

// ArticuloWithCantidad 对应Java的ArticuloWithCantidad内部类
type ArticuloWithCantidad struct {
	Articulo *model.Articulo
	Nombre   string
	Precio   float64
	Cantidad int
	NombreES string
	NombreCN string
	ClassID  int
}

// ReturnedArticulosCalculationResult 对应Java的ReturnedArticulosCalculationResult内部类
type ReturnedArticulosCalculationResult struct {
	Data          []model.AlbaranProveedorArticulo
	CantidadTotal float64
	Total         float64
}

func NewReturnGoodsService(db *gorm.DB, articuloService model.ArticuloService, procedureService *ProcedureService,
	articuloClaseService model.ArticuloClaseService, ivaTipoService model.IvaTipoService) *ReturnGoodsService {
	return &ReturnGoodsService{
		db:                   db,
		articuloService:      articuloService,
		procedureService:     procedureService,
		articuloClaseService: articuloClaseService,
		ivaTipoService:       ivaTipoService,
	}
}

// ListAllReturnedOrder 列出所有退货的单据
func (s *ReturnGoodsService) ListAllReturnedOrder(code string, bloqueado *string, documentoNo *int64) []model.Documento {
	var orders []model.AlbaranProveedorDto
	query := `
		SELECT alp.*, prov.NombreES as ProNombreES 
		FROM albaranproveedor AS alp, proveedor AS prov
		WHERE 1 = 1 AND alp.EmpresaID = prov.EmpresaID
	`
	params := []interface{}{}

	if code != "" {
		query += ` AND (alp.EmpresaID LIKE ? OR prov.NombreES LIKE ? OR prov.NombreCN LIKE ?)`
		params = append(params, "%"+code+"%", "%"+code+"%", "%"+code+"%")
	}

	if bloqueado != nil {
		query += ` AND alp.Bloqueado = ?`
		params = append(params, *bloqueado)
	}

	query += ` ORDER BY alp.DocumentoNo DESC`

	if err := s.db.Raw(query, params...).Scan(&orders).Error; err != nil {
		return nil
	}

	// 过滤总金额小于等于0的记录（退货单）
	var filteredOrders []model.Documento
	for _, order := range orders {
		if order.Total <= 0 {
			if documentoNo != nil && *documentoNo > 0 {
				if order.DocumentoNo == *documentoNo {
					filteredOrders = append(filteredOrders, order.ConvertToDocumento())
				}
			} else {
				filteredOrders = append(filteredOrders, order.ConvertToDocumento())
			}
		}
	}

	return filteredOrders
}

// ListAllArticulosByDocumentoNo 列出所有退货单中的商品
func (s *ReturnGoodsService) ListAllArticulosByDocumentoNo(documentoNo string) []model.AlbaranProveedorArticulo {
	var articulos []model.AlbaranProveedorArticulo
	s.db.Where("DocumentoNo = ?", documentoNo).Find(&articulos)
	return articulos
}

// IsOperadorOwner 检查某个职员是否是某个单据的操作员
func (s *ReturnGoodsService) IsOperadorOwner(documentoNo int64, empleadoId int) bool {
	var albaranproveedor model.AlbaranProveedor
	if err := s.db.Where("DocumentoNo = ?", documentoNo).First(&albaranproveedor).Error; err != nil {
		return false
	}
	return albaranproveedor.OperadorID == int64(empleadoId)
}

// SubmitReturnGoods 提交退货单数据
func (s *ReturnGoodsService) SubmitReturnGoods(empleadoId int, data model.ReturnGoodsListDTO, flag bool) bool {
	// 生成签名防止重复提交
	sign := fmt.Sprintf("e:%s,d:", data.EmpresaId)
	for _, articulo := range data.ArticuloList {
		sign += fmt.Sprintf("|%d|%s|%s|%d|%f", articulo.Type, articulo.Value, articulo.Nombre, articulo.Amount, articulo.Precio)
	}
	log.Printf("[INFO] sign:%v", sign)
	processListMutex.Lock()
	if returnGoodsProcessList[sign] {
		processListMutex.Unlock()
		log.Printf("[WARN] 已经有一个相同的提交进货/退货单操作正在进行")
		return false
	}
	returnGoodsProcessList[sign] = true
	processListMutex.Unlock()

	defer func() {
		processListMutex.Lock()
		delete(returnGoodsProcessList, sign)
		processListMutex.Unlock()
	}()

	if empleadoId == 0 {
		// 若操作职员ID为空 则默认为1号
		empleadoId = 1
	}

	log.Printf("[INFO] 开始处理退货单数据.....")
	empresaId := data.EmpresaId

	// 合并商品数据
	megeredArticulos := s.megerArticulo(data)
	for _, e := range megeredArticulos {
		log.Printf("商品: [%s/%s]: -%d 个", e.Articulo.ArticuloID, e.Articulo.CodigoBarra, e.Cantidad)
	}

	// 产生新的 DocumentoNo
	formattedDate := time.Now().Format("2006-01-02")
	documentoNo := s.procedureService.GenerateDocumentoNo("AlbaranProveedor", formattedDate)

	// 插入详细退货记录
	counter := 0
	cantidadTotal := 0.0
	total := 0.0

	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	for _, e := range megeredArticulos {
		albaranproveedorarticulo := s.nestedToAlbaranProveedorArticulo(documentoNo, e, counter)
		if flag {
			albaranproveedorarticulo.Precio = math.Abs(albaranproveedorarticulo.Precio)
			albaranproveedorarticulo.Cantidad = math.Abs(albaranproveedorarticulo.Cantidad)
		}

		if err := tx.Create(&albaranproveedorarticulo).Error; err != nil {
			tx.Rollback()
			return false
		}
		cantidad := float64(e.Cantidad)
		if cantidad >= 0 {
			cantidadTotal += cantidad
		} else {
			cantidadTotal -= cantidad
		}
		if flag {
			cantidadTotal = math.Abs(cantidadTotal)
		}
		totalNum := albaranproveedorarticulo.Precio * float64(e.Cantidad)
		if totalNum < 0 {
			total += totalNum
		} else {
			if flag {
				total += totalNum
			} else {
				total -= totalNum
			}
		}
		//total += math.Abs(albaranproveedorarticulo.Precio) * math.Abs(float64(e.Cantidad))
		//if flag {
		//	total = total * 1.0
		//} else {
		//	total = total * -1.0
		//}

		counter++
	}
	i, err := strconv.ParseInt(empresaId, 10, 64)
	if err != nil {
		log.Printf("[ERROR] empresaId:%v 转换INT 失败", empresaId)
		return false
	}
	// 插入退货单据记录
	albaranproveedor := model.AlbaranProveedor{
		DocumentoNo:            documentoNo,
		EmpresaID:              i,
		Total:                  total,
		CantidadTotal:          cantidadTotal,
		LineaTotal:             counter, // counter 从0开始
		Fecha:                  time.Now(),
		FechaEntrada:           time.Now(),
		TiempoUltimoActualizar: time.Now(),
		OperadorID:             int64(empleadoId),
		AlmacenID:              1,
		PrecioUsado:            10,
	}

	if err := tx.Create(&albaranproveedor).Error; err != nil {
		tx.Rollback()
		return false
	}

	tx.Commit()
	log.Printf("[INFO] 处理退货单数据完成.....")
	return true
}

// megerArticulo 合并商品数据，对应Java的megerArticulo方法
func (s *ReturnGoodsService) megerArticulo(data model.ReturnGoodsListDTO) []ArticuloWithCantidad {
	// 合并相同的数据 防止前端提交多个重复的商品
	articulosMap := make(map[string]model.ReturnGoodsListArticulo)

	for _, a := range data.ArticuloList {
		key := ""
		if a.Type == 0 && a.Value == "0" {
			// 不合并 0 号商品
			key = fmt.Sprintf("%d-%s-%s", a.Type, a.Value, generateUUID())
		} else {
			key = fmt.Sprintf("%d-%s", a.Type, a.Value)
		}

		if existing, exists := articulosMap[key]; exists {
			// 合并数量
			articulosMap[key] = model.ReturnGoodsListArticulo{
				Type:     existing.Type,
				Value:    existing.Value,
				Amount:   existing.Amount + a.Amount,
				Precio:   existing.Precio,
				Nombre:   existing.Nombre,
				NombreCN: existing.NombreCN,
				NombreES: existing.NombreES,
				ClassID:  existing.ClassID,
			}
		} else {
			articulosMap[key] = a
		}
	}

	// 获取商品详细信息
	var detailedArticulos []ArticuloWithCantidad
	for _, e := range articulosMap {
		var articulos []model.Articulo

		// 当条形码或商品编码为零/空：不查库，直接用名称与价格生成临时商品
		if e.Value == "0" || e.Value == "" {
			if e.Nombre != "" {
				art := model.Articulo{
					ArticuloID:  "0",
					CodigoBarra: "0",
					NombreES:    e.NombreES,
					NombreCN:    e.NombreCN,
					ClaseID:     e.ClassID,
				}
				articulos = []model.Articulo{art}
			} else {
				log.Printf("[WARN] 商品编码为零且商品名称为空，跳过此商品")
				continue
			}
		} else {
			// 正常情况：根据商品编码或条形码查询
			articulos = s.articuloService.GetByArticuloIDOrCodigoBarraActually(e.Value)
		}

		if len(articulos) > 0 {
			detailedArticulos = append(detailedArticulos, ArticuloWithCantidad{
				Articulo: &articulos[0],
				Cantidad: e.Amount,
				Precio:   e.Precio,
				Nombre:   e.Nombre,
				NombreCN: e.NombreCN,
				NombreES: e.NombreES,
				ClassID:  e.ClassID,
			})
		} else {
			log.Printf("[WARN] 未找到商品: Value=%s, Nombre=%s", e.Value, e.Nombre)
		}
	}

	// 最终合并完的商品列表
	finalMap := make(map[string]ArticuloWithCantidad)
	for _, e := range detailedArticulos {
		key := ""
		if e.Articulo.ArticuloID == "0" {
			key = e.Articulo.ArticuloID + generateUUID()
		} else {
			key = e.Articulo.ArticuloID
		}

		if existing, exists := finalMap[key]; exists {
			existing.Cantidad += e.Cantidad
			finalMap[key] = existing
		} else {
			finalMap[key] = e
		}
	}

	var result []ArticuloWithCantidad
	for _, v := range finalMap {
		result = append(result, v)
	}

	return result
}

// nestedToAlbaranProveedorArticulo 转换商品数据，对应Java的nestedToAlbaranProveedorArticulo方法
func (s *ReturnGoodsService) nestedToAlbaranProveedorArticulo(documentoNo int64, e ArticuloWithCantidad, ordenNo int) model.AlbaranProveedorArticulo {
	// 通过classID 查询 articuloclase，然后取结果中得IVAID字段，然后通过IVAID字段查询表ivatipo字段信息，取结果IVA和REQ写入albaranproveedorarticulo中
	albaranproveedorarticulo := model.AlbaranProveedorArticulo{
		DocumentoNo: documentoNo,
		ArticuloID:  e.Articulo.ArticuloID,
		CodigoBarra: e.Articulo.CodigoBarra,
		NombreES:    e.Articulo.NombreES,
		NombreCN:    e.Articulo.NombreCN,
		OrdenNo:     ordenNo,
		Temporal:    0,
	}

	// 通过classID查询articuloclase，获取IVAID
	if e.ClassID > 0 {
		articuloClase := s.articuloClaseService.GetCategoryById(e.ClassID)
		if articuloClase != nil {
			// 通过IVAID查询ivatipo，获取IVA和REQ
			ivaTipo := s.ivaTipoService.GetIvaById(int(articuloClase.IVAID))
			if ivaTipo != nil {
				albaranproveedorarticulo.IVA = ivaTipo.IVA
				albaranproveedorarticulo.REQ = ivaTipo.REQ
				log.Printf("[INFO] 成功设置商品 %s 的IVA=%.3f, REQ=%.3f", e.Articulo.ArticuloID, ivaTipo.IVA, ivaTipo.REQ)
			} else {
				log.Printf("[WARN] 未找到IVA税率信息: IVAID=%d", articuloClase.IVAID)
			}
		} else {
			log.Printf("[WARN] 未找到商品分类信息: ClassID=%d", e.ClassID)
		}
	} else {
		log.Printf("[WARN] 商品分类ID无效: ClassID=%d", e.ClassID)
	}

	// 设置商品名称
	//enName := e.Articulo.NombreES
	//if e.Nombre != "" {
	//	albaranproveedorarticulo.NombreES = e.Nombre
	//} else {
	//	albaranproveedorarticulo.NombreES = enName
	//}

	// 设置中文名称
	//cnName := e.Articulo.NombreCN
	//if cnName == "" {
	//	albaranproveedorarticulo.NombreCN = ""
	//} else {
	//	albaranproveedorarticulo.NombreCN = cnName
	//}

	// 设置数量（负数）
	albaranproveedorarticulo.Cantidad = -math.Abs(float64(e.Cantidad))

	// 设置价格
	if e.Precio > 0 {
		albaranproveedorarticulo.Precio = roundTo4Decimals(e.Precio)
		albaranproveedorarticulo.Temporal = 1
	} else {
		albaranproveedorarticulo.Precio = e.Articulo.PrecioDetalle
		albaranproveedorarticulo.Temporal = 0
	}

	return albaranproveedorarticulo
}

// AddReturnGoods 向退货单添加商品

func (s *ReturnGoodsService) AddReturnGoods(empleadoId int, documentoNo int64, data model.ReturnGoodsListDTO) bool {
	// 校验单据归属
	if !s.IsOperadorOwner(documentoNo, empleadoId) {
		return false
	}

	// 合并商品数据
	merged := s.megerArticulo(data)
	if len(merged) == 0 {
		return true
	}

	// 计算起始序号
	existing := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	startOrden := 0
	for _, it := range existing {
		if it.OrdenNo >= startOrden {
			startOrden = it.OrdenNo + 1
		}
	}

	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	orden := startOrden
	for _, e := range merged {
		row := s.nestedToAlbaranProveedorArticulo(documentoNo, e, orden)
		if err := tx.Create(&row).Error; err != nil {
			tx.Rollback()
			return false
		}
		orden++
	}

	// 回写单据汇总（保持退货为负）
	updated := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	calc := s.calculateReturnedArticulos(updated)

	var albaran model.AlbaranProveedor
	if err := tx.Where("DocumentoNo = ?", documentoNo).First(&albaran).Error; err != nil {
		tx.Rollback()
		return false
	}
	albaran.Total = -math.Abs(calc.Total)
	albaran.CantidadTotal = -math.Abs(calc.CantidadTotal)
	albaran.LineaTotal = len(updated)
	albaran.TiempoUltimoActualizar = time.Now()

	if err := tx.Save(&albaran).Error; err != nil {
		tx.Rollback()
		return false
	}

	tx.Commit()
	return true
}

// DeleteReturnedGoods 删除退货单
func (s *ReturnGoodsService) DeleteReturnedGoods(empleadoId int, documentoNo int64) bool {
	if !s.IsOperadorOwner(documentoNo, empleadoId) {
		// 非此单据的操作员不可操作
		return false
	}

	tx := s.db.Begin()
	defer func() {
		if r := recover(); r != nil {
			tx.Rollback()
		}
	}()

	if err := tx.Where("DocumentoNo = ?", documentoNo).Delete(&model.AlbaranProveedor{}).Error; err != nil {
		tx.Rollback()
		return false
	}

	if err := tx.Where("DocumentoNo = ?", documentoNo).Delete(&model.AlbaranProveedorArticulo{}).Error; err != nil {
		tx.Rollback()
		return false
	}

	tx.Commit()
	return true
}

// DeleteReturnedGood 删除退货单中的商品
func (s *ReturnGoodsService) DeleteReturnedGood(empleadoId int, documentoNo int64, code, name string) bool {
	if !s.IsOperadorOwner(documentoNo, empleadoId) {
		// 非此单据的操作员不可操作
		return false
	}

	// 如果code为空或为"0"，则根据商品名称进行查询
	if code == "" || code == "0" {
		return s.fxUpdateAlbaranProveedorWithCustomActionForReturnGoodsByName(documentoNo, name, func(list []model.AlbaranProveedorArticulo, e *model.AlbaranProveedorArticulo) {
			// 删除数据库记录
			s.db.Where("DocumentoNo = ? AND NombreES = ?", documentoNo, name).Delete(&model.AlbaranProveedorArticulo{})
		})
	}

	return s.fxUpdateAlbaranProveedorWithCustomActionForReturnGoods(documentoNo, code, func(list []model.AlbaranProveedorArticulo, e *model.AlbaranProveedorArticulo) {
		// 删除数据库记录
		if name != "" {
			s.db.Where("DocumentoNo = ? AND (ArticuloID = ? OR CodigoBarra = ?) AND NombreES = ?", documentoNo, code, code, name).Delete(&model.AlbaranProveedorArticulo{})
		} else {
			s.db.Where("DocumentoNo = ? AND (ArticuloID = ? OR CodigoBarra = ?)", documentoNo, code, code).Delete(&model.AlbaranProveedorArticulo{})
		}
	})
}

// UpdateReturnedGood 更新某个退货记录中的商品数量
func (s *ReturnGoodsService) UpdateReturnedGood(empleadoId int, documentoNo int64, code string, amount int, append bool, precio *float64, name string) bool {
	if !s.IsOperadorOwner(documentoNo, empleadoId) {
		// 非此单据的操作员不可操作
		return false
	}

	// 如果code为空或为"0"，则根据商品名称进行查询
	if code == "" || code == "0" {
		if name == "" {
			log.Printf("[WARN] 商品编码和商品名称都为空，无法更新商品数量")
			return false
		}

		log.Printf("[INFO] 商品编码为空，根据商品名称 '%s' 进行查询更新", name)
		return s.fxUpdateAlbaranProveedorWithCustomActionForReturnGoodsByName(documentoNo, name, func(list []model.AlbaranProveedorArticulo, e *model.AlbaranProveedorArticulo) {
			if append {
				e.Cantidad = -(math.Abs(e.Cantidad) + float64(amount))
			} else {
				e.Cantidad = -float64(amount)
			}

			if precio != nil && *precio >= 0 {
				e.Precio = *precio
				s.db.Model(e).Where("NombreEs = ?", name).Update("Precio", e.Precio)
			}

			log.Printf("更新退货单 %d 中的商品 '%s' 的数量为 %d, 进价为 %f", documentoNo, name, amount, e.Precio)
			s.db.Model(e).Where("NombreEs = ?", name).Update("Cantidad", e.Cantidad)
		})
	}

	return s.fxUpdateAlbaranProveedorWithCustomActionForReturnGoods(documentoNo, code, func(list []model.AlbaranProveedorArticulo, e *model.AlbaranProveedorArticulo) {
		if append {
			e.Cantidad = -(math.Abs(e.Cantidad) + float64(amount))
		} else {
			e.Cantidad = -float64(amount)
		}

		if precio != nil && *precio >= 0 {
			e.Precio = *precio
			s.db.Model(e).Where("ArticuloID = ?", code).Update("Precio", e.Precio)
		}

		log.Printf("更新退货单 %d 中的 商品 %s 的数量为 %d, 进价为 %f", documentoNo, code, amount, e.Precio)
		s.db.Model(e).Where("ArticuloID = ?", code).Update("Cantidad", e.Cantidad)
	})
}

// UpdateBloqueado 更新退货单的锁定状态
func (s *ReturnGoodsService) UpdateBloqueado(documentoNo int64, bloqueado string) bool {
	bloqueadoValue := "0"
	if bloqueado == "1" {
		bloqueadoValue = "1"
	}
	return s.db.Model(&model.AlbaranProveedor{}).Where("DocumentoNo = ?", documentoNo).Update("Bloqueado", bloqueadoValue).Error == nil
}

// fxUpdateAlbaranProveedorWithCustomActionForReturnGoods 更新退货单的通用方法
func (s *ReturnGoodsService) fxUpdateAlbaranProveedorWithCustomActionForReturnGoods(documentoNo int64, code string, action func([]model.AlbaranProveedorArticulo, *model.AlbaranProveedorArticulo)) bool {
	var albaranproveedor model.AlbaranProveedor
	if err := s.db.Where("DocumentoNo = ?", documentoNo).First(&albaranproveedor).Error; err != nil {
		log.Printf("[DEBUG] 无法查询到单据: [%d]", documentoNo)
		return false
	}

	albaranproveedorarticulos := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	found := false

	for i := range albaranproveedorarticulos {
		if albaranproveedorarticulos[i].ArticuloID == code || albaranproveedorarticulos[i].CodigoBarra == code {
			action(albaranproveedorarticulos, &albaranproveedorarticulos[i])
			found = true
			break
		}
	}

	if !found {
		log.Printf("[DEBUG] 无法查询到商品: [%s]", code)
		return false
	}

	// 重新获取更新后的商品列表
	updatedArticulos := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	result := s.calculateReturnedArticulos(updatedArticulos)
	albaranproveedor.Total = -math.Abs(result.Total)
	albaranproveedor.CantidadTotal = -math.Abs(result.CantidadTotal)
	albaranproveedor.LineaTotal = len(result.Data)
	albaranproveedor.TiempoUltimoActualizar = time.Now()
	albaranproveedor.AlmacenID = 1
	albaranproveedor.PrecioUsado = 10

	return s.db.Save(&albaranproveedor).Error == nil
}

// fxUpdateAlbaranProveedorWithCustomActionForReturnGoodsByName 根据商品名称更新退货单的通用方法
func (s *ReturnGoodsService) fxUpdateAlbaranProveedorWithCustomActionForReturnGoodsByName(documentoNo int64, productName string, action func([]model.AlbaranProveedorArticulo, *model.AlbaranProveedorArticulo)) bool {
	var albaranproveedor model.AlbaranProveedor
	if err := s.db.Where("DocumentoNo = ?", documentoNo).First(&albaranproveedor).Error; err != nil {
		log.Printf("[DEBUG] 无法查询到单据: [%d]", documentoNo)
		return false
	}

	albaranproveedorarticulos := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	found := false

	for i := range albaranproveedorarticulos {
		if albaranproveedorarticulos[i].NombreES == productName || albaranproveedorarticulos[i].NombreCN == productName {
			action(albaranproveedorarticulos, &albaranproveedorarticulos[i])
			found = true
			break
		}
	}

	if !found {
		log.Printf("[DEBUG] 无法查询到商品: [%s]", productName)
		return false
	}

	// 重新获取更新后的商品列表
	updatedArticulos := s.ListAllArticulosByDocumentoNo(strconv.FormatInt(documentoNo, 10))
	result := s.calculateReturnedArticulos(updatedArticulos)
	albaranproveedor.Total = -math.Abs(result.Total)
	albaranproveedor.CantidadTotal = -math.Abs(result.CantidadTotal)
	albaranproveedor.LineaTotal = len(result.Data)
	albaranproveedor.TiempoUltimoActualizar = time.Now()
	albaranproveedor.AlmacenID = 1
	albaranproveedor.PrecioUsado = 10

	return s.db.Save(&albaranproveedor).Error == nil
}

// calculateReturnedArticulos 计算退货单的总金额和总数量
func (s *ReturnGoodsService) calculateReturnedArticulos(data []model.AlbaranProveedorArticulo) ReturnedArticulosCalculationResult {
	var result ReturnedArticulosCalculationResult
	result.Data = data

	cantidadTotal := 0.0
	total := 0.0

	for _, item := range data {
		cantidadTotal += math.Abs(item.Cantidad)
		total += math.Abs(item.Cantidad) * item.Precio
	}

	result.CantidadTotal = cantidadTotal
	result.Total = total

	return result
}

// 辅助方法
func (awc *ArticuloWithCantidad) overWritingPrecio() bool {
	return awc.Precio > 0
}

func (awc *ArticuloWithCantidad) overWritingNombre() bool {
	return awc.Nombre != ""
}

func roundTo4Decimals(value float64) float64 {
	return math.Round(value*10000) / 10000
}

func generateUUID() string {
	hash := md5.Sum([]byte(fmt.Sprintf("%d", time.Now().UnixNano())))
	time.Sleep(time.Millisecond * 10)
	return fmt.Sprintf("%x", hash)[:10]
}
