package service

import (
	"gorm.io/gorm"
	"log"
	"supermarket-go/internal/model"
	"time"
)

type TicketDiarioTotalServiceImpl struct {
	db *gorm.DB
}

func NewTicketDiarioTotalService(db *gorm.DB) model.TicketDiarioTotalService {
	return &TicketDiarioTotalServiceImpl{db: db}
}

// RecordTicket 更新日收款总额表
// paid 支付金额 按顺序为 现金 刷卡 代金券
func (s *TicketDiarioTotalServiceImpl) RecordTicket(paid []float64) bool {
	log.Printf("[INFO] 开始更新日总额统计: paid=%v", paid)

	// 确保paid数组有3个元素
	if len(paid) < 3 {
		paid = append(paid, make([]float64, 3-len(paid))...)
	}

	// 获取今天的日期（YYYY-MM-DD格式）
	today := time.Now().Format("2006-01-02")
	todayDate, err := time.Parse("2006-01-02", today)
	if err != nil {
		log.Printf("[ERROR] 解析日期失败: %v", err)
		return false
	}
	todayDateStr := todayDate.Format("2006-01-02")
	// 查找今天的记录
	var existing model.TicketDiarioTotal
	err = s.db.Where("Fecha = ?", todayDateStr).First(&existing).Error
	if err != nil && err != gorm.ErrRecordNotFound {
		log.Printf("[ERROR] 查询日总额记录失败: %v", err)
		return false
	}

	log.Printf("[INFO] 日统计 %s: existing=%+v", today, existing)

	// 如果记录不存在，创建新记录
	if err == gorm.ErrRecordNotFound {
		existing = model.TicketDiarioTotal{
			Fecha:    todayDate,
			Total:    0.0,
			Efectivo: 0.0,
			Tarjeta:  0.0,
			Vale:     0.0,
		}
	}

	// 计算总付款金额
	totalPaid := paid[0] + paid[1] + paid[2]

	// 更新记录
	existing.Total += totalPaid
	existing.Efectivo += paid[0]
	existing.Tarjeta += paid[1]
	existing.Vale += paid[2]

	// 保存或更新记录
	var result error
	if err == gorm.ErrRecordNotFound {
		result = s.db.Create(&existing).Error
	} else {
		result = s.db.Save(&existing).Error
	}

	if result != nil {
		log.Printf("[ERROR] 保存日总额记录失败: %v", result)
		return false
	}

	log.Printf("[INFO] 成功更新日总额统计: Total=%.4f, Efectivo=%.4f, Tarjeta=%.4f, Vale=%.4f",
		existing.Total, existing.Efectivo, existing.Tarjeta, existing.Vale)
	return true
}
