package main

import (
	"fmt"
	"log"
	"supermarket-go/internal/controller"
	"supermarket-go/internal/service"
	"supermarket-go/internal/util"

	"github.com/gin-gonic/gin"
	"supermarket-go/internal/model"
)

// 日志中间件
func LoggerMiddleware() gin.HandlerFunc {
	return gin.LoggerWithFormatter(func(param gin.LogFormatterParams) string {
		// 记录HTTP请求日志
		util.LogRequest(param.Method, param.Path, param.ClientIP, param.StatusCode, param.Latency)

		// 返回空字符串，因为我们自己处理日志
		return ""
	})
}

func main() {
	// 初始化日志系统
	util.InitLogger()

	log.Println("[INFO] ========================================")
	log.Println("[INFO] 开始启动 Supermarket Go 服务...")
	log.Println("[INFO] ========================================")

	// 初始化数据库
	log.Println("[INFO] 正在初始化数据库连接...")
	model.InitDB()

	port := model.AppConfig.Server.Port
	if port == 0 {
		port = 8080
		log.Println("[WARN] 配置文件中未指定端口，使用默认端口 8080")
	}

	log.Printf("[INFO] 正在初始化 Gin 路由引擎...")
	r := gin.New()

	// 使用自定义日志中间件
	r.Use(LoggerMiddleware())
	r.Use(gin.Recovery())

	// 注册articulo路由
	log.Println("[INFO] 正在注册 articulo 相关路由...")
	controller.RegisterArticuloRoutes(r)
	
	// 初始化articulo导入相关服务
	log.Println("[INFO] 正在初始化 articulo import 相关服务...")
	articuloTableImportService := service.NewArticuloTableImportService(model.DB)
	controller.RegisterArticuloImportRoutes(r, articuloTableImportService)

	// 初始化service并注册duihuo路由
	log.Println("[INFO] 正在初始化 duihuo 相关服务...")
	albaranproveedorService := service.NewAlbaranproveedorService(model.DB)
	albaranproveedorarticuloService := service.NewAlbaranproveedorarticuloService(model.DB)
	controller.RegisterDuihuoRoutes(r, albaranproveedorService, albaranproveedorarticuloService)

	// 注册其他路由
	log.Println("[INFO] 正在注册 cliente 相关路由...")
	clienteService := service.NewClienteService(model.DB)
	controller.RegisterClientRoutes(r, clienteService)

	// 初始化stock相关服务
	log.Println("[INFO] 正在初始化 stock 相关服务...")
	articuloService := service.NewArticuloService(model.DB)
	stockService := service.NewStockService(model.DB)
	procedureService := service.NewProcedureService(model.DB)
	controller.RegisterStockRoutes(r, stockService, articuloService)

	log.Println("[INFO] 正在初始化 empleado 认证相关服务...")
	empleadoAuthService := service.NewEmpleadoAuthService(model.DB)
	empleadoService := service.NewEmpleadoService(model.DB)
	controller.RegisterEmpleadoAuthRoutes(r, empleadoAuthService, empleadoService)

	log.Println("[INFO] 正在初始化 return goods 相关服务...")
	returnGoodsService := service.NewReturnGoodsService(model.DB, articuloService, procedureService)
	controller.RegisterReturnGoodsRoutes(r, returnGoodsService)

	log.Println("[INFO] 正在初始化 order 相关服务...")
	controller.RegisterOrderRoutes(r)

	log.Println("[INFO] ========================================")
	log.Printf("[INFO] Supermarket Go 服务启动成功，监听端口: %d", port)
	log.Println("[INFO] 服务已就绪，等待请求...")
	log.Println("[INFO] ========================================")

	r.Run(fmt.Sprintf(":%d", port))
}
