package controller

import (
	"fmt"
	"log"
	"net/http"
	"net/url"
	"regexp"
	"strconv"
	"strings"
	"supermarket-go/internal/model"
	"supermarket-go/internal/util"

	"github.com/PuerkitoBio/goquery"
	"github.com/gin-gonic/gin"
)

// ArticuloImportController 商品导入控制器
type ArticuloImportController struct {
	articuloTableImportService model.ArticuloTableImportService
}

// NewArticuloImportController 创建商品导入控制器
func NewArticuloImportController(articuloTableImportService model.ArticuloTableImportService) *ArticuloImportController {
	return &ArticuloImportController{
		articuloTableImportService: articuloTableImportService,
	}
}

// RegisterArticuloImportRoutes 注册商品导入路由
func RegisterArticuloImportRoutes(r *gin.Engine, articuloTableImportService model.ArticuloTableImportService) {
	log.Println("[INFO] 注册 articulo import 路由组: /articulo/import")
	controller := NewArticuloImportController(articuloTableImportService)

	g := r.Group("/articulo/import")
	{
		g.POST("/byUrl", controller.UploadExcel)
		g.POST("/upload", controller.UploadData)
	}
	log.Println("[INFO] articulo import 路由注册完成")
}

// UploadExcel 通过URL上传Excel
func (c *ArticuloImportController) UploadExcel(ctx *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品导入URL请求 =====")

	url := ctx.PostForm("url")
	if url == "" {
		log.Printf("[ERROR] URL参数为空")
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("URL参数不能为空", nil))
		return
	}

	log.Printf("[INFO] 解析URL: %s", url)

	result, err := c.articuloTableImportService.Resolve(url)
	if err != nil {
		log.Printf("[ERROR] 解析URL失败: %v", err)
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("Could not import article table", nil))
		return
	}

	if result == nil {
		log.Printf("[WARN] 无法解析URL: %s", url)
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("Could not import article table", nil))
		return
	}

	// 获取表格数据
	tableData, err := c.GetTableData(url)
	if err != nil {
		log.Printf("[WARN] 无法解析GetTableDataL: %s,err:%v", url, err.Error())
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("Could not GetTableData table", nil))
		return
	}

	// 翻译商品名称
	//translatedData := c.translateArticleNames(tableData)

	log.Printf("[INFO] 成功解析商品数据，共 %d 条记录", len(tableData))

	ctx.JSON(http.StatusOK, util.BuildSuccessResponseData(tableData))
}

func (c *ArticuloImportController) GetTableData(url string) ([]model.TableRow, error) {
	// 发送HTTP GET请求
	resp, err := http.Get(url)
	if err != nil {
		return nil, fmt.Errorf("请求失败: %v", err)
	}
	defer resp.Body.Close()

	// 检查响应状态
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("请求失败，状态码: %d", resp.StatusCode)
	}

	// 解析HTML
	doc, err := goquery.NewDocumentFromReader(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("解析HTML失败: %v", err)
	}
	// 解析商品表格（处理<br>分隔）
	var products []model.TableRow
	doc.Find("table tr").Each(func(i int, s *goquery.Selection) {
		// 跳过表头行
		if i == 0 {
			return
		}

		tds := s.Find("td")
		if tds.Length() < 4 {
			return
		}

		// 提取商品图片地址
		imageURL := ""
		img := tds.Eq(0).Find("img")
		if img.Length() > 0 {
			src, exists := img.Attr("src")
			if exists {
				imageURL = resolveURL(url, src)
			}
		}

		// 提取附加信息的原始HTML，按<br>分割（核心处理）
		//attachmentHtml, _ := tds.Eq(0).Html()
		//attachmentParts := processHtmlBreaks(attachmentHtml)

		// 提取代码相关的原始HTML（通常在第二个td），按<br>分割
		codeHtml, _ := tds.Eq(1).Html()
		codeParts := processHtmlBreaks(codeHtml)

		// 从代码部分提取条形码和描述
		var barcode, description string
		if len(codeParts) >= 3 {
			// 假设格式：[代码, 条形码, 描述...]
			barcode = codeParts[1]
			description = strings.Join(codeParts[2:], " ")
		}

		// 提取单价
		priceStr := strings.TrimSpace(tds.Eq(2).Text())
		price, _ := strconv.ParseFloat(priceStr, 64)

		// 提取数量
		qttyStr := strings.TrimSpace(tds.Eq(3).Text())
		qtty, _ := strconv.Atoi(qttyStr)

		// 计算小计
		//totalPrice := price * float64(qtty)
		// 匹配中文
		reCN := regexp.MustCompile(`[\p{Han}]+`)
		// 匹配英文和空格
		reEN := regexp.MustCompile(`[A-Za-z ]+`)

		cn := strings.Join(reCN.FindAllString(description, -1), "")
		en := strings.TrimSpace(strings.Join(reEN.FindAllString(description, -1), ""))
		// 添加到商品列表
		products = append(products, model.TableRow{
			//CodeParts:          codeParts, // 按<br>分割的代码部分
			BarCode:  barcode,
			Name:     description,
			NombreCN: cn,
			NombreES: en,
			Price:    price,
			Quantity: qtty,
			//TotalPrice:         totalPrice,
			Icon:               imageURL,
			SalePrice:          price,
			ID:                 codeParts[0],
			ClassID:            -1,
			SuggestedSalePrice: -1,
		})
	})

	return products, nil
}

// UploadData 上传商品数据
func (c *ArticuloImportController) UploadData(ctx *gin.Context) {
	log.Println("[INFO] ===== 开始处理商品数据上传请求 =====")

	// 获取Authorization token
	token := ctx.GetHeader("Authorization")
	if token == "" {
		log.Printf("[ERROR] Authorization token为空")
		response := util.BuildResponse(403, "验证信息无效", "")
		ctx.JSON(http.StatusForbidden, response)
		return
	}

	// 从token中获取员工ID
	empleadoID, err := util.GetEmpleadoIdFromToken(token)
	if err != nil {
		log.Printf("[ERROR] 解析token失败: %v", err)
		response := util.BuildResponse(403, "验证信息无效", "")
		ctx.JSON(http.StatusForbidden, response)
		return
	}

	log.Printf("[INFO] 员工ID: %d", empleadoID)

	// 解析请求体
	var dto model.ArticuloImportUploadDTO
	if err := ctx.ShouldBindJSON(&dto); err != nil {
		log.Printf("[ERROR] 解析请求体失败: %v", err)
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("请求体格式错误", nil))
		return
	}

	log.Printf("[INFO] 上传参数: type=%d, providerId=%d, updateSalePrice=%v, updateOnly=%v, dataCount=%d",
		dto.Type, dto.ProviderID, dto.UpdateSalePrice, dto.UpdateOnly, len(dto.Data))

	// 处理重复数据，合并商品数量
	processedData := c.processDuplicateData(dto.Data, dto.Type)

	log.Printf("[INFO] 处理后的数据条数: %d", len(processedData))

	// 调用服务导入数据
	success, err := c.articuloTableImportService.Import(dto.ProviderID, empleadoID, dto.Type, processedData, dto.UpdateSalePrice, dto.UpdateOnly)
	if err != nil {
		log.Printf("[ERROR] 导入数据失败: %v", err)
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("Could not import article table", nil))
		return
	}

	if success {
		log.Printf("[INFO] 商品数据导入成功")
		ctx.JSON(http.StatusOK, util.BuildSuccessResponseData(""))
	} else {
		log.Printf("[ERROR] 商品数据导入失败")
		ctx.JSON(http.StatusBadRequest, util.BuildBadRequestResponse("Could not import article table", nil))
	}
}

// translateArticleNames 翻译商品名称
func (c *ArticuloImportController) translateArticleNames(data []model.TableRow) []model.TableRow {
	log.Printf("[INFO] 翻译商品名称，共 %d 条记录", len(data))

	// 提取商品名称
	var names []string
	for _, item := range data {
		names = append(names, item.Name)
	}

	// 分批翻译（每批60个）
	var translatedNames []string
	batchSize := 60
	for i := 0; i < len(names); i += batchSize {
		end := i + batchSize
		if end > len(names) {
			end = len(names)
		}
		batch := names[i:end]
		translatedBatch := util.TranslateToES(batch)
		translatedNames = append(translatedNames, translatedBatch...)
	}

	// 更新商品名称
	for i := range data {
		if i < len(translatedNames) {
			data[i].Name = translatedNames[i]
		}
	}

	log.Printf("[INFO] 翻译完成，共翻译 %d 个商品名称", len(translatedNames))
	return data
}

// convertTableRow2ToTableRow 将 TableRow2 转换为 TableRow
func convertTableRow2ToTableRow(item model.TableRow2) model.TableRow {
	// 转换 Quantity 从 string 到 int
	quantity := 0
	if item.Quantity != "" {
		if q, err := strconv.Atoi(item.Quantity); err == nil {
			quantity = q
		}
	}

	// 转换 ClassID 从 string 到 int
	classID := 0
	if item.ClassID != "" {
		if c, err := strconv.Atoi(item.ClassID); err == nil {
			classID = c
		}
	}
	price, _ := strconv.ParseFloat(item.Price, 64)
	return model.TableRow{
		Icon:               item.Icon,
		ID:                 item.ID,
		BarCode:            item.BarCode,
		Name:               item.Name,
		Price:              price,
		SalePrice:          item.SalePrice,
		Quantity:           quantity,
		ClassID:            classID,
		SuggestedSalePrice: item.SuggestedSalePrice,
	}
}

// processDuplicateData 处理重复数据，合并商品数量
func (c *ArticuloImportController) processDuplicateData(data []model.TableRow2, importType int) []model.TableRow {
	// 按匹配条件分组
	groupedData := make(map[string][]model.TableRow)

	for _, item := range data {
		// 将 TableRow2 转换为 TableRow
		tableRow := convertTableRow2ToTableRow(item)

		var key string
		switch importType {
		case 0: // 按ID匹配
			key = tableRow.ID
		case 1: // 按条码匹配
			key = tableRow.BarCode
		default: // ID和条码都匹配
			key = tableRow.ID + ":" + tableRow.BarCode
		}

		groupedData[key] = append(groupedData[key], tableRow)
	}

	// 处理分组后的数据
	var result []model.TableRow
	for _, items := range groupedData {
		if len(items) == 1 {
			// 不重复，直接添加
			result = append(result, items[0])
		} else {
			// 重复，合并数量
			mergedItem := items[0]
			totalQuantity := 0
			for _, item := range items {
				totalQuantity += item.Quantity
			}
			mergedItem.Quantity = totalQuantity
			result = append(result, mergedItem)
		}
	}

	return result
}

// 处理HTML中的<br>标签，转换为换行符并清理多余空格
func processHtmlBreaks(html string) []string {
	// 将<br>、<br/>等标签替换为换行符
	reBr := regexp.MustCompile(`<br\s*/?>`)
	text := reBr.ReplaceAllString(html, "\n")
	// 去除HTML标签
	reTag := regexp.MustCompile(`<[^>]+>`)
	text = reTag.ReplaceAllString(text, "")
	// 按换行分割
	lines := strings.Split(text, "\n")
	// 过滤空行并清理每行空格
	var result []string
	for _, line := range lines {
		cleanLine := strings.TrimSpace(line)
		if cleanLine != "" {
			result = append(result, cleanLine)
		}
	}
	return result
}

// 处理相对URL，转换为绝对URL
func resolveURL(baseURL, relativeURL string) string {
	if relativeURL == "" {
		return ""
	}
	base, err := url.Parse(baseURL)
	if err != nil {
		return relativeURL
	}
	ref, err := url.Parse(relativeURL)
	if err != nil {
		return relativeURL
	}
	return base.ResolveReference(ref).String()
}
