package model

import (
	"fmt"
	"time"
)

type AlbaranProveedor struct {
	DocumentoNo             int64      `gorm:"primaryKey;column:DocumentoNo"`
	DocumentoNoEnServidor   int64      `gorm:"column:DocumentoNoEnServidor"`
	DocumentoNoOriginal     int64      `gorm:"column:DocumentoNoOriginal"`
	EmpresaID               int64      `gorm:"column:EmpresaID"`
	SucursalID              uint16     `gorm:"column:SucursalID"`
	Descuento               float64    `gorm:"column:Descuento"`
	Descuento2              float64    `gorm:"column:Descuento2"`
	DescuentoDirecto        float64    `gorm:"column:DescuentoDirecto"`
	ReglaIVA                uint8      `gorm:"column:ReglaIVA"`
	IVASerie                uint8      `gorm:"column:IVASerie"`
	PrecioUsado             uint8      `gorm:"column:PrecioUsado"`
	Total                   float64    `gorm:"column:Total"`
	DescuentoTotal          float64    `gorm:"column:DescuentoTotal"`
	DescuentoExtra          float64    `gorm:"column:DescuentoExtra"`
	Ganancias               float64    `gorm:"column:Ganancias"`
	PagoTotal               float64    `gorm:"column:PagoTotal"`
	CantidadTotal           float64    `gorm:"column:CantidadTotal"`
	PaqueteTotal            float64    `gorm:"column:PaqueteTotal"`
	CajaTotal               float64    `gorm:"column:CajaTotal"`
	VolumenPesoTotal        float64    `gorm:"column:VolumenPesoTotal"`
	LineaTotal              int        `gorm:"column:LineaTotal"`
	Fecha                   time.Time  `gorm:"column:Fecha"` // 可为 NULL
	FechaEntrada            time.Time  `gorm:"column:FechaEntrada"`
	FechaOriginal           *time.Time `gorm:"column:FechaOriginal"` // 可为 NULL
	TiempoUltimoEditar      *time.Time `gorm:"column:TiempoUltimoEditar"`
	TiempoUltimoActualizar  time.Time  `gorm:"column:TiempoUltimoActualizar"`
	DocumentoOrigenTipo     string     `gorm:"column:DocumentoOrigenTipo;size:40"`
	DocumentoOrigenNo       uint64     `gorm:"column:DocumentoOrigenNo"`
	DocumentoDestinoTipo    string     `gorm:"column:DocumentoDestinoTipo;size:40"`
	DocumentoDestinoNo      uint64     `gorm:"column:DocumentoDestinoNo"`
	AgenteID                uint16     `gorm:"column:AgenteID"`
	TransportistaID         uint16     `gorm:"column:TransportistaID"`
	TransporteModoID        uint16     `gorm:"column:TransporteModoID"`
	EnvioGasto              float64    `gorm:"column:EnvioGasto"`
	OperadorID              int64      `gorm:"column:OperadorID"`
	VerificadorID           uint16     `gorm:"column:VerificadorID"`
	AlmacenID               uint16     `gorm:"column:AlmacenID"`
	DocumentoAdjuntoSimbolo string     `gorm:"column:DocumentoAdjuntoSimbolo;size:60"`
	OrdenadorNombre         string     `gorm:"column:OrdenadorNombre;size:16"`
	DeviceUUID              string     `gorm:"column:DeviceUUID;size:64"`
	DocumentoHash           string     `gorm:"column:DocumentoHash;size:255"`
	SubTiendaID             string     `gorm:"column:SubTiendaID;size:16"`
	DocumentoIntercambioNo  int64      `gorm:"column:DocumentoIntercambioNo"`
	Bloqueado               bool       `gorm:"column:Bloqueado"`
	Anulado                 bool       `gorm:"column:Anulado"`
	Modificando             bool       `gorm:"column:Modificando"`
	EmailMandado            bool       `gorm:"column:EmailMandado"`
	ProgressID              uint8      `gorm:"column:ProgressID"`
	ProgressStatus          uint8      `gorm:"column:ProgressStatus"`
	LogID                   uint64     `gorm:"column:LogID"`
	HayAdjunto              bool       `gorm:"column:HayAdjunto"`
	Imprimido               bool       `gorm:"column:Imprimido"`
	Observacion             *string    `gorm:"column:Observacion;size:2048"`
}

type AlbaranProveedorDto struct {
	AlbaranProveedor
	ProNombreES string `gorm:"column:ProNombreES"`
}

type Documento struct {
	DocumentoNo             int64   `json:"documentoNo"`
	DocumentoNoEnServidor   int     `json:"documentoNoEnServidor"`
	DocumentoNoOriginal     int     `json:"documentoNoOriginal"`
	EmpresaID               int     `json:"empresaID"`
	SucursalID              int     `json:"sucursalID"`
	Descuento               float64 `json:"descuento"`
	Descuento2              float64 `json:"descuento2"`
	DescuentoDirecto        float64 `json:"descuentoDirecto"`
	ReglaIVA                int     `json:"reglaIVA"`
	PrecioUsado             int     `json:"precioUsado"`
	Total                   float64 `json:"total"`
	DescuentoTotal          float64 `json:"descuentoTotal"`
	DescuentoExtra          float64 `json:"descuentoExtra"`
	Ganancias               float64 `json:"ganancias"`
	PagoTotal               float64 `json:"pagoTotal"`
	CantidadTotal           float64 `json:"cantidadTotal"`
	PaqueteTotal            float64 `json:"paqueteTotal"`
	CajaTotal               float64 `json:"cajaTotal"`
	VolumenPesoTotal        float64 `json:"volumenPesoTotal"`
	LineaTotal              int     `json:"lineaTotal"`
	Fecha                   int64   `json:"fecha"`
	FechaEntrada            int64   `json:"fechaEntrada"`
	FechaOriginal           *int64  `json:"fechaOriginal"`
	TiempoUltimoEditar      *int64  `json:"tiempoUltimoEditar"`
	TiempoUltimoActualizar  int64   `json:"tiempoUltimoActualizar"`
	DocumentoOrigenTipo     string  `json:"documentoOrigenTipo"`
	DocumentoOrigenNo       int     `json:"documentoOrigenNo"`
	DocumentoDestinoTipo    string  `json:"documentoDestinoTipo"`
	DocumentoDestinoNo      int     `json:"documentoDestinoNo"`
	AgenteID                int     `json:"agenteID"`
	TransportistaID         int     `json:"transportistaID"`
	TransporteModoID        int     `json:"transporteModoID"`
	EnvioGasto              float64 `json:"envioGasto"`
	OperadorID              int     `json:"operadorID"`
	VerificadorID           int     `json:"verificadorID"`
	AlmacenID               int     `json:"almacenID"`
	DocumentoAdjuntoSimbolo string  `json:"documentoAdjuntoSimbolo"`
	OrdenadorNombre         string  `json:"ordenadorNombre"`
	DeviceUUID              string  `json:"deviceUUID"`
	DocumentoHash           string  `json:"documentoHash"`
	SubTiendaID             string  `json:"subTiendaID"`
	DocumentoIntercambioNo  int     `json:"documentoIntercambioNo"`
	Bloqueado               int     `json:"bloqueado"`
	Anulado                 int     `json:"anulado"`
	Modificando             int     `json:"modificando"`
	EmailMandado            int     `json:"emailMandado"`
	ProgressID              int     `json:"progressID"`
	ProgressStatus          int     `json:"progressStatus"`
	LogID                   int     `json:"logID"`
	HayAdjunto              int     `json:"hayAdjunto"`
	Imprimido               int     `json:"imprimido"`
	Observacion             *string `json:"observacion"`
	IvaSerie                int     `json:"ivaserie"`
	ProNombreES             string  `json:"proNombreES"`
	ProveedorID             string  `json:"proveedorID"`
}

// TableName overrides the table name used by GORM
func (AlbaranProveedor) TableName() string {
	return "albaranproveedor"
}

// ConvertToDocumento 将 AlbaranProveedorDto 转换为 Documento
func (dto AlbaranProveedorDto) ConvertToDocumento() Documento {
	return Documento{
		DocumentoNo:             dto.DocumentoNo,
		DocumentoNoEnServidor:   int(dto.DocumentoNoEnServidor),
		DocumentoNoOriginal:     int(dto.DocumentoNoOriginal),
		EmpresaID:               int(dto.EmpresaID),
		SucursalID:              int(dto.SucursalID),
		Descuento:               dto.Descuento,
		Descuento2:              dto.Descuento2,
		DescuentoDirecto:        dto.DescuentoDirecto,
		ReglaIVA:                int(dto.ReglaIVA),
		PrecioUsado:             int(dto.PrecioUsado),
		Total:                   dto.Total,
		DescuentoTotal:          dto.DescuentoTotal,
		DescuentoExtra:          dto.DescuentoExtra,
		Ganancias:               dto.Ganancias,
		PagoTotal:               dto.PagoTotal,
		CantidadTotal:           dto.CantidadTotal,
		PaqueteTotal:            dto.PaqueteTotal,
		CajaTotal:               dto.CajaTotal,
		VolumenPesoTotal:        dto.VolumenPesoTotal,
		LineaTotal:              dto.LineaTotal,
		Fecha:                   dto.Fecha.UnixMilli(),
		FechaEntrada:            dto.FechaEntrada.UnixMilli(),
		FechaOriginal:           convertTimeToUnixMilli(dto.FechaOriginal),
		TiempoUltimoEditar:      convertTimeToUnixMilli(dto.TiempoUltimoEditar),
		TiempoUltimoActualizar:  dto.TiempoUltimoActualizar.UnixMilli(),
		DocumentoOrigenTipo:     dto.DocumentoOrigenTipo,
		DocumentoOrigenNo:       int(dto.DocumentoOrigenNo),
		DocumentoDestinoTipo:    dto.DocumentoDestinoTipo,
		DocumentoDestinoNo:      int(dto.DocumentoDestinoNo),
		AgenteID:                int(dto.AgenteID),
		TransportistaID:         int(dto.TransportistaID),
		TransporteModoID:        int(dto.TransporteModoID),
		EnvioGasto:              dto.EnvioGasto,
		OperadorID:              int(dto.OperadorID),
		VerificadorID:           int(dto.VerificadorID),
		AlmacenID:               int(dto.AlmacenID),
		DocumentoAdjuntoSimbolo: dto.DocumentoAdjuntoSimbolo,
		OrdenadorNombre:         dto.OrdenadorNombre,
		DeviceUUID:              dto.DeviceUUID,
		DocumentoHash:           dto.DocumentoHash,
		SubTiendaID:             dto.SubTiendaID,
		DocumentoIntercambioNo:  int(dto.DocumentoIntercambioNo),
		Bloqueado:               boolToInt(dto.Bloqueado),
		Anulado:                 boolToInt(dto.Anulado),
		Modificando:             boolToInt(dto.Modificando),
		EmailMandado:            boolToInt(dto.EmailMandado),
		ProgressID:              int(dto.ProgressID),
		ProgressStatus:          int(dto.ProgressStatus),
		LogID:                   int(dto.LogID),
		HayAdjunto:              boolToInt(dto.HayAdjunto),
		Imprimido:               boolToInt(dto.Imprimido),
		Observacion:             dto.Observacion,
		IvaSerie:                int(dto.IVASerie),
		ProNombreES:             dto.ProNombreES,
		ProveedorID:             fmt.Sprintf("%d", dto.EmpresaID), // ProveedorID 对应 EmpresaID
	}
}

// convertTimeToUnixMilli 将 *time.Time 转换为 *int64 (Unix 毫秒时间戳)
func convertTimeToUnixMilli(t *time.Time) *int64 {
	if t == nil {
		return nil
	}
	unix := t.UnixMilli()
	return &unix
}

// boolToInt 将 bool 转换为 int
func boolToInt(b bool) int {
	if b {
		return 1
	}
	return 0
}
