package model

import (
	"fmt"
	"log"
	"os"
	"time"

	"gorm.io/driver/mysql"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
	"gopkg.in/yaml.v2"
)

type DBConfig struct {
	Host     string `yaml:"host"`
	Port     int    `yaml:"port"`
	User     string `yaml:"user"`
	Password string `yaml:"password"`
	DBName   string `yaml:"dbname"`
	Charset  string `yaml:"charset"`
	ParseTime bool  `yaml:"parseTime"`
	Loc      string `yaml:"loc"`
}

type Config struct {
	Server struct {
		Port int `yaml:"port"`
	} `yaml:"server"`
	Database DBConfig `yaml:"database"`
}

var DB *gorm.DB
var AppConfig Config

func InitDB() {
	log.Println("[INFO] 开始初始化数据库连接...")
	
	f, err := os.Open("config.yaml")
	if err != nil {
		log.Fatalf("[ERROR] 无法打开配置文件: %v", err)
	}
	defer f.Close()
	
	decoder := yaml.NewDecoder(f)
	if err := decoder.Decode(&AppConfig); err != nil {
		log.Fatalf("[ERROR] 解析配置文件失败: %v", err)
	}
	
	cfg := AppConfig.Database
	log.Printf("[INFO] 数据库配置: Host=%s, Port=%d, DBName=%s, User=%s", 
		cfg.Host, cfg.Port, cfg.DBName, cfg.User)
	
	dsn := fmt.Sprintf("%s:%s@tcp(%s:%d)/%s?charset=%s&parseTime=%t&loc=%s",
		cfg.User, cfg.Password, cfg.Host, cfg.Port, cfg.DBName, cfg.Charset, cfg.ParseTime, cfg.Loc)
	
	// 配置GORM日志
	gormLogger := logger.New(
		log.New(os.Stdout, "\r\n", log.LstdFlags), // io writer
		logger.Config{
			SlowThreshold:             time.Second,     // 慢SQL阈值
			LogLevel:                  logger.Info,     // 日志级别
			IgnoreRecordNotFoundError: true,            // 忽略记录未找到错误
			Colorful:                  true,            // 彩色打印
		},
	)
	
	DB, err = gorm.Open(mysql.Open(dsn), &gorm.Config{
		Logger: gormLogger,
	})
	if err != nil {
		log.Fatalf("[ERROR] 数据库连接失败: %v", err)
	}
	
	// 获取底层sql.DB对象并设置连接池参数
	sqlDB, err := DB.DB()
	if err != nil {
		log.Fatalf("[ERROR] 获取数据库实例失败: %v", err)
	}
	
	// 设置连接池参数
	sqlDB.SetMaxIdleConns(10)
	sqlDB.SetMaxOpenConns(100)
	sqlDB.SetConnMaxLifetime(time.Hour)
	
	log.Println("[INFO] 数据库连接成功，连接池已配置")
	log.Printf("[INFO] 连接池配置: MaxIdleConns=%d, MaxOpenConns=%d, ConnMaxLifetime=%v", 
		10, 100, time.Hour)
} 