package model

import (
	"time"
)

// OrderDTO 订单数据传输对象
type OrderDTO struct {
	AdditionalDiscount float64   `json:"additionalDiscount"` // 整单额外折扣
	CheckOut           bool      `json:"checkOut"`           // 是否结账
	EmpresaID          int64     `json:"empresaId"`          // 公司ID
	Items              []Item    `json:"items"`              // 商品项列表
	Paid               []float64 `json:"paid"`               // 各种支付方式金额
	TicketNo           int64     `json:"ticketNo"`           // 小票编号
}

// Item 订单商品项
type Item struct {
	Amount    int     `json:"amount"`    // 数量
	ArticleID string  `json:"articleId"` // 商品ID
	Discount  float64 `json:"discount"`  // 商品折扣
	NameCN    string  `json:"nameCN"`    // 中文名称
	NameES    string  `json:"nameES"`    // 西语名称
	Price     float64 `json:"price"`     // 单价
}

// TicketColgando 挂单信息
type TicketColgando struct {
	DocumentoNo             int64      `gorm:"column:DocumentoNo;primaryKey;autoIncrement"`
	DocumentoNoEnServidor   int64      `gorm:"column:DocumentoNoEnServidor;not null;default:0"`
	DocumentoNoOriginal     int64      `gorm:"column:DocumentoNoOriginal;not null;default:0"`
	Fecha                   time.Time  `gorm:"column:Fecha"`
	FechaEntrada            time.Time  `gorm:"column:FechaEntrada;not null;default:CURRENT_TIMESTAMP"`
	FechaOriginal           *time.Time `gorm:"column:FechaOriginal"`
	TiempoUltimoEditar      *time.Time `gorm:"column:TiempoUltimoEditar"`
	TiempoUltimoActualizar  *time.Time `gorm:"column:TiempoUltimoActualizar"`
	CobrarTipo              int        `gorm:"column:CobrarTipo;not null;default:0"`
	Descuento               float64    `gorm:"column:Descuento;type:decimal(5,2);not null;default:0.00"`
	Descuento2              float64    `gorm:"column:Descuento2;type:decimal(5,2);not null;default:0.00"`
	DescuentoDirecto        float64    `gorm:"column:DescuentoDirecto;type:decimal(10,4);not null;default:0.0000"`
	Total                   float64    `gorm:"column:Total;type:decimal(14,4);not null;default:0.0000"`
	DescuentoTotal          float64    `gorm:"column:DescuentoTotal;type:decimal(10,4);not null;default:0.0000"`
	DescuentoExtra          float64    `gorm:"column:DescuentoExtra;type:decimal(5,2);not null;default:0.00"`
	Ganancias               float64    `gorm:"column:Ganancias;type:decimal(12,4);not null;default:0.0000"`
	CantidadTotal           float64    `gorm:"column:CantidadTotal;type:decimal(12,4);not null;default:0.0000"`
	PaqueteTotal            float64    `gorm:"column:PaqueteTotal;type:decimal(12,4);not null;default:0.0000"`
	CajaTotal               float64    `gorm:"column:CajaTotal;type:decimal(12,4);not null;default:0.0000"`
	VolumenPesoTotal        float64    `gorm:"column:VolumenPesoTotal;type:decimal(12,4);not null;default:0.0000"`
	LineaTotal              int        `gorm:"column:LineaTotal;not null;default:0"`
	ReglaIVA                int        `gorm:"column:ReglaIVA;not null;default:0"`
	IVASerie                int        `gorm:"column:IVASerie;not null;default:0"`
	PrecioUsado             int        `gorm:"column:PrecioUsado;not null;default:1"`
	EmpresaID               int64      `gorm:"column:EmpresaID;not null;default:0"`
	SucursalID              uint16     `gorm:"column:SucursalID;not null;default:0"`
	ClienteNombre           string     `gorm:"column:ClienteNombre;size:255;not null;default:''"`
	CobroIntentado          int        `gorm:"column:CobroIntentado;not null;default:0"`
	Anulado                 int        `gorm:"column:Anulado;not null;default:0"`
	OperadorID              uint16     `gorm:"column:OperadorID;not null;default:0"`
	AgenteID                uint16     `gorm:"column:AgenteID;not null;default:0"`
	TransportistaID         uint16     `gorm:"column:TransportistaID;not null;default:0"`
	TransporteModoID        uint16     `gorm:"column:TransporteModoID;not null;default:0"`
	EnvioGasto              float64    `gorm:"column:EnvioGasto;type:decimal(14,4);not null;default:0.0000"`
	AlmacenID               uint16     `gorm:"column:AlmacenID;not null;default:1"`
	DocumentoAdjuntoSimbolo string     `gorm:"column:DocumentoAdjuntoSimbolo;size:60;not null;default:''"`
	CocheNo                 string     `gorm:"column:CocheNo;size:40;not null;default:''"`
	OrdenadorNombre         string     `gorm:"column:OrdenadorNombre;size:16;not null;default:''"`
	DeviceUUID              string     `gorm:"column:DeviceUUID;size:64;not null;default:''"`
	DocumentoHash           string     `gorm:"column:DocumentoHash;size:255;not null;default:''"`
	SubTiendaID             string     `gorm:"column:SubTiendaID;size:16;not null;default:''"`
	DocumentoIntercambioNo  int64      `gorm:"column:DocumentoIntercambioNo;not null;default:0"`
	ColgarNota              string     `gorm:"column:ColgarNota;size:60;not null;default:''"`
	Observacion             *string    `gorm:"column:Observacion;size:2048"`
}

// TableName 指定表名
func (TicketColgando) TableName() string {
	return "ticketcolgando"
}

// TicketColgandoArticulo 挂单商品详情
type TicketColgandoArticulo struct {
	DocumentoNo              uint64  `gorm:"column:DocumentoNo;primaryKey"`
	ArticuloID               string  `gorm:"column:ArticuloID;size:20;not null;default:''"`
	PropCount                uint16  `gorm:"column:PropCount;not null;default:0"`
	PropID1                  string  `gorm:"column:PropID1;size:4;not null;default:''"`
	PropID2                  string  `gorm:"column:PropID2;size:4;not null;default:''"`
	Propiedad                string  `gorm:"column:Propiedad;size:160;not null;default:''"`
	SerialNo                 string  `gorm:"column:SerialNo;size:4096;not null;default:''"`
	CodigoBarra              string  `gorm:"column:CodigoBarra;size:20;not null;default:''"`
	CodigoBarraBalanza       int     `gorm:"column:CodigoBarraBalanza;not null;default:0"`
	NombreES                 string  `gorm:"column:NombreES;size:255;not null;default:''"`
	NombreCN                 string  `gorm:"column:NombreCN;size:80;not null;default:''"`
	Precio                   float64 `gorm:"column:Precio;type:decimal(12,4);not null;default:0.0000"`
	Cantidad                 float64 `gorm:"column:Cantidad;type:decimal(12,4);not null;default:0.0000"`
	IVA                      float64 `gorm:"column:IVA;type:decimal(6,3);not null;default:0.000"`
	REQ                      float64 `gorm:"column:REQ;type:decimal(6,3);not null;default:0.000"`
	Descuento                float64 `gorm:"column:Descuento;type:decimal(5,2);not null;default:0.00"`
	DescuentoCambioProhibido int     `gorm:"column:DescuentoCambioProhibido;not null;default:0"`
	PrecioCoste              float64 `gorm:"column:PrecioCoste;type:decimal(12,4);not null;default:0.0000"`
	Comentario               *string `gorm:"column:Comentario;size:1024"`
	NecesitaSerialNo         int     `gorm:"column:NecesitaSerialNo;not null;default:0"`
	Temporal                 int     `gorm:"column:Temporal;not null;default:0"`
	OrdenNo                  int     `gorm:"column:OrdenNo;not null;default:0"`
}

// TableName 指定表名
func (TicketColgandoArticulo) TableName() string {
	return "ticketcolgandoarticulo"
}

// NestedTicketColgando 嵌套挂单信息
type NestedTicketColgando struct {
	TicketColgando TicketColgando           `json:"ticketColgando"` // 挂单信息
	Details        []TicketColgandoArticulo `json:"details"`        // 挂单商品集合
}

// Ticket 订单信息
type Ticket struct {
	DocumentoNo             int64      `gorm:"column:DocumentoNo;primaryKey"`
	DocumentoNoEnServidor   int64      `gorm:"column:DocumentoNoEnServidor"`
	DocumentoNoOriginal     int64      `gorm:"column:DocumentoNoOriginal"`
	Fecha                   time.Time  `gorm:"column:Fecha"` // 可为 NULL
	FechaEntrada            time.Time  `gorm:"column:FechaEntrada;autoCreateTime"`
	FechaOriginal           *time.Time `gorm:"column:FechaOriginal"`
	TiempoUltimoEditar      *time.Time `gorm:"column:TiempoUltimoEditar"`
	TiempoUltimoActualizar  time.Time  `gorm:"column:TiempoUltimoActualizar"`
	DocumentoOrigenTipo     string     `gorm:"column:DocumentoOrigenTipo"`
	DocumentoOrigenNo       uint64     `gorm:"column:DocumentoOrigenNo"`
	DocumentoDestinoTipo    string     `gorm:"column:DocumentoDestinoTipo"`
	DocumentoDestinoNo      uint64     `gorm:"column:DocumentoDestinoNo"`
	CobrarTipo              uint8      `gorm:"column:CobrarTipo"`
	FormaDePago             string     `gorm:"column:FormaDePago"`
	Efectivo                float64    `gorm:"column:Efectivo"`
	Descuento               float64    `gorm:"column:Descuento"`
	Descuento2              float64    `gorm:"column:Descuento2"`
	DescuentoDirecto        float64    `gorm:"column:DescuentoDirecto"`
	Total                   float64    `gorm:"column:Total"`
	DescuentoTotal          float64    `gorm:"column:DescuentoTotal"`
	DescuentoExtra          float64    `gorm:"column:DescuentoExtra"`
	DescuentoUsarRegla      uint16     `gorm:"column:DescuentoUsarRegla"`
	Ganancias               float64    `gorm:"column:Ganancias"`
	CobroTotal              float64    `gorm:"column:CobroTotal"`
	CantidadTotal           float64    `gorm:"column:CantidadTotal"`
	PaqueteTotal            float64    `gorm:"column:PaqueteTotal"`
	CajaTotal               float64    `gorm:"column:CajaTotal"`
	VolumenPesoTotal        float64    `gorm:"column:VolumenPesoTotal"`
	LineaTotal              int        `gorm:"column:LineaTotal"`
	ReglaIVA                uint8      `gorm:"column:ReglaIVA"`
	IVASerie                uint8      `gorm:"column:IVASerie"`
	PrecioUsado             uint8      `gorm:"column:PrecioUsado"`
	CajonID                 uint16     `gorm:"column:CajonID"`
	EmpresaID               int64      `gorm:"column:EmpresaID"`
	SucursalID              uint16     `gorm:"column:SucursalID"`
	ClienteIVAClase         uint8      `gorm:"column:ClienteIVAClase"`
	ClienteReglaIVA         uint8      `gorm:"column:ClienteReglaIVA"`
	UsarClienteReglaIVA     bool       `gorm:"column:UsarClienteReglaIVA"`
	AgenteID                uint16     `gorm:"column:AgenteID"`
	TransportistaID         uint16     `gorm:"column:TransportistaID"`
	TransporteModoID        uint16     `gorm:"column:TransporteModoID"`
	EnvioGasto              float64    `gorm:"column:EnvioGasto"`
	CocheNo                 string     `gorm:"column:CocheNo"`
	OperadorID              uint16     `gorm:"column:OperadorID"`
	VerificadorID           uint16     `gorm:"column:VerificadorID"`
	AlmacenID               uint16     `gorm:"column:AlmacenID"`
	DocumentoAdjuntoSimbolo string     `gorm:"column:DocumentoAdjuntoSimbolo"`
	PreFormaDePagoNo        uint       `gorm:"column:PreFormaDePagoNo"`
	OrdenadorNombre         string     `gorm:"column:OrdenadorNombre"`
	DeviceUUID              string     `gorm:"column:DeviceUUID"`
	DocumentoHash           string     `gorm:"column:DocumentoHash"`
	SubTiendaID             string     `gorm:"column:SubTiendaID"`
	DocumentoIntercambioNo  int64      `gorm:"column:DocumentoIntercambioNo"`
	AbonoTotal              float64    `gorm:"column:AbonoTotal"`
	Cobrado                 int8       `gorm:"column:Cobrado"` // 默认值为 -1，不适合 bool
	Bloqueado               bool       `gorm:"column:Bloqueado"`
	Anulado                 bool       `gorm:"column:Anulado"`
	Modificando             bool       `gorm:"column:Modificando"`
	EmailMandado            bool       `gorm:"column:EmailMandado"`
	ProgressID              uint8      `gorm:"column:ProgressID"`
	ProgressStatus          uint8      `gorm:"column:ProgressStatus"`
	LogID                   uint64     `gorm:"column:LogID"`
	HayAdjunto              bool       `gorm:"column:HayAdjunto"`
	Imprimido               bool       `gorm:"column:Imprimido"`
	Observacion             *string    `gorm:"column:Observacion"`
}

// TableName 指定表名
func (Ticket) TableName() string {
	return "ticket"
}

// TicketArticulo 订单商品详情
type TicketArticulo struct {
	DocumentoNo              uint64  `gorm:"column:DocumentoNo;primaryKey;default:0"`
	ArticuloID               string  `gorm:"column:ArticuloID;size:20;not null;default:''"`
	PropCount                uint16  `gorm:"column:PropCount;not null;default:0"`
	PropID1                  string  `gorm:"column:PropID1;size:4;not null;default:''"`
	PropID2                  string  `gorm:"column:PropID2;size:4;not null;default:''"`
	Propiedad                string  `gorm:"column:Propiedad;size:160;not null;default:''"`
	SerialNo                 string  `gorm:"column:SerialNo;size:4096;not null;default:''"`
	CodigoBarra              string  `gorm:"column:CodigoBarra;size:20;not null;default:''"`
	CodigoBarraBalanza       bool    `gorm:"column:CodigoBarraBalanza;not null;default:false"`
	NombreES                 string  `gorm:"column:NombreES;size:255;not null;default:''"`
	NombreCN                 string  `gorm:"column:NombreCN;size:80;not null;default:''"`
	Precio                   float64 `gorm:"column:Precio;type:decimal(12,4);not null;default:0.0000"`
	Cantidad                 float64 `gorm:"column:Cantidad;type:decimal(12,4);not null;default:0.0000"`
	IVA                      float64 `gorm:"column:IVA;type:decimal(6,3);not null;default:0.000"`
	REQ                      float64 `gorm:"column:REQ;type:decimal(6,3);not null;default:0.000"`
	Descuento                float64 `gorm:"column:Descuento;type:decimal(5,2);not null;default:0.00"`
	DescuentoCambioProhibido bool    `gorm:"column:DescuentoCambioProhibido;not null;default:false"`
	PrecioCoste              float64 `gorm:"column:PrecioCoste;type:decimal(12,4);not null;default:0.0000"`
	Comentario               *string `gorm:"column:Comentario;size:1024"` // 可为 NULL
	NecesitaSerialNo         bool    `gorm:"column:NecesitaSerialNo;not null;default:false"`
	Temporal                 bool    `gorm:"column:Temporal;not null;default:false"`
	OrdenNo                  int     `gorm:"column:OrdenNo;not null;default:0"`
}

// TableName 指定表名
func (TicketArticulo) TableName() string {
	return "ticketarticulo"
}

// NestedTicket 嵌套订单信息
type NestedTicket struct {
	Ticket  Ticket           `json:"ticket"`  // 订单信息
	Details []TicketArticulo `json:"details"` // 订单商品集合
}

// OrderService 订单服务接口
type OrderService interface {
	// GetTicketById 通过订单号获取订单
	GetTicketById(ticketId int64) *NestedTicket

	// GetAllSuspendedTickets 获取所有挂单
	GetAllSuspendedTickets(detailed bool) []NestedTicketColgando

	// GetSuspendedTicket 通过挂单号获取挂单
	GetSuspendedTicket(documentoNo int64, detailed bool) *NestedTicketColgando

	// SubmitSuspendedTicket 提交挂单
	SubmitSuspendedTicket(empleadoId int, data OrderDTO) *int64

	// DeleteSuspendedTicket 删除挂单
	DeleteSuspendedTicket(documentoNo int64) bool

	// CheckOutSuspendedTicket 按挂单号结算
	CheckOutSuspendedTicket(empleadoId int, documentoNo int64, paid []float64) *int64

	// CheckOutSuspendedTicketByData 结算挂单数据
	CheckOutSuspendedTicketByData(empleadoId int, colgando NestedTicketColgando, paid []float64) *int64
}

// 计算原始总价
func (o *OrderDTO) CalculateOriginalTotal() float64 {
	total := 0.0
	for _, item := range o.Items {
		total += item.Price * float64(item.Amount)
	}
	return total
}

// 计算折扣总价
func (o *OrderDTO) CalculateDiscountedTotal(additionDiscount float64) float64 {
	total := 0.0
	for _, item := range o.Items {
		total += item.Price * float64(item.Amount) * (item.Discount + additionDiscount)
	}
	return total
}

// 将Item转换为TicketColgandoArticulo
func (item *Item) MapToColgandoArticulo(orderIndex int) TicketColgandoArticulo {
	return TicketColgandoArticulo{
		ArticuloID: item.ArticleID,
		NombreES:   item.NameES,
		NombreCN:   item.NameCN,
		Precio:     item.Price,
		Cantidad:   float64(item.Amount),
		OrdenNo:    orderIndex,
		Descuento:  item.Discount * 100, // 转换为百分比
	}
}

// TicketCobro 付款记录
type TicketCobro struct {
	DocumentoNo uint64    `gorm:"column:DocumentoNo;primaryKey"`
	Fecha       time.Time `gorm:"column:Fecha"`
	Importe     float64   `gorm:"column:Importe;type:decimal(14,4);not null;default:0.0000"`
	FormaDePago string    `gorm:"column:FormaDePago;size:20;not null;default:''"`
	OperadorID  uint16    `gorm:"column:OperadorID;not null;default:0"`
}

// TableName 指定表名
func (TicketCobro) TableName() string {
	return "ticketcobro"
}
