package service

import (
	"crypto/md5"
	"fmt"
	"io"
	"log"
	"supermarket-go/internal/model"

	"gorm.io/gorm"
)

// ArticulodibujoServiceImpl 商品图片服务实现
type ArticulodibujoServiceImpl struct {
	db              *gorm.DB
	dibujoService   *DibujoService
	articuloService model.ArticuloService
}

// NewArticulodibujoService 创建商品图片服务
func NewArticulodibujoService(db *gorm.DB) ArticulodibujoService {
	return &ArticulodibujoServiceImpl{
		db:              db,
		dibujoService:   NewDibujoService(db),
		articuloService: NewArticuloService(db),
	}
}

// SaveOrUpdateArticuloDibujo 保存或更新商品图片
func (s *ArticulodibujoServiceImpl) SaveOrUpdateArticuloDibujo(articuloID string, inputStream io.Reader) (bool, error) {
	// 读取图片数据
	bytes, err := io.ReadAll(inputStream)
	if err != nil {
		log.Printf("[ERROR] 读取图片数据失败: %v", err)
		return false, err
	}

	// 计算MD5
	md5Hash := md5.Sum(bytes)
	md5Str := fmt.Sprintf("%x", md5Hash)

	// 检查是否存在重复图片
	duplicatedImage, err := s.GetByMD5(md5Str)
	if err != nil {
		log.Printf("[ERROR] 查询重复图片失败: %v", err)
		return false, err
	}

	var dibujoID int64
	if duplicatedImage == nil {
		// 创建新的图片记录
		dibujo := &model.Dibujo{
			Dibujo: bytes,
		}

		if err := s.dibujoService.Save(dibujo); err != nil {
			log.Printf("[ERROR] 保存图片失败: %v", err)
			return false, err
		}
		dibujoID = dibujo.DibujoID
	} else {
		// 使用已存在的图片
		log.Printf("[INFO] 图片 %s (md5) 已存在, id: %d", md5Str, duplicatedImage.DibujoID)
		dibujoID = duplicatedImage.DibujoID
	}

	// 查询现有的商品图片记录
	var existingList []model.ArticuloDibujo
	if err := s.db.Where("ArticuloID = ?", articuloID).Find(&existingList).Error; err != nil {
		log.Printf("[ERROR] 查询商品图片记录失败: %v", err)
		return false, err
	}

	// 如果数量大于1，先删除多余的记录
	if len(existingList) > 1 {
		if err := s.db.Where("ArticuloID = ?", articuloID).Delete(&model.ArticuloDibujo{}).Error; err != nil {
			log.Printf("[ERROR] 删除多余的商品图片记录失败: %v", err)
			return false, err
		}
		existingList = nil
	}

	log.Printf("[INFO] 保存商品 %s 图片, md5: %s", articuloID, md5Str)

	// 保存或更新商品图片关联
	var articuloDibujo *model.ArticuloDibujo
	if len(existingList) > 0 {
		// 更新现有记录
		articuloDibujo = &existingList[0]
		articuloDibujo.DibujoID = dibujoID
		if err := s.db.Save(articuloDibujo).Error; err != nil {
			log.Printf("[ERROR] 更新商品图片记录失败: %v", err)
			return false, err
		}
	} else {
		// 创建新记录
		articuloDibujo = &model.ArticuloDibujo{
			ArticuloID: articuloID,
			DibujoID:   dibujoID,
			DibujoMD5:  md5Str,
			OrdenNo:    1,
		}
		if err := s.db.Create(articuloDibujo).Error; err != nil {
			log.Printf("[ERROR] 创建商品图片记录失败: %v", err)
			return false, err
		}
	}

	// 更新商品表中的图片ID
	if err := s.articuloService.UpdateDibujoID(articuloID, dibujoID); err != nil {
		log.Printf("[ERROR] 更新商品图片ID失败: %v", err)
		return false, err
	}

	return true, nil
}

// GetByMD5 根据MD5获取商品图片
func (s *ArticulodibujoServiceImpl) GetByMD5(md5 string) (*model.ArticuloDibujo, error) {
	var articuloDibujo model.ArticuloDibujo
	if err := s.db.Where("DibujoMD5 = ?", md5).First(&articuloDibujo).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &articuloDibujo, nil
}

// GetByIdExactly 根据商品ID获取商品图片
func (s *ArticulodibujoServiceImpl) GetByIdExactly(id string) (*model.ArticuloDibujo, error) {
	var articuloDibujo model.ArticuloDibujo
	if err := s.db.Where("ArticuloID = ?", id).First(&articuloDibujo).Error; err != nil {
		if err == gorm.ErrRecordNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &articuloDibujo, nil
}
