package service

import (
	"fmt"
	"log"
	"gorm.io/gorm"
	"supermarket-go/internal/model"
)

type StockServiceImpl struct {
	db *gorm.DB
}

func NewStockService(db *gorm.DB) model.StockService {
	log.Println("[INFO] 创建新的 StockService 实例")
	return &StockServiceImpl{db: db}
}

// GetByArticuloId 根据商品ID获取库存信息
func (s *StockServiceImpl) GetByArticuloId(id string) *model.Stock {
	log.Printf("[INFO] 查询商品库存: ArticuloID=%s", id)
	
	var stock model.Stock
	err := s.db.Where("ArticuloID = ?", id).First(&stock).Error
	if err != nil {
		if err == gorm.ErrRecordNotFound {
			log.Printf("[WARN] 未找到库存记录: ArticuloID=%s", id)
		} else {
			log.Printf("[ERROR] 查询库存失败: ArticuloID=%s, Error=%v", id, err)
		}
		return nil
	}
	
	log.Printf("[INFO] 成功查询到库存: ArticuloID=%s, Stock=%.2f", id, stock.Stock)
	return &stock
}

// SaveOrUpdateStock 保存或更新库存数量
func (s *StockServiceImpl) SaveOrUpdateStock(articuloId string, deltaStock int) bool {
	log.Printf("[INFO] 开始保存或更新库存: ArticuloID=%s, DeltaStock=%d", articuloId, deltaStock)
	
	existing := s.GetByArticuloId(articuloId)
	
	if existing == nil {
		// 创建新的库存记录
		log.Printf("[INFO] 创建新的库存记录: ArticuloID=%s, Stock=%d", articuloId, deltaStock)
		fmt.Printf("+ 将 %s 的库存更新为: %d\n", articuloId, deltaStock)
		
		newStock := model.Stock{
			AlmacenID:  1,
			ArticuloID: articuloId,
			Stock:      float64(deltaStock),
		}
		err := s.db.Create(&newStock).Error
		if err != nil {
			log.Printf("[ERROR] 创建库存记录失败: ArticuloID=%s, Error=%v", articuloId, err)
			return false
		}
		
		log.Printf("[INFO] 成功创建库存记录: ArticuloID=%s, Stock=%.2f", articuloId, newStock.Stock)
		return true
	} else {
		// 更新现有库存记录
		newStockValue := existing.Stock + float64(deltaStock)
		log.Printf("[INFO] 更新现有库存记录: ArticuloID=%s, 原库存=%.2f, 新库存=%.2f", 
			articuloId, existing.Stock, newStockValue)
		fmt.Printf("* 将 %s 的库存更新为: %f\n", articuloId, newStockValue)
		
		err := s.db.Model(&model.Stock{}).
			Where("ArticuloID = ?", articuloId).
			Update("stock", newStockValue).Error
		if err != nil {
			log.Printf("[ERROR] 更新库存记录失败: ArticuloID=%s, Error=%v", articuloId, err)
			return false
		}
		
		log.Printf("[INFO] 成功更新库存记录: ArticuloID=%s, Stock=%.2f", articuloId, newStockValue)
		return true
	}
}

 